/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v2.processors;

import static com.sap.cds.adapter.odata.v2.utils.TypeConverterUtils.convertToType;
import static com.sap.cds.adapter.odata.v2.utils.UriInfoUtils.getSimpleProperty;
import static com.sap.cds.services.utils.model.CdsModelUtils.getTargetEntity;
import static com.sap.cds.services.utils.model.CqnUtils.andPredicate;
import static com.sap.cds.services.utils.model.CqnUtils.modifiedWhere;
import static javax.servlet.http.HttpServletResponse.SC_CREATED;
import static javax.servlet.http.HttpServletResponse.SC_NOT_MODIFIED;
import static javax.servlet.http.HttpServletResponse.SC_NO_CONTENT;
import static javax.servlet.http.HttpServletResponse.SC_OK;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.olingo.odata2.api.edm.EdmEntityType;
import org.apache.olingo.odata2.api.edm.EdmException;
import org.apache.olingo.odata2.api.edm.EdmFunctionImport;
import org.apache.olingo.odata2.api.edm.EdmLiteral;
import org.apache.olingo.odata2.api.edm.EdmSimpleType;
import org.apache.olingo.odata2.api.edm.EdmType;
import org.apache.olingo.odata2.api.processor.ODataErrorContext;
import org.apache.olingo.odata2.api.processor.ODataResponse;
import org.apache.olingo.odata2.api.uri.KeyPredicate;
import org.apache.olingo.odata2.api.uri.NavigationSegment;
import org.apache.olingo.odata2.api.uri.UriInfo;
import org.apache.olingo.odata2.core.uri.UriInfoImpl;
import org.apache.olingo.odata2.core.uri.UriType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.Result;
import com.sap.cds.adapter.odata.v2.CdsRequestGlobals;
import com.sap.cds.adapter.odata.v2.processors.request.CdsODataRequest;
import com.sap.cds.adapter.odata.v2.processors.response.CdsODataResponse;
import com.sap.cds.adapter.odata.v2.query.CustomQueryLoader;
import com.sap.cds.adapter.odata.v2.query.NextLinkInfo;
import com.sap.cds.adapter.odata.v2.query.SystemQueryLoader;
import com.sap.cds.adapter.odata.v2.utils.AggregateTransformation;
import com.sap.cds.adapter.odata.v2.utils.ETagHelper;
import com.sap.cds.adapter.odata.v2.utils.TypeConverterUtils;
import com.sap.cds.adapter.odata.v2.utils.UriInfoUtils;
import com.sap.cds.impl.DataProcessor;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Delete;
import com.sap.cds.ql.Insert;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.Update;
import com.sap.cds.ql.cqn.AnalysisResult;
import com.sap.cds.ql.cqn.CqnAnalyzer;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsService;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.changeset.ChangeSetContext;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.HttpHeaders;
import com.sap.cds.services.utils.ListUtils;
import com.sap.cds.services.utils.ODataUtils;
import com.sap.cds.services.utils.ResultUtils;
import com.sap.cds.services.utils.SessionContextUtils;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.services.utils.model.CdsModelUtils;
import com.sap.cds.util.DataUtils;

public class CdsProcessor {

	public static final String $VALUE = "$value";

	private static final Logger logger = LoggerFactory.getLogger(CdsProcessor.class);
	private static final String CORE_MEDIA_TYPE = "Core.MediaType";
	private final CdsRequestGlobals globals;
	private final UriInfoUtils uriUtils;

	public CdsProcessor(CdsRequestGlobals globals) {
		this.globals = globals;
		this.uriUtils = new UriInfoUtils(globals);
	}

	public ODataResponse processRequest(CdsODataRequest request, Function<CdsODataRequest, CdsODataResponse> cdsProcessor, Function<CdsODataResponse, ODataResponse> responseProcessor) {
		ChangeSetContext changeSetContext = ChangeSetContext.getCurrent();
		if (changeSetContext == null) {
			return globals.getRuntime().changeSetContext().run((context) -> {
				return processInRequestContext(request, cdsProcessor, responseProcessor, context);
			});
		} else {
			return processInRequestContext(request, cdsProcessor, responseProcessor, changeSetContext);
		}
	}

	private ODataResponse processInRequestContext(CdsODataRequest request, Function<CdsODataRequest, CdsODataResponse> cdsProcessor, Function<CdsODataResponse, ODataResponse> responseProcessor, ChangeSetContext changeSetContext) {
		return globals.getRuntime().requestContext().clearMessages().parameters(request).recalculateFeatureToggles().run(requestContext -> {
			CdsODataResponse response;
			try {
				response = cdsProcessor.apply(request);
			} catch (ServiceException e) {
				markForCancel(changeSetContext, e);

				// only treat unexpected exceptions as errors
				if (e.getErrorStatus().getHttpStatus() >= 500 && e.getErrorStatus().getHttpStatus() < 600) {
					logger.error(e.getMessage(), e);
				} else {
					logger.debug(e.getMessage(), e);
				}
				response = new CdsODataResponse(e);
			} catch (Exception e) { // NOSONAR
				markForCancel(changeSetContext, e);

				// assume internal server error
				logger.error(e.getMessage(), e);
				response = new CdsODataResponse(e);
			}

			try {
				if(response.isSuccess()) {
					return responseProcessor.apply(response);
				} else {
					ODataErrorContext context = new ODataErrorContext();
					context.setException(response.getException());
					context.setContentType(request.getContentType());
					return new ErrorCallback().handleError(context);
				}
			} catch (Exception e) { // NOSONAR
				markForCancel(changeSetContext, e);
				throw e;
			}
		});
	}

	private void markForCancel(ChangeSetContext changeSetContext, Throwable e) {
		changeSetContext.markForCancel();
		logger.info("Exception marked the ChangeSet {} as cancelled: {}", changeSetContext.getId(), e.getMessage());
	}

	public CdsODataResponse get(CdsODataRequest request) {
		try {
			ApplicationService applicationService = globals.getApplicationService();
			CdsModel model = globals.getModel();
			CdsEntity cdsEntity = model.getEntity(request.getLastResourceName());
			CdsService cdsServiceModel = applicationService.getDefinition();
			UriInfoImpl uriInfo = (UriInfoImpl)request.getUriInfo();

			Map<String, Object> parameters = new HashMap<>();
			Select<?> select = Select.from(toPathExpression(request, parameters));
			CdsEntity target = getTargetEntity(select.ref(), model);

			// System query parameters
			NextLinkInfo nextLinkInfo = SystemQueryLoader.applySystemQueryOptions(select, uriInfo, true,
					cdsServiceModel, cdsEntity);

			// Custom query parameters
			CustomQueryLoader.applySystemQueryOptions(select, request.getQueryParams());

			// Transform select if required by request type
			if (!uriInfo.getPropertyPath().isEmpty()) {
				select.columns(getSimpleProperty(uriInfo).getName());
			} else if (uriInfo.isCount()) {
				select.columns(CQL.count().as("count"));
			} else {
				SystemQueryLoader.updateSelectColumns(select, uriInfo, cdsEntity);
			}
			CdsElement etagElement = ETagHelper.getETagElement(cdsEntity);

			boolean readById = (uriInfo.getUriType() == UriType.URI2
					|| (uriInfo.getUriType() == UriType.URI6A));

			if (readById && etagElement != null && ETagHelper.isETagHeaderInRequest(request)) {
				if(ETagHelper.hasIfNoneMatchHeaderWithAsteriskValue(request)) {
					throw new ErrorStatusException(CdsErrorStatuses.ETAG_FAILED);
				}
				select = modifiedWhere(select, andPredicate(ETagHelper.getETagPredicate(request, etagElement)));
			}

			// Is media entity
			String mimeType = null;
			if (uriInfo.getTargetEntitySet().getEntityType().hasStream()) {
				CdsElement mediaElement = getMediaTypeElement(select.ref(), model);
				mimeType = getMediaType(mediaElement);
				select.columns(mediaElement.getName());
			}

			// Aggregation
			AggregateTransformation aggregation = new AggregateTransformation(target, select, uriInfo);
			boolean isAggregateEntity = aggregation.applyAggregation();

			Result result = applicationService.run(select, parameters);

			if (readById && etagElement != null && ETagHelper.isETagHeaderInRequest(request) && result.rowCount() == 0) {
				if (request.getHeader(HttpHeaders.IF_NONE_MATCH) != null) {
					return new CdsODataResponse(SC_NOT_MODIFIED, result);
				}
				throw new ErrorStatusException(CdsErrorStatuses.ETAG_FAILED);
			}

			if (result.rowCount() == 0 && readById) { // Throw exception if the entity does not exist for read calls
				logger.debug("Entity not found: " + select.ref());
				throw new ErrorStatusException(CdsErrorStatuses.ENTITY_INSTANCE_NOT_FOUND,
						request.getLastResourceName(),
						CdsModelUtils.getTargetKeysAsString(model, select));
			}

			return new CdsODataResponse(SC_OK, result, nextLinkInfo, mimeType, isAggregateEntity);
		} catch (EdmException e) {
			throw new ServiceException(e);
		}
	}

	public CdsODataResponse post(CdsODataRequest request) {
		try {
			ApplicationService applicationService = globals.getApplicationService();
			CdsModel model = globals.getModel();

			String qualifiedEntityName = request.getLastResourceName();
			CdsEntity cdsEntity = model.getEntity(qualifiedEntityName);

			StructuredType<?> ref = toPathExpression(request, new HashMap<>());
			// remove the final filter, as CDS4J does not support it on CqnInsert statements
			// a filter might occur, if the post was triggered as an upsert operation during patch
			ref.filter((CqnPredicate) null);
			Insert insert = Insert.into(ref).entry(request.getBodyMap());

			Result result = null;
			if (DraftUtils.isDraftEnabled(cdsEntity)) {
				result = ((DraftService) applicationService).newDraft(insert);
			} else {
				result = applicationService.run(insert);
			}

			return new CdsODataResponse(SC_CREATED, result);
		} catch (EdmException e) {
			throw new ServiceException(e);
		}
	}

	public CdsODataResponse delete(CdsODataRequest request) {
		try {
			ApplicationService applicationService = globals.getApplicationService();
			CdsModel model = globals.getModel();

			@SuppressWarnings("rawtypes")
			Delete delete = Delete.from(toPathExpression(request, new HashMap<>()));

			Result result;
			CdsEntity entity = globals.getModel().getEntity(request.getLastResourceName());
			CdsElement etagElement = ETagHelper.getETagElement(entity);
			if (etagElement != null) {
				if (ETagHelper.isETagHeaderInRequest(request)) {
					if(ETagHelper.hasIfNoneMatchHeaderWithAsteriskValue(request)) {
						throw new ErrorStatusException(CdsErrorStatuses.ETAG_FAILED);
					}
					delete = modifiedWhere(delete, andPredicate(ETagHelper.getETagPredicate(request, etagElement)));
				} else {
					throw new ErrorStatusException(CdsErrorStatuses.ETAG_REQUIRED);
				}
			}

			if (draftEvent(delete.ref(), entity, globals.getModel())) {
				result = ((DraftService) applicationService).cancelDraft(delete);
			} else {
				result = applicationService.run(delete);
			}
			if (result.rowCount() == 0) {
				if (etagElement != null) {
					throw new ErrorStatusException(CdsErrorStatuses.ETAG_FAILED);
				}
				logger.debug("Entity not found: " + delete.ref());
				throw new ErrorStatusException(CdsErrorStatuses.ENTITY_INSTANCE_NOT_FOUND,
						request.getLastResourceName(),
						CdsModelUtils.getTargetKeysAsString(model, delete));
			}

			return new CdsODataResponse(SC_NO_CONTENT, null);
		} catch (EdmException e) {
			throw new ServiceException(e);
		}
	}

	public CdsODataResponse put(CdsODataRequest request) {
		try {
			CdsModel model = globals.getModel();

			String qualifiedEntityName = request.getLastResourceName();
			CdsEntity cdsEntity = model.getEntity(qualifiedEntityName);

			// fill up unspecified values
			DataUtils dataUtils = DataUtils.create(() -> SessionContextUtils.toSessionContext(CdsRequestGlobals.currentContext()));
			DataProcessor processor = DataProcessor.create() //
					.addGenerator((p, e, t) -> DataUtils.hasDefaultValue(e, t),
							(p, e, isNull) -> isNull ? null : dataUtils.defaultValue(e))
					.action(CdsProcessor::defaultToNull);
			processor.process(request.getBodyMap(), cdsEntity);

			// run a default patch
			return patchImpl(request, model, cdsEntity);
		} catch (EdmException e) {
			throw new ServiceException(e);
		}
	}

	public CdsODataResponse patch(CdsODataRequest request) {
		try {
			CdsModel model = globals.getModel();
			String qualifiedEntityName = request.getLastResourceName();
			CdsEntity cdsEntity = model.getEntity(qualifiedEntityName);

			return patchImpl(request, model, cdsEntity);

		} catch (EdmException e) {
			throw new ServiceException(e);
		}
	}

	private CdsODataResponse patchImpl(CdsODataRequest request, CdsModel model, CdsEntity cdsEntity) throws EdmException {
		ApplicationService applicationService = globals.getApplicationService();
		StructuredType<?> pathExpression = toPathExpression(request, new HashMap<>());

		// add target keys from the URL to the payload
		// this also ensures that target key values are not updatable
		AnalysisResult analysis = CqnAnalyzer.create(globals.getModel()).analyze(pathExpression.asRef());
		request.getBodyMap().putAll(analysis.targetKeyValues());

		//Media type
		if (request.getUriInfo().getTargetEntitySet().getEntityType().hasStream()) {
			Map<String, Object> data = request.getBodyMap();
			if (data.containsKey($VALUE)) {
				CdsElement mediaElement = getMediaTypeElement(pathExpression.asRef(), model);
				data.put(mediaElement.getName(), data.remove($VALUE));
			}
		}

		Update<?> update = Update.entity(pathExpression).data(request.getBodyMap());

		CdsElement etagElement = ETagHelper.getETagElement(cdsEntity);
		if (etagElement != null) {
			if (ETagHelper.isETagHeaderInRequest(request)) {
				update = modifiedWhere(update, andPredicate(ETagHelper.getETagPredicate(request, etagElement)));
			} else {
				throw new ErrorStatusException(CdsErrorStatuses.ETAG_REQUIRED);
			}
		}


		Result updateResult;
		if (draftEvent(update.ref(), cdsEntity, model)) {
			updateResult = ((DraftService) applicationService).patchDraft(update);
		} else {
			updateResult = applicationService.run(update);
		}
		long rowCount = updateResult.rowCount();

		if(etagElement != null) {
			boolean ifNoneMatchAsterisk = ETagHelper.hasIfNoneMatchHeaderWithAsteriskValue(request);
			if ((rowCount == 0 && !ifNoneMatchAsterisk) || (rowCount > 0 && ifNoneMatchAsterisk)) {
				throw new ErrorStatusException(CdsErrorStatuses.ETAG_FAILED);
			}
		}

		if(rowCount == 0) {
			// trigger an insert
			return post(request);
		}
		return new CdsODataResponse(SC_NO_CONTENT, updateResult);
	}

	private static void defaultToNull(CdsStructuredType type, Map<String, Object> data) {
		List<String> fkElements = new ArrayList<>();
		CqnVisitor visitor = new CqnVisitor() { // NOSONAR

			@Override
			public void visit(CqnElementRef elementRef) {
				if(elementRef.segments().size() == 1) {
					String name = elementRef.firstSegment();
					if(!"$self".equals(name)) {
						fkElements.add(name);
					}
				}
			}

		};

		type.associations()
		.map(e -> e.getType().as(CdsAssociationType.class))
		.map(a -> a.onCondition())
		.filter(o -> o.isPresent())
		.map(o -> o.get())
		.forEach(p -> p.accept(visitor));

		// do not set default null values for keys, foreign keys and associations
		type.elements()
		.filter(e -> !e.isKey())
		.filter(e -> !e.getType().isAssociation())
		.filter(e -> CdsAnnotations.ODATA_FOREIGN_KEY_FOR.getOrDefault(e) == null)
		.filter(e -> !fkElements.contains(e.getName()))
		.forEach(e -> data.putIfAbsent(e.getName(), null));
	}

	@SuppressWarnings("unchecked")
	public CdsODataResponse function(CdsODataRequest request) {
		CdsModel model = globals.getModel();
		EdmFunctionImport functionImport = request.getUriInfo().getFunctionImport();
		Map<String, EdmLiteral> functionParams = request.getUriInfo().getFunctionImportParameters();

		Result result = null;
		try {
			EventContext context;
			Optional<String> boundEntityName = getBoundEntityName(functionImport);
			if (boundEntityName.isPresent()) {
				String qualifiedEntityName = boundEntityName.get();
				CdsEntity cdsEntity = model.getEntity(qualifiedEntityName);
				String entityName = ODataUtils.toODataName(cdsEntity.getName());
				validateETagForWriteWithSelect(request, cdsEntity);

				String functionName = functionImport.getName();
				functionName = functionName.substring(entityName.length() + 1); // <ENTITY_NAME>_<FUNCTION_NAME> --> <FUNCTION_NAME>

				context = EventContext.create(functionName, qualifiedEntityName);
				Set<String> keyNames = com.sap.cds.util.CdsModelUtils.keyNames(cdsEntity);
				context.put("cqn", Select.from(getPathExpression(cdsEntity, keyNames, functionParams)));
				functionParams.forEach((key, value ) -> {
					if (!keyNames.contains(key)) {
						context.put(key, TypeConverterUtils.convertToType(value.getType(), value.getLiteral()));
					}
				});
			} else {
				context = EventContext.create(functionImport.getName(), null);
				functionParams.forEach((key, value ) -> context.put(key, TypeConverterUtils.convertToType(value.getType(), value.getLiteral())));
			}

			globals.getApplicationService().emit(context);
			Object returnValue = context.get("result");
			if (functionImport.getReturnType() == null) {
				return new CdsODataResponse(SC_NO_CONTENT, null);
			} else {
				EdmType returnType = functionImport.getReturnType().getType();
				if(returnValue != null) {
					try {
						if (returnType instanceof EdmSimpleType) {
							Map<String, Object> data = new HashMap<>();
							data.put(functionImport.getName(), returnValue);
							result = ResultUtils.convert(ListUtils.getList(data));
						} else if(returnValue instanceof Iterable) {
							result = ResultUtils.convert((Iterable<Map<String, Object>>) returnValue);
						} else if (returnValue instanceof Map) {
							result = ResultUtils.convert(ListUtils.getList((Map<String, Object>) returnValue));
						} else {
							throw new ErrorStatusException(CdsErrorStatuses.UNSUPPORTED_ACTION_FUNCTION_RETURN_TYPE, returnValue.getClass().getName());
						}
					} catch (ClassCastException e) {
						throw new ErrorStatusException(CdsErrorStatuses.UNSUPPORTED_ACTION_FUNCTION_RETURN_TYPE, returnValue.getClass().getName(), e);
					}
				}
			}
		} catch (EdmException e) {
			throw new ErrorStatusException(CdsErrorStatuses.FUNCTION_NOT_FOUND, e);
		}
		return new CdsODataResponse(SC_OK, result);
	}

	private Optional<String> getBoundEntityName(EdmFunctionImport functionImport) throws EdmException {
		try {
			if (functionImport.getAnnotations().getAnnotationAttributes() != null) {
				return functionImport.getAnnotations().getAnnotationAttributes().stream()
						.filter(a -> "sap".equals(a.getPrefix()) && "action-for".equals(a.getName()))
						.map(a -> this.globals.getCdsEntityNames().getOrDefault(a.getText(), a.getText())).findFirst();
			}
			return Optional.empty();
		} catch (EdmException e) {
			throw new ErrorStatusException(CdsErrorStatuses.RESOLVING_FUNCTION_IMPORT_ANNOTATION_FAILED, functionImport.getName(), e);
		}
	}

	private StructuredType<?> getPathExpression(CdsEntity rootEntity, Collection<String> keyNames, Map<String, EdmLiteral> functionParams) throws EdmException {
		Map<String, Object> keyPairs = new HashMap<>();
		functionParams.forEach((key, value ) -> {
			if (keyNames.contains(key)) {
				keyPairs.put(key, TypeConverterUtils.convertToType(value.getType(), value.getLiteral()));
			}
		});
		return CQL.entity(rootEntity.getQualifiedName()).matching(keyPairs);
	}

	private void validateETagForWriteWithSelect(CdsODataRequest request, CdsEntity cdsEntity) throws EdmException {
		CdsElement etagElement = ETagHelper.getETagElement(cdsEntity);
		if (etagElement != null) {
			if (ETagHelper.isETagHeaderInRequest(request)) {
				Map<String, Object> parameters = new HashMap<>();
				CqnSelect select = Select.from(toPathExpression(request, parameters));
				CqnSelect etagSelect = modifiedWhere(select, andPredicate(ETagHelper.getETagPredicate(request, etagElement)));

				Result queryResult = globals.getRuntime().requestContext().clearMessages().run(requestContext -> {
					return globals.getApplicationService().run(etagSelect, parameters);
				});

				long rowCount = queryResult.rowCount();
				boolean ifNoneMatchAsterisk = ETagHelper.hasIfNoneMatchHeaderWithAsteriskValue(request);
				if ((rowCount == 0 && !ifNoneMatchAsterisk) || (rowCount > 0 && ifNoneMatchAsterisk)) {
					throw new ErrorStatusException(CdsErrorStatuses.ETAG_FAILED);
				}
			} else {
				throw new ErrorStatusException(CdsErrorStatuses.ETAG_REQUIRED);
			}
		}
	}

	private boolean draftEvent(CqnStructuredTypeRef ref, CdsEntity entity, CdsModel model) {
		Map<String, Object> targetKeys = CqnAnalyzer.create(model).analyze(ref).targetKeyValues();
		return DraftUtils.isDraftEnabled(entity) && targetKeys.containsKey(Drafts.IS_ACTIVE_ENTITY) && !(Boolean) targetKeys.get(Drafts.IS_ACTIVE_ENTITY);
	}

	private StructuredType<?> toPathExpression(CdsODataRequest request, Map<String, Object> parameters) throws EdmException {

		UriInfo uriInfo = request.getUriInfo();
		StructuredType<?> parent = null;

		// Root entity
		EdmEntityType rootEntityType = uriInfo.getStartEntitySet().getEntityType();
		Map<String, Object> keys = getFilterKeys(uriInfo.getKeyPredicates());

		if (this.uriUtils.isParametersEntityType(rootEntityType)) {
			parameters.putAll(keys);
		} else {
			String entityName = this.uriUtils.getCdsEntityName(rootEntityType);
			// TODO: skip the "matching(keys)" in case of singleton resource?
			parent = CQL.entity(entityName).matching(keys);
		}

		// Navigation
		for (NavigationSegment segment : uriInfo.getNavigationSegments()) {
			// we really expect navigationType to be an EdmEntityType, but olingo just returns EdmType here???
			EdmType navigaionType = segment.getNavigationProperty().getType();
			keys = getFilterKeys(segment.getKeyPredicates());

			if ((navigaionType instanceof EdmEntityType) && (this.uriUtils.isParametersEntityType((EdmEntityType) navigaionType))) {
				parameters.putAll(keys);
			} else {
				if(parent == null) {
					String entityName = (navigaionType instanceof EdmEntityType) ? this.uriUtils.getCdsEntityName((EdmEntityType) navigaionType) : segment.getNavigationProperty().getName();
					parent = CQL.entity(entityName).matching(keys);
				} else {
					parent = parent.to(segment.getNavigationProperty().getName()).matching(keys);
				}
			}
		}

		if(parent == null) {
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_PARAMETERIZED_VIEW);
		}

		return parent;
	}

	private Map<String, Object> getFilterKeys(List<KeyPredicate> keyPredicates) throws EdmException {
		Map<String, Object> keys = new HashMap<>();
		for (KeyPredicate key : keyPredicates) {
			String name = key.getProperty().getName();
			EdmType type = key.getProperty().getType();
			keys.put(name, convertToType(type, key.getLiteral()));
		}

		return keys;
	}

	private CdsElement getMediaTypeElement(CqnStructuredTypeRef structuredTypeRef, CdsModel model) {
		CdsEntity entity = getTargetEntity(structuredTypeRef, model);
		List<CdsElement> mediaElements = entity.elements().filter(e -> e.findAnnotation(CORE_MEDIA_TYPE).isPresent())
				.collect(Collectors.toList());
		if (mediaElements.size() != 1) {
			throw new ServiceException("Number of Core.MediaType elements must be exaclty 1");
		}

		return mediaElements.get(0);
	}

	private String getMediaType(CdsElement element) {
		return element.annotations().filter(a -> CORE_MEDIA_TYPE.equals(a.getName())).map(a -> a.getValue().toString())
				.findFirst().orElseGet(() -> {
					throw new ServiceException(element.getName() + " is not a Core.MediaType element");
				});
	}

}
