/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v2.processors;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import org.apache.olingo.odata2.api.commons.HttpHeaders;
import org.apache.olingo.odata2.api.commons.HttpStatusCodes;
import org.apache.olingo.odata2.api.ep.EntityProvider;
import org.apache.olingo.odata2.api.ep.EntityProviderException;
import org.apache.olingo.odata2.api.exception.ODataPreconditionFailedException;
import org.apache.olingo.odata2.api.exception.ODataPreconditionRequiredException;
import org.apache.olingo.odata2.api.processor.ODataErrorCallback;
import org.apache.olingo.odata2.api.processor.ODataErrorContext;
import org.apache.olingo.odata2.api.processor.ODataResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.adapter.odata.v2.CdsRequestGlobals;
import com.sap.cds.adapter.odata.v2.utils.MessagesUtils;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.messages.Messages;

public class ErrorCallback implements ODataErrorCallback {

	private static final Logger LOG = LoggerFactory.getLogger(ErrorCallback.class);

	@Override
	public ODataResponse handleError(ODataErrorContext context) {
		Locale locale = CdsRequestGlobals.currentContext().getParameterInfo().getLocale();
		context.setLocale(locale == null ? Locale.ENGLISH : locale);

		if (context.getException().getCause() instanceof EntityProviderException) {
			context.setHttpStatus(HttpStatusCodes.BAD_REQUEST);
			context.setErrorCode(HttpStatusCodes.BAD_REQUEST.name());
		} else if (context.getException() instanceof ServiceException) {
			ServiceException exp = (ServiceException)context.getException();
			context.setErrorCode(exp.getErrorStatus().getCodeString());
			context.setException(exp);
			context.setHttpStatus(HttpStatusCodes.fromStatusCode(exp.getErrorStatus().getHttpStatus()));
			context.setMessage(exp.getLocalizedMessage(CdsRequestGlobals.currentContext().getParameterInfo().getLocale()));
			context.setTarget(MessagesUtils.getTarget(exp.getMessageTarget()));
			setDetailedMessages(context, CdsRequestGlobals.currentContext().getMessages());
		} else if (context.getException() instanceof ODataPreconditionRequiredException) {
			ODataPreconditionRequiredException exp = (ODataPreconditionRequiredException) context.getException();
			context.setErrorCode("428001");
			context.setException(exp);
			context.setHttpStatus(exp.getHttpStatus());
			context.setMessage(exp.getLocalizedMessage());
		} else if (context.getException() instanceof ODataPreconditionFailedException) {
			ODataPreconditionFailedException exp = (ODataPreconditionFailedException) context.getException();
			context.setErrorCode("412001");
			context.setException(exp);
			context.setHttpStatus(exp.getHttpStatus());
			context.setMessage(exp.getLocalizedMessage());
		}

		int statusCode = context.getHttpStatus().getStatusCode();
		if (statusCode >= 500 && statusCode < 600) {
			LOG.error(context.getMessage(), context.getException());
		} else {
			LOG.debug(context.getMessage(), context.getException());
		}

		ODataResponse odataResponse = EntityProvider.writeErrorDocument(context);
		return ODataResponse.fromResponse(odataResponse).header(HttpHeaders.CONTENT_TYPE, context.getContentType())
				.status(context.getHttpStatus()).build();
	}

	private static void setDetailedMessages(ODataErrorContext context, Messages messages) {
		List<ODataErrorContext> details = new ArrayList<>();
		if (messages != null ) {
			messages.stream().forEach(m -> {
				ODataErrorContext detail = new ODataErrorContext();
				detail.setErrorCode(m.getCode());
				detail.setMessage(m.getMessage());
				detail.setSeverity(m.getSeverity().toString().toLowerCase(Locale.ENGLISH));
				detail.setTarget(MessagesUtils.getTarget(m.getTarget()));
				details.add(detail);
			});
		}
		context.setErrorDetails(details);
	}

}
