/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v2.utils;

import static com.sap.cds.impl.builder.model.ElementRefImpl.element;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.olingo.odata2.api.edm.EdmException;
import org.apache.olingo.odata2.api.uri.SelectItem;
import org.apache.olingo.odata2.api.uri.UriInfo;

import com.sap.cds.impl.parser.builder.SortSpecBuilder;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.ElementRef;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.Value;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnSelectListValue;
import com.sap.cds.ql.cqn.CqnSortSpecification;
import com.sap.cds.ql.impl.SelectListValueBuilder;
import com.sap.cds.reflect.CdsAnnotatable;
import com.sap.cds.reflect.CdsAnnotation;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

public class AggregateTransformation {

	private static final String AGGREGATION_DEFAULT = "Aggregation.default";
	private static final String ANALYTICS_MEASURE = "Analytics.Measure";
	private static final String SUPPORTED_RESTRICTIONS = "Aggregation.ApplySupported.PropertyRestrictions";
	private static final String SAP_AGGREGATION_ROLE = "sap.aggregation-role";

	public static final String AGGREGATE_ID = "ID__";

	private static final String COUNT_DISTINCT = "COUNT_DISTINCT";
	private static final List<String> ALLOWED_AGGREGATES = Arrays.asList("SUM", "MAX", "MIN", "AVG");

	private final CdsEntity target;
	private final Select<?> select;
	private final UriInfo uriInfo;

	private List<CqnSelectListItem> allDimensions;
	private List<CqnSelectListItem> selectListItems;

	public AggregateTransformation(CdsEntity target, Select<?> select, UriInfo uriInfo) {
		this.target = target;
		this.select = select;
		this.uriInfo = uriInfo;
	}

	public boolean applyAggregation() {
		if (isAggregateEntity()) {
			allDimensions = new ArrayList<>();
			selectListItems = new ArrayList<>();
			uriInfo.getSelect().forEach(this::collectListItems);

			select.columns(selectListItems);
			select.groupBy(allDimensions);
			select.orderBy(getOrderBy());

			return true;
		}
		return false;
	}

	private void collectListItems(SelectItem i) {
		try {
			String itemName = i.getProperty().getName();
			// exclude technical ID__ from select list as it is not in CDS Model
			if (isAggregateID(itemName, target)) {
				return;
			}
			target.getQualifier();
			CqnSelectListValue sli = SelectListValueBuilder.select(itemName).build();
			CdsElement element = target.getElement(itemName);

			if (isMeasure(element)) {
				selectListItems.add(asAggregateFunction(element).as(itemName));
			} else {
				// if element is not an aggregate, add it to group by
				selectListItems.add(sli);
				allDimensions.add(sli);
			}
		} catch (EdmException e) {
			throw new ServiceException(e);
		}
	}

	private Value<?> asAggregateFunction(CdsElement element) {
		Value<?> functionCall;
		ElementRef<Object> elementRef = element(element.getName());
		String aggregateFunctionName = getAggregation(element);

		if (ALLOWED_AGGREGATES.contains(aggregateFunctionName)) {
			functionCall = CQL.func(aggregateFunctionName, elementRef);
		} else if (COUNT_DISTINCT.equals(aggregateFunctionName)) {
			functionCall = CQL.countDistinct(elementRef);
		} else {
			throw new ErrorStatusException(CdsErrorStatuses.FUNCTION_NOT_FOUND, aggregateFunctionName);
		}
		return functionCall;
	}

	private boolean isAggregateEntity() {
		return getAnnotatedValue(target, SUPPORTED_RESTRICTIONS, false);
	}

	private boolean isMeasure(CdsElement element) {
		boolean isAnalyticsMeasure = getAnnotatedValue(element, ANALYTICS_MEASURE, false);
		if (isAnalyticsMeasure) {
			return true;
		}
		return "measure".equals(getAnnotatedValue(element, SAP_AGGREGATION_ROLE, ""));
	}

	private String getAggregation(CdsElement element) {
		Map<String, String> annotatedValue = getAnnotatedValue(element, AGGREGATION_DEFAULT,
				Collections.singletonMap("#", "#"));
		return annotatedValue.get("#");
	}

	private <T> T getAnnotatedValue(CdsAnnotatable annotatable, String annotation, T fallBackValue) {
		try {
			return annotatable.<T>findAnnotation(annotation).map(CdsAnnotation::getValue).orElse(fallBackValue);
		} catch (ClassCastException ex) {
			throw new ServiceException("The type of annotation value for " + annotatable + " is not a "
					+ fallBackValue.getClass().getName(), ex);
		}
	}

	private List<CqnSortSpecification> getOrderBy() {
		return select.orderBy().stream().map(this::getItem).collect(Collectors.toList());
	}

	private CqnSortSpecification getItem(CqnSortSpecification orderByItem) {
		return select.items().stream()
				.filter(item -> item.asValue().displayName().equals(orderByItem.item().displayName()))
				.map(item -> SortSpecBuilder.by(item.asValue()).order(orderByItem.order()).build()).findFirst()
				.orElseGet(() -> orderByItem);
	}

	public static boolean isAggregateID(String element, CdsEntity entity) {
		return AGGREGATE_ID.equals(element) && !(entity.findElement(element).isPresent());
	}
}
