/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v2.utils;

import java.math.BigDecimal;
import java.nio.charset.Charset;
import java.sql.Time;
import java.sql.Timestamp;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.UUID;

import org.apache.olingo.odata2.api.edm.EdmType;

import com.sap.cds.util.CdsTypeUtils;

public class TypeConverterUtils {

	private static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss");

	/**
	 * Convert string value of EdmType to a corresponding java type. Used for conversion of
	 * filter values.
	 *
	 * @param type  Edm Type
	 * @param value string representation of a value
	 * @return Object
	 */
	public static Object convertToType(EdmType type, String value) {
		if (value == null) {
			return null;
		}

		Object typedValue;

		switch (type.toString()) {
		case "Edm.Decimal":
			typedValue = new BigDecimal(value);
			break;
		case "Edm.Double":
			typedValue = Double.parseDouble(value);
			break;
		case "Edm.Single":
			typedValue = Float.parseFloat(value);
			break;
		case "Edm.Int64":
			typedValue = Long.parseLong(value);
			break;
		case "Edm.Int32":
		case "Edm.Int16":
		case "System.Uint7":
		case "Edm.Byte":
		case "Edm.SByte":
		case "System.Bit":
			typedValue = Integer.parseInt(value);
			break;
		case "Edm.Binary":
			typedValue = value.getBytes(Charset.defaultCharset());
			break;
		case "Edm.String":
			if (value.length() >= 2 && value.startsWith("'") && value.endsWith("'")) {
				typedValue = value.substring(1, value.length() - 1);
			} else {
				typedValue = value;
			}
			break;
		case "Edm.TimeOfDay":
			typedValue = LocalTime.parse(value);
			break;
		case "Edm.Date":
			typedValue = LocalDate.parse(value);
			break;
		case "Edm.DateTime":
			typedValue = LocalDate.from(DATE_FORMATTER.parse(value));
			break;
		case "Edm.DateTimeOffset":
			typedValue = ZonedDateTime.parse(value).toInstant();
			break;
		case "Edm.Boolean":
			typedValue = Boolean.parseBoolean(value);
			break;
		case "Edm.Guid":
			typedValue = CdsTypeUtils.parseUuid(value);
			break;
		default:
			typedValue = value;
		}

		return typedValue;
	}

	/**
	 * Convert value of EdmType to a corresponding java type supported by Olingo
	 * library. This is used for type conversion of a response.
	 *
	 * @param type  Edm Type
	 * @param value string representation of a value
	 * @return Object
	 */
	public static Object getValueBasedOnTypeOfResultSet(EdmType type, Object value) {
		if (value == null) {
			return null;
		}

		Object typedValue = null;

		switch (type.toString()) {
		case "Edm.Time":
			if (value instanceof LocalTime) {
				typedValue = new Calendar.Builder()
						.set(Calendar.HOUR_OF_DAY,((LocalTime)value).getHour())
						.set(Calendar.MINUTE,((LocalTime)value).getMinute())
						.set(Calendar.SECOND,((LocalTime)value).getSecond())
						.set(Calendar.MILLISECOND,((LocalTime)value).getNano()/1000000)
						.build();
			}
			break;
		case "Edm.DateTime":
			if (value instanceof LocalDateTime) {
				Instant instant = ((LocalDateTime)value).toInstant(ZoneOffset.UTC);
				typedValue = instant.toEpochMilli();
			} else if (value instanceof LocalDate) {
				typedValue = toEpochMili((LocalDate)value); // Cds type Date is mapped to Edm DateTime. Odata V2 does not have Edm Date type
			}
			break;
		case "Edm.DateTimeOffset":
			if (value instanceof Instant) {
				try {
					typedValue = Timestamp.from((Instant) value);
				} catch (DateTimeParseException ex) {
					typedValue = GregorianCalendar.from(ZonedDateTime.parse(value.toString() + "Z"));
				}
			}
			break;
		case "Edm.Guid":
			typedValue = UUID.fromString(value.toString());
			break;
		default:
			typedValue = value;
		}

		return typedValue;
	}

	/**
	 * Convert value of EdmType to a corresponding java type supported by CDS4J
	 * library. This is used for type conversion of a request payload.
	 *
	 * @param type  Edm Type
	 * @param value string representation of a value
	 * @return Object
	 */
	public static Object getValueBasedOnTypeOfRequestPayload(EdmType type, Object value) {
		if(value == null) {
			return null;
		}

		Object typedValue = null;

		switch (type.toString()) {
		case "Edm.Time":
			if (value instanceof GregorianCalendar) {
				typedValue = ((GregorianCalendar)value).toZonedDateTime().toLocalTime();
			} else if (value instanceof Time) {
				typedValue = ((Time)value).toLocalTime();
			}
			break;
		case "Edm.DateTime":
			if (value instanceof GregorianCalendar) {
				typedValue = ((GregorianCalendar)value).toZonedDateTime().toLocalDate(); // Cds type Date is mapped to Edm DateTime.Odata V2 does not have Edm Date type
			} else if (value instanceof Timestamp) {
				typedValue = ((Timestamp)value).toLocalDateTime();
			}
			break;
		case "Edm.DateTimeOffset":
			if (value instanceof GregorianCalendar) {
				typedValue = ((GregorianCalendar)value).toInstant();
			} else if (value instanceof Timestamp) {
				typedValue = ((Timestamp)value).toInstant();
			}
			break;
		case "Edm.Guid":
			typedValue =  CdsTypeUtils.parseUuid(value);
			break;
		default:
			typedValue = value;
		}

		return typedValue;
	}

	private static Long toEpochMili(LocalDate localDate) {
		return localDate.atStartOfDay().toInstant(ZoneOffset.UTC).toEpochMilli();
	}

}
