/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v2;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Stream;

import com.sap.cds.adapter.ServletAdapterFactory;
import com.sap.cds.adapter.UrlResourcePath;
import com.sap.cds.reflect.CdsAction;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsFunction;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.CdsRuntimeAware;
import com.sap.cds.services.utils.ODataUtils;
import com.sap.cds.services.utils.path.CdsResourcePath;
import com.sap.cds.services.utils.path.CdsServicePath;
import com.sap.cds.services.utils.path.UrlPathUtil;
import com.sap.cds.services.utils.path.UrlResourcePathBuilder;

public class CdsODataV2ServletFactory implements ServletAdapterFactory, CdsRuntimeAware {

	public static final String PROTOCOL_KEY = "odata-v2";

	private CdsRuntime runtime;

	@Override
	public void setCdsRuntime(CdsRuntime runtime) {
		this.runtime = runtime;
	}

	@Override
	public boolean isEnabled() {
		return runtime.getEnvironment().getCdsProperties().getOdataV2().getEndpoint().isEnabled();
	}

	@Override
	public Object create() {
		return new CdsODataV2Servlet(runtime);
	}

	@Override
	public String getBasePath() {
		return UrlPathUtil.normalizeBasePath(runtime.getEnvironment().getCdsProperties().getOdataV2().getEndpoint().getPath());
	}

	@Override
	public String[] getMappings() {
		Stream<CdsResourcePath> cdsServicePaths = CdsServicePath.servicePaths(runtime, PROTOCOL_KEY);
		return cdsServicePaths.map(servicePath -> UrlResourcePathBuilder.path(getBasePath(), servicePath.getPath()).recursive().build().getPath()).toArray(l -> new String[l]);
	}

	@Override
	public UrlResourcePath getServletPath() {

		final boolean openMetadataEndpoint = !runtime.getEnvironment().getCdsProperties().getSecurity().getAuthentication().isAuthenticateMetadataEndpoints();

		Stream<CdsResourcePath> cdsServicePaths = CdsServicePath.servicePaths(runtime, PROTOCOL_KEY);
		return UrlResourcePathBuilder.path(getBasePath())
				.recursive()
				.subPaths(
						cdsServicePaths.flatMap(cdsServicePath -> {

							boolean servicePublic = cdsServicePath.isPublic() && !cdsServicePath.subPaths().anyMatch(p -> !p.isPublic());
							return Stream.of(

									// Paths: /service/$metadata
									UrlResourcePathBuilder.path(cdsServicePath.getPath(), "$metadata").isPublic(openMetadataEndpoint || cdsServicePath.isPublic()).build(),

									// Path: /service/$batch
									UrlResourcePathBuilder.path(cdsServicePath.getPath(), "$batch").isPublic(servicePublic).build(),

									// Paths: /service, /service/entity/** and /service/function
									UrlResourcePathBuilder.path(cdsServicePath.getPath())
									.isPublic(cdsServicePath.isPublic())
									.subPaths( cdsServicePath.subPaths().flatMap(cdsSubPath -> {

										if (cdsSubPath.hasType(CdsEntity.class)) {
											// Path: /service/entity/** and bound actions and functions /service/entity_action
											return getEntitySubPaths(cdsServicePath, cdsSubPath).stream();
										} else {
											// Path: /service/action-or-function
											return actionOrFunction(cdsSubPath, cdsSubPath.getPath(), cdsServicePath.isPublic());
										}

									}) ).build(),

									// Paths: /service/
									UrlResourcePathBuilder.path(cdsServicePath.getPath(), "/")
									.isPublic(cdsServicePath.isPublic())
									.build()

									);
						})
						).build();
	}


	private static List<UrlResourcePath> getEntitySubPaths(CdsResourcePath cdsServicePath, CdsResourcePath cdsSubPath) {
		boolean isPublic = cdsServicePath.isPublic() && cdsSubPath.isPublic();

		ArrayList<UrlResourcePath> result = new ArrayList<>();

		// Path: /service/entity/**
		result.add(UrlResourcePathBuilder.path(ODataUtils.toODataName(cdsSubPath.getPath()))
				.recursive()
				.isPublic(isPublic)
				.publicEvents(cdsSubPath.publicEvents().map(UrlPathUtil::cdsEvent2HttpMethod).filter(Objects::nonNull))
				.build());

		// Path: /service/entity(*)/**
		result.add(UrlResourcePathBuilder.path( cdsSubPath.getPath() + "(*)" ).recursive()
				.isPublic(isPublic)
				.publicEvents(cdsSubPath.publicEvents().map(UrlPathUtil::cdsEvent2HttpMethod).filter(Objects::nonNull))
				.build());

		// Path: /service/entity_action
		cdsSubPath.subPaths().forEach(p -> {
			String actionOrFunctionPath = cdsSubPath.getPath() + "_" + p.getPath();
			actionOrFunction(p, actionOrFunctionPath, isPublic).forEach(urlPath -> result.add(urlPath));
		} );

		return result;
	}


	private static Stream<UrlResourcePath> actionOrFunction(CdsResourcePath p, String actionOrFunctionPath, boolean parentIsPublic) {

		if (p.hasType(CdsAction.class) || p.hasType(CdsFunction.class)) {
			String httpMethod = p.hasType(CdsAction.class) ? UrlPathUtil.HttpMethod.POST.name() : UrlPathUtil.HttpMethod.GET.name();

			// function, function(*) resp. action and action(*)
			return Stream.of(
					UrlResourcePathBuilder.path(actionOrFunctionPath)
					.isPublic(false)
					.publicEvents( parentIsPublic && p.isPublic() ? Stream.of(httpMethod) : Stream.empty() )
					.build(),

					UrlResourcePathBuilder.path(actionOrFunctionPath + "(*)")
					.isPublic(false)
					.publicEvents( parentIsPublic && p.isPublic() ? Stream.of(httpMethod) : Stream.empty() )
					.build()
					);
		}

		return Stream.empty();
	}
}
