/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v2;

import java.io.PrintWriter;
import java.util.stream.Stream;

import com.sap.cds.adapter.IndexContentProvider;
import com.sap.cds.reflect.CdsDefinition;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.services.utils.ODataUtils;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.services.utils.path.CdsResourcePath;
import com.sap.cds.services.utils.path.CdsServicePath;

public class ODataV2IndexContentProvider implements IndexContentProvider {

	private static final String SERVICE = """
			                <h3 class="header">
			                    <a href="%s/"><span>%s</span></a><span>/</span><a href="%s/$metadata"><span class="metadata">$metadata</span></a>
			                </h3>
			""";

	private static final String ENTITY_START = """
			                <ul>
			""";

	private static final String ENTITY = """
			                    <li>
			                        <a href="%s"><span>%s</span></a>
			                    </li>
			""";

	private static final String ENTITY_END = """
			                </ul>
			""";

	private final CdsRuntime runtime;

	public ODataV2IndexContentProvider(CdsRuntime runtime) {
		this.runtime = runtime;
	}

	@Override
	public String getSectionTitle() {
		return "OData V2 endpoints";
	}

	@Override
	public void writeContent(PrintWriter out, String contextPath) {
		Stream<CdsResourcePath> cdsServicePaths = CdsServicePath.servicePaths(runtime, CdsODataV2ServletFactory.PROTOCOL_KEY);
		CdsODataV2ServletFactory servletFactory = new CdsODataV2ServletFactory();
		servletFactory.setCdsRuntime(runtime);
		String basePath = servletFactory.getBasePath();
		String theBasePath = contextPath + (basePath.equals("/") ? "" : basePath);

		cdsServicePaths.filter(p -> p.getCdsDefinition() instanceof CdsService).sorted().forEach(s -> {
			String path = theBasePath + "/" + s.getPath();
			out.format(SERVICE, path, path, path);
			out.write(ENTITY_START);
			s.subPaths().filter(e -> isExposedEntity(e.getCdsDefinition())).sorted().forEach(e -> {
				String entityPath = path + "/" + ODataUtils.toODataName(e.getPath());
				out.format(ENTITY, entityPath, e.getPath());
			});
			out.write(ENTITY_END);
		});
	}

	private boolean isExposedEntity(CdsDefinition definition) {
		if(definition instanceof CdsEntity entity) {
			boolean autoexposed = CdsAnnotations.AUTOEXPOSED.isTrue(definition);
			boolean autoexpose = CdsAnnotations.AUTOEXPOSE.isTrue(definition);
			String name = definition.getName();
			return !(name.endsWith("_drafts") ||
					name.endsWith("_texts") ||
					name.endsWith(".texts") ||
					name.endsWith("DraftAdministrativeData") ||
					(!DraftUtils.isDraftEnabled(entity) && autoexposed && !autoexpose));
		}
		return false;
	}

}
