/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v2.metadata.model;

import java.io.ByteArrayInputStream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.hash.Hashing;
import com.sap.cds.adapter.odata.v2.metadata.MetadataInfo;
import com.sap.cds.adapter.odata.v2.metadata.ODataV2EdmProvider;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.model.DynamicModelUtils;

/**
 * Abstract base for classes that implement the parsing of the edmx model.
 */
public abstract class AbstractEdmxProviderAccessor {

	private static Logger log = LoggerFactory.getLogger(AbstractEdmxProviderAccessor.class);
	protected final static String XML = ".xml";

	private static DynamicModelUtils utils;
	private static AbstractEdmxProviderAccessor defaultInstance;
	private static AbstractEdmxProviderAccessor instance;

	public static void initialize(CdsRuntime runtime) {
		utils = new DynamicModelUtils(runtime);
		defaultInstance = new DefaultEdmxProviderAccessor(runtime);

		if (utils.isDynamicModelEnabled()) {
			instance = new DynamicEdmxProviderAccessor(runtime);
			log.info("Initialized Dynamic EDMX V2 Provider");
		} else {
			instance = defaultInstance;
			log.info("Initialized Default EDMX V2 Provider");
		}
	}

	public static AbstractEdmxProviderAccessor getInstance() {
		if (defaultInstance == instance || utils.useStaticModel()) {
			return defaultInstance;
		}
		return instance;
	}

	public abstract MetadataInfo getMetadataInfo(String serviceName);

	protected ODataV2EdmProvider loadMetadataFiles(String serviceName, byte[] edmxBytes) {
		if (edmxBytes == null || edmxBytes.length == 0) {
			return null; // file not found
		}

		log.info("Loading OData V2 metadata for service '{}'", serviceName);
		try {
			ODataV2EdmProvider edmxProvider = new ODataV2EdmProvider(new ByteArrayInputStream(edmxBytes));
			return edmxProvider;
		} catch (Exception e) { // NOSONAR
			throw new IllegalArgumentException("The provided bytes don't contain OData V2 metadata", e);
		}
	}

	protected String calculateMetadataEtag(byte[] edmxBytes) {
		if(edmxBytes == null || edmxBytes.length == 0) {
			return null;
		}

		return Hashing.sha256().hashBytes(edmxBytes).toString();
	}

}
