/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v2.metadata.model;

import java.nio.charset.StandardCharsets;
import java.util.Locale;
import java.util.Set;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.adapter.odata.v2.metadata.MetadataInfo;
import com.sap.cds.adapter.odata.v2.metadata.ODataV2EdmProvider;
import com.sap.cds.mtx.MetaDataAccessor;
import com.sap.cds.mtx.ModelId;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.LocaleUtils;
import com.sap.cds.services.utils.model.DynamicModelUtils;

/**
 * An implementation of the EDMX parsing that uses the MTX Sidecar or the Model Provider to retrieve the EDMX files.
 */
public class DynamicEdmxProviderAccessor extends AbstractEdmxProviderAccessor {

	private static final String V2 = "v2";

	private static Logger log = LoggerFactory.getLogger(DynamicEdmxProviderAccessor.class);

	private final MetaDataAccessor<MetadataInfo> accessor;
	private final CdsRuntime runtime;

	public DynamicEdmxProviderAccessor(CdsRuntime runtime) {
		this.runtime = runtime;
		this.accessor = new DynamicModelUtils(runtime).createMetadataAccessor((edmx, service) -> {
			byte[] edmxBytes = edmx.getBytes(StandardCharsets.UTF_8);
			ODataV2EdmProvider edmProvider;
			try {
				edmProvider = loadMetadataFiles(service, edmxBytes);
			} catch (Exception e) { // NOSONAR
				throw new ErrorStatusException(CdsErrorStatuses.INVALID_METADATA_V2_DYNAMIC, service, e);
			}
			String etag = calculateMetadataEtag(edmxBytes);
			return new MetadataInfo(edmxBytes, edmProvider, etag);
		}, null, null);
	}

	@Override
	public MetadataInfo getMetadataInfo(String serviceName) {
		RequestContext requestContext = RequestContext.getCurrent(runtime);
		Locale locale = requestContext.getParameterInfo().getLocale();
		String localeString = locale != null ? LocaleUtils.getLocaleForBundle(locale).toString() : Locale.ENGLISH.getLanguage();
		UserInfo userInfo = requestContext.getUserInfo();
		String tenant = userInfo.getTenant();
		Set<String> features = requestContext.getFeatureTogglesInfo().getEnabledFeatureToggles().map(ft -> ft.getName()).collect(Collectors.toSet());
		ModelId modelId = ModelId.create(tenant).odata(serviceName, localeString, V2).features(features).build();
		int maxAgeSeconds = Integer.MAX_VALUE;

		log.debug("Retrieving model for service name '{}', locale '{}' and features {}", serviceName, localeString, features);
		try {
			return accessor.getEdmx(modelId, maxAgeSeconds);
		} catch (ServiceException e) {
			throw e;
		} catch (Throwable t) {
			throw new ErrorStatusException(CdsErrorStatuses.EDMX_READ_FAILED, serviceName, t);
		}
	}

}
