/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v2.processors;

import java.util.Locale;
import java.util.stream.Collectors;

import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.olingo.odata2.api.commons.HttpContentType;
import org.apache.olingo.odata2.api.commons.HttpHeaders;
import org.apache.olingo.odata2.api.commons.HttpStatusCodes;
import org.apache.olingo.odata2.api.ep.EntityProvider;
import org.apache.olingo.odata2.api.ep.EntityProviderException;
import org.apache.olingo.odata2.api.exception.ODataPreconditionFailedException;
import org.apache.olingo.odata2.api.exception.ODataPreconditionRequiredException;
import org.apache.olingo.odata2.api.processor.ODataErrorCallback;
import org.apache.olingo.odata2.api.processor.ODataErrorContext;
import org.apache.olingo.odata2.api.processor.ODataResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.adapter.odata.v2.utils.MessagesUtils;
import com.sap.cds.services.ErrorStatus;
import com.sap.cds.services.ErrorStatuses;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.application.ApplicationLifecycleService;
import com.sap.cds.services.application.ErrorResponseEventContext.ErrorResponse;
import com.sap.cds.services.messages.Message;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

public class ErrorCallback implements ODataErrorCallback {

	private static final Logger logger = LoggerFactory.getLogger(ErrorCallback.class);
	private final CdsRuntime runtime;

	public ErrorCallback(CdsRuntime runtime) {
		this.runtime = runtime;
	}

	@Override
	public ODataResponse handleError(ODataErrorContext context) {
		RequestContext requestContext = RequestContext.getCurrent(runtime);
		Locale locale = requestContext.getParameterInfo().getLocale();
		context.setLocale(locale == null ? Locale.ENGLISH : locale);
		context.setContentType(negotiateContentType(requestContext, context.getContentType()));

		ServiceException exception;
		if (context.getException() instanceof ServiceException) {
			exception = (ServiceException) context.getException();
		} else if (context.getException().getCause() instanceof EntityProviderException) {
			Throwable rootCause = ExceptionUtils.getRootCause(context.getException().getCause());
			exception = new ServiceException(ErrorStatuses.BAD_REQUEST, rootCause.getMessage(), context.getException());
		} else if (context.getException() instanceof ODataPreconditionRequiredException) {
			exception = new ServiceException(CdsErrorStatuses.ETAG_REQUIRED, context.getMessage(), context.getException());
		} else if (context.getException() instanceof ODataPreconditionFailedException) {
			exception = new ServiceException(CdsErrorStatuses.ETAG_FAILED, context.getMessage(), context.getException());
		} else {
			exception = new ServiceException(toErrorStatus(context), context.getMessage(), context.getException());
		}
		handleWithApplicationEvent(requestContext, exception, context);

		int statusCode = context.getHttpStatus().getStatusCode();
		if (statusCode >= 500 && statusCode < 600) {
			logger.error(context.getMessage(), context.getException());
		} else {
			logger.debug(context.getMessage(), context.getException());
		}

		ODataResponse odataResponse = EntityProvider.writeErrorDocument(context);
		return ODataResponse.fromResponse(odataResponse).header(HttpHeaders.CONTENT_TYPE, context.getContentType())
				.status(context.getHttpStatus()).build();
	}

	@VisibleForTesting
	static void handleWithApplicationEvent(RequestContext requestContext, ServiceException exception, ODataErrorContext errorContext) {
		ApplicationLifecycleService applicationLifecycleService = requestContext.getServiceCatalog()
			.getService(ApplicationLifecycleService.class, ApplicationLifecycleService.DEFAULT_NAME);
		try {
			ErrorResponse response = applicationLifecycleService.errorResponse(exception);

			Message message = response.getMessages().get(0);
			errorContext.setHttpStatus(HttpStatusCodes.fromStatusCode(response.getHttpStatus()));
			errorContext.setErrorCode(message.getCode() != null ? message.getCode() : String.valueOf(response.getHttpStatus()));
			errorContext.setSeverity(toExternalSeverity(message));
			errorContext.setMessage(message.getMessage());
			errorContext.setTarget(MessagesUtils.getTarget(message.getTarget()));

			errorContext.setErrorDetails(response.getMessages().stream().skip(1).map(m -> {
				ODataErrorContext detail = new ODataErrorContext();
				detail.setErrorCode(m.getCode());
				detail.setSeverity(toExternalSeverity(m));
				detail.setMessage(m.getMessage());
				detail.setTarget(MessagesUtils.getTarget(m.getTarget()));
				return detail;
			}).collect(Collectors.toList()));
		} catch (Exception e) {
			logger.error("Unexpected exception in error response handling", e);
			fallback(requestContext, errorContext);
		}
	}

	@VisibleForTesting
	static String negotiateContentType(RequestContext requestContext, String contentType) {
		// For error responses, only XML and JSON allowed.
		if (HttpContentType.APPLICATION_JSON.equals(contentType) ||
			HttpContentType.APPLICATION_JSON.equals(requestContext.getParameterInfo().getHeader(HttpHeaders.ACCEPT)) ||
			"json".equals(requestContext.getParameterInfo().getQueryParameter("format"))) {
			return HttpContentType.APPLICATION_JSON;
		} else {
			return HttpContentType.APPLICATION_XML;
		}
	}

	private static ErrorStatus toErrorStatus(ODataErrorContext context) {
		return new ErrorStatus() {

			@Override
			public int getHttpStatus() {
				if (context.getHttpStatus() == null) {
					return ErrorStatuses.SERVER_ERROR.getHttpStatus();
				}
				return context.getHttpStatus().getStatusCode();
			}

			@Override
			public String getCodeString() {
				return context.getErrorCode();
			}

		};
	}

	private static void fallback(RequestContext requestContext, ODataErrorContext errorContext) {
		ServiceException exp = new ErrorStatusException(ErrorStatuses.SERVER_ERROR);
		errorContext.setErrorCode(exp.getErrorStatus().getCodeString());
		errorContext.setException(exp);
		errorContext.setSeverity(Message.Severity.ERROR.toString().toLowerCase(Locale.ENGLISH));
		errorContext.setHttpStatus(HttpStatusCodes.fromStatusCode(exp.getErrorStatus().getHttpStatus()));
		errorContext.setMessage(exp.getLocalizedMessage(requestContext.getParameterInfo().getLocale()));
	}

	private static String toExternalSeverity(Message m) {
		return m.getSeverity().toString().toLowerCase(Locale.ENGLISH);
	}
}
