/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v2.processors;

import static com.sap.cds.adapter.odata.v2.utils.UriInfoUtils.getSimpleProperty;
import static jakarta.servlet.http.HttpServletResponse.SC_NOT_MODIFIED;
import static org.apache.olingo.odata2.api.ep.EntityProvider.writeBinary;
import static org.apache.olingo.odata2.api.ep.EntityProvider.writeEntry;
import static org.apache.olingo.odata2.api.ep.EntityProvider.writeFeed;
import static org.apache.olingo.odata2.api.ep.EntityProvider.writeFunctionImport;
import static org.apache.olingo.odata2.api.ep.EntityProvider.writeProperty;
import static org.apache.olingo.odata2.api.ep.EntityProvider.writeText;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.olingo.odata2.api.ODataCallback;
import org.apache.olingo.odata2.api.batch.BatchHandler;
import org.apache.olingo.odata2.api.batch.BatchRequestPart;
import org.apache.olingo.odata2.api.batch.BatchResponsePart;
import org.apache.olingo.odata2.api.commons.HttpHeaders;
import org.apache.olingo.odata2.api.commons.HttpStatusCodes;
import org.apache.olingo.odata2.api.commons.InlineCount;
import org.apache.olingo.odata2.api.commons.ODataHttpHeaders;
import org.apache.olingo.odata2.api.edm.EdmException;
import org.apache.olingo.odata2.api.edm.EdmFunctionImport;
import org.apache.olingo.odata2.api.edm.EdmProperty;
import org.apache.olingo.odata2.api.edm.EdmServiceMetadata;
import org.apache.olingo.odata2.api.edm.EdmType;
import org.apache.olingo.odata2.api.ep.EntityProvider;
import org.apache.olingo.odata2.api.ep.EntityProviderBatchProperties;
import org.apache.olingo.odata2.api.ep.EntityProviderException;
import org.apache.olingo.odata2.api.ep.EntityProviderReadProperties;
import org.apache.olingo.odata2.api.ep.EntityProviderWriteProperties;
import org.apache.olingo.odata2.api.ep.EntityProviderWriteProperties.ODataEntityProviderPropertiesBuilder;
import org.apache.olingo.odata2.api.ep.entry.ODataEntry;
import org.apache.olingo.odata2.api.exception.ODataException;
import org.apache.olingo.odata2.api.processor.ODataContext;
import org.apache.olingo.odata2.api.processor.ODataErrorContext;
import org.apache.olingo.odata2.api.processor.ODataRequest;
import org.apache.olingo.odata2.api.processor.ODataResponse;
import org.apache.olingo.odata2.api.processor.ODataResponse.ODataResponseBuilder;
import org.apache.olingo.odata2.api.processor.ODataSingleProcessor;
import org.apache.olingo.odata2.api.uri.ExpandSelectTreeNode;
import org.apache.olingo.odata2.api.uri.PathInfo;
import org.apache.olingo.odata2.api.uri.PathSegment;
import org.apache.olingo.odata2.api.uri.UriInfo;
import org.apache.olingo.odata2.api.uri.UriParser;
import org.apache.olingo.odata2.api.uri.info.DeleteUriInfo;
import org.apache.olingo.odata2.api.uri.info.GetEntitySetCountUriInfo;
import org.apache.olingo.odata2.api.uri.info.GetEntitySetUriInfo;
import org.apache.olingo.odata2.api.uri.info.GetEntityUriInfo;
import org.apache.olingo.odata2.api.uri.info.GetFunctionImportUriInfo;
import org.apache.olingo.odata2.api.uri.info.GetMediaResourceUriInfo;
import org.apache.olingo.odata2.api.uri.info.GetMetadataUriInfo;
import org.apache.olingo.odata2.api.uri.info.GetServiceDocumentUriInfo;
import org.apache.olingo.odata2.api.uri.info.GetSimplePropertyUriInfo;
import org.apache.olingo.odata2.api.uri.info.PostUriInfo;
import org.apache.olingo.odata2.api.uri.info.PutMergePatchUriInfo;
import org.apache.olingo.odata2.core.commons.ContentType;
import org.apache.olingo.odata2.core.commons.Encoder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Sets;
import com.sap.cds.adapter.odata.v2.CdsRequestGlobals;
import com.sap.cds.adapter.odata.v2.metadata.MetadataInfo;
import com.sap.cds.adapter.odata.v2.processors.request.CdsODataRequest;
import com.sap.cds.adapter.odata.v2.processors.request.PayloadProcessor;
import com.sap.cds.adapter.odata.v2.processors.response.CdsODataResponse;
import com.sap.cds.adapter.odata.v2.processors.response.ResultSetProcessor;
import com.sap.cds.adapter.odata.v2.query.NextLinkInfo;
import com.sap.cds.adapter.odata.v2.utils.CheckableInputStream;
import com.sap.cds.adapter.odata.v2.utils.ETagHelper;
import com.sap.cds.adapter.odata.v2.utils.MessagesUtils;
import com.sap.cds.adapter.odata.v2.utils.UriInfoUtils;
import com.sap.cds.services.ErrorStatuses;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ETagUtils;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.StringUtils;

public class OlingoProcessor extends ODataSingleProcessor {

	private static final Logger accessLogger = LoggerFactory.getLogger("com.sap.cds.adapter.odata.v2.BatchAccess");

	// these headers are not propagated from the parent batch request to its child requests
	private final static Set<String> batchHeaderPropagationBlacklist = new HashSet<>(Arrays.asList(
		HttpHeaders.ACCEPT, HttpHeaders.ACCEPT_ENCODING, HttpHeaders.ACCEPT_CHARSET,
		HttpHeaders.CONTENT_TYPE, HttpHeaders.CONTENT_LENGTH, HttpHeaders.CONTENT_ENCODING,
		HttpHeaders.CONTENT_LANGUAGE, HttpHeaders.CONTENT_LOCATION, "Content-ID",
		"Content-Transfer-Encoding", "MIME-Version",
		HttpHeaders.IF_MATCH, HttpHeaders.IF_NONE_MATCH
	));

	private final MetadataInfo metadataInfo;
	private final CdsProcessor cdsProcessor;
	private final CdsRequestGlobals globals;
	private final UriInfoUtils uriUtils;

	public OlingoProcessor(MetadataInfo metadataInfo, CdsRequestGlobals globals) {
		this.metadataInfo = metadataInfo;
		this.globals = globals;
		this.cdsProcessor = new CdsProcessor(globals);
		this.uriUtils = new UriInfoUtils(this.globals);
	}

	@Override
	public ODataResponse readEntity(GetEntityUriInfo uriInfo, String contentType) {
		CdsODataRequest request = new CdsODataRequest(getContext(), (UriInfo) uriInfo, contentType, globals);
		return cdsProcessor.processRequest(request, cdsProcessor::get, (response) -> {
			try {
				URI serviceRoot = getContext().getPathInfo().getServiceRoot();
				if (response.getStatusCode() == SC_NOT_MODIFIED) {
					return buildODataResponse(request, null, response, contentType);
				}
				Map<String, ODataCallback> callbacks = new HashMap<String, ODataCallback>();
				List<Map<String, Object>> data = ResultSetProcessor.postProcessResponsePayload(
						uriInfo.getTargetEntitySet().getEntityType(), response.getResult(), callbacks, serviceRoot,
						response.generateID(), true);
				EntityProviderWriteProperties writeProperties = getPropertiesBuilder(getContext(), (UriInfo) uriInfo).callbacks(callbacks).build();
				ODataResponse odataResponse = writeEntry(contentType, uriInfo.getTargetEntitySet(), data.get(0), writeProperties);
				return buildODataResponse(request, odataResponse, response, contentType);
			} catch (EntityProviderException e) {
				throw new ErrorStatusException(CdsErrorStatuses.RESPONSE_SERIALIZATION_FAILED, e);
			} catch (ODataException e) {
				throw new ServiceException(e);
			}
		});
	}

	@Override
	public ODataResponse readEntitySet(final GetEntitySetUriInfo uriInfo, final String contentType) {
		CdsODataRequest request = new CdsODataRequest(getContext(), (UriInfo) uriInfo, contentType, globals);
		return cdsProcessor.processRequest(request, cdsProcessor::get, (response) -> {
			try {
				URI serviceRoot = getContext().getPathInfo().getServiceRoot();
				ODataEntityProviderPropertiesBuilder propertiesBuilder = getPropertiesBuilder(getContext(), (UriInfo) uriInfo);

				long inlineCount = response.getResult().inlineCount();
				if (inlineCount >= 0) {
					propertiesBuilder.inlineCount((int) inlineCount);
					propertiesBuilder.inlineCountType(InlineCount.ALLPAGES);
				}

				NextLinkInfo nextLinkInfo = response.getNextLinkInfo();
				if(nextLinkInfo != null && response.getResult().rowCount() >= nextLinkInfo.getPageSize()) {
					final String skiptoken = "$skiptoken";
					String odataPath = getContext().getPathInfo().getODataSegments().stream().map(e -> e.getPath()).collect(Collectors.joining("/"));
					Map<String, String> query = ((org.apache.olingo.odata2.core.ODataRequestImpl)getContext().getParameter("~odataRequest")).getQueryParameters();
					String filteredQuery = query.entrySet().stream().map(e -> e.getKey() + "=" + Encoder.encode(e.getValue())).filter(q -> !q.startsWith(skiptoken)).collect(Collectors.joining("&"));
					String skiptokenQuery = skiptoken + "=" + nextLinkInfo.getNextSkipToken();
					String fullQuery = StringUtils.isEmpty(filteredQuery) ? skiptokenQuery : (filteredQuery + "&" + skiptokenQuery);
					String nextLink = odataPath + "?" + fullQuery;
					propertiesBuilder.nextLink(nextLink);
				}

				Map<String, ODataCallback> callbacks = new HashMap<String, ODataCallback>();
				List<Map<String, Object>> data = ResultSetProcessor.postProcessResponsePayload(
						uriInfo.getTargetEntitySet().getEntityType(), response.getResult(), callbacks, serviceRoot,
						response.generateID(), true);
				EntityProviderWriteProperties writeProperties = propertiesBuilder.callbacks(callbacks).build();
				ODataResponse odataResponse = writeFeed(contentType, uriInfo.getTargetEntitySet(), data, writeProperties);
				return buildODataResponse(request, odataResponse, response, contentType);
			} catch (EntityProviderException e) {
				throw new ErrorStatusException(CdsErrorStatuses.RESPONSE_SERIALIZATION_FAILED, e);
			} catch (ODataException e) {
				throw new ServiceException(e);
			}
		});
	}

	@Override
	public ODataResponse countEntitySet(final GetEntitySetCountUriInfo uriInfo, final String contentType) {
		CdsODataRequest request = new CdsODataRequest(getContext(), (UriInfo) uriInfo, contentType, globals);
		return cdsProcessor.processRequest(request, cdsProcessor::get, (response) -> {
			try {
				ODataResponse odataResponse = writeText(toText(response));
				return buildODataResponse(request, odataResponse, response, contentType);
			} catch (EntityProviderException e) {
				throw new ErrorStatusException(CdsErrorStatuses.RESPONSE_SERIALIZATION_FAILED, e);
			}
		});
	}

	@Override
	public ODataResponse readEntitySimpleProperty(final GetSimplePropertyUriInfo uriInfo, final String contentType) {
		CdsODataRequest request = new CdsODataRequest(getContext(), (UriInfo) uriInfo, contentType, globals);
		return cdsProcessor.processRequest(request, cdsProcessor::get, (response) -> {
			try {
				EdmProperty property = getSimpleProperty((UriInfo) uriInfo);
				ODataResponse odataResponse = writeProperty(contentType, property, toSingleValue(response, property));
				return buildODataResponse(request, odataResponse, response, contentType);
			} catch (EntityProviderException e) {
				throw new ErrorStatusException(CdsErrorStatuses.RESPONSE_SERIALIZATION_FAILED, e);
			}
		});
	}

	@Override
	public ODataResponse readEntitySimplePropertyValue(final GetSimplePropertyUriInfo uriInfo, final String contentType) {
		CdsODataRequest request = new CdsODataRequest(getContext(), (UriInfo) uriInfo, contentType, globals);
		return cdsProcessor.processRequest(request, cdsProcessor::get, (response) -> {
			try {
				ODataResponse odataResponse = writeText(toText(response));
				return buildODataResponse(request, odataResponse, response, contentType);
			} catch (EntityProviderException e) {
				throw new ErrorStatusException(CdsErrorStatuses.RESPONSE_SERIALIZATION_FAILED, e);
			}
		});
	}

	@Override
	public ODataResponse deleteEntity(final DeleteUriInfo uriInfo, final String contentType) {
		CdsODataRequest request = new CdsODataRequest(getContext(), (UriInfo) uriInfo, contentType, globals);
		return cdsProcessor.processRequest(request, cdsProcessor::delete, (response) -> {
			return buildODataResponse(request, null, response, contentType);
		});
	}

	@Override
	public ODataResponse createEntity(final PostUriInfo uriInfo, final InputStream content, final String requestContentType, final String contentType) throws ODataException {
		CdsODataRequest request = new CdsODataRequest(getContext(), (UriInfo) uriInfo, contentType, globals);

		URI serviceRoot = getContext().getPathInfo().getServiceRoot();
		ODataEntityProviderPropertiesBuilder propertiesBuilder = EntityProviderWriteProperties.serviceRoot(serviceRoot);

		CheckableInputStream checkableContent = new CheckableInputStream(content);
		try {
			EntityProviderReadProperties properties = EntityProviderReadProperties.init().mergeSemantic(false).build();
			ODataEntry entry = EntityProvider.readEntry(requestContentType, uriInfo.getTargetEntitySet(), checkableContent, properties);
			propertiesBuilder.expandSelectTree(entry.getExpandSelectTree());
			request.setBodyMap(PayloadProcessor.processRequestPayload(uriInfo.getTargetType(), entry));
		} catch (EntityProviderException e) {
			if(checkableContent.isDataRead()) {
				throw new ErrorStatusException(CdsErrorStatuses.INVALID_PAYLOAD, ExceptionUtils.getRootCause(e).getLocalizedMessage(), e);
			} else {
				// we gracefully handle the no content available case
				request.setBodyMap(new HashMap<>());
			}
		}

		return cdsProcessor.processRequest(request, cdsProcessor::post, (response) -> {
			try {
				Map<String, ODataCallback> callbacks = new HashMap<String, ODataCallback>();
				List<Map<String, Object>> data = ResultSetProcessor.postProcessResponsePayload(uriInfo.getTargetEntitySet().getEntityType(), response.getResult(), callbacks, serviceRoot);
				propertiesBuilder.callbacks(callbacks).isDataBasedPropertySerialization(true);
				ODataResponse odataResponse = writeEntry(contentType, uriInfo.getTargetEntitySet(), data.get(0), propertiesBuilder.build());
				return buildODataResponse(request, odataResponse, response, contentType);
			} catch (EntityProviderException e) {
				throw new ErrorStatusException(CdsErrorStatuses.RESPONSE_SERIALIZATION_FAILED, e);
			} catch (ODataException e) {
				throw new ServiceException(e);
			}
		});
	}

	@Override
	public ODataResponse updateEntity(final PutMergePatchUriInfo uriInfo, final InputStream content, final String requestContentType, final boolean merge, final String contentType) throws ODataException {
		CdsODataRequest request = new CdsODataRequest(getContext(), (UriInfo) uriInfo, contentType, globals);

		URI serviceRoot = getContext().getPathInfo().getServiceRoot();
		ODataEntityProviderPropertiesBuilder propertiesBuilder = EntityProviderWriteProperties.serviceRoot(serviceRoot);

		try {
			EntityProviderReadProperties properties = EntityProviderReadProperties.init().mergeSemantic(merge).build();
			ODataEntry entry = EntityProvider.readEntry(requestContentType, uriInfo.getTargetEntitySet(), content, properties);
			propertiesBuilder.expandSelectTree(entry.getExpandSelectTree());
			request.setBodyMap(PayloadProcessor.processRequestPayload(uriInfo.getTargetType(), entry));
		} catch (EntityProviderException e) {
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_PAYLOAD, ExceptionUtils.getRootCause(e).getLocalizedMessage());
		}

		return cdsProcessor.processRequest(request, merge ? cdsProcessor::patch : cdsProcessor::put, (response) -> {
			try {
				ODataResponse odataResponse = null;
				String etag = null;

				if (response.getStatusCode() != 204) {
					Map<String, ODataCallback> callbacks = new HashMap<String, ODataCallback>();
					List<Map<String, Object>> data = ResultSetProcessor.postProcessResponsePayload(uriInfo.getTargetEntitySet().getEntityType(), response.getResult(), callbacks, serviceRoot);
					propertiesBuilder.callbacks(callbacks).isDataBasedPropertySerialization(true);
					odataResponse = writeEntry(contentType, uriInfo.getTargetEntitySet(), data.get(0), propertiesBuilder.build());
				} else {
					etag = ETagHelper.getEtag(response.getResult().single(), globals.getModel().getEntity(request.getLastResourceName()));
				}

				return buildODataResponse(request, odataResponse, response, contentType, etag);
			} catch (EntityProviderException e) {
				throw new ErrorStatusException(CdsErrorStatuses.RESPONSE_SERIALIZATION_FAILED, e);
			} catch (ODataException e) {
				throw new ServiceException(e);
			}
		});
	}

	@Override
	public ODataResponse executeFunctionImport(final GetFunctionImportUriInfo uriInfo, final String contentType) {
		CdsODataRequest cdsRequest = new CdsODataRequest(getContext(), (UriInfo) uriInfo, contentType, globals);
		return cdsProcessor.processRequest(cdsRequest, cdsProcessor::function, (response) -> {
			try {
				URI serviceRoot = getContext().getPathInfo().getServiceRoot();
				ODataEntityProviderPropertiesBuilder propertiesBuilder = EntityProviderWriteProperties.serviceRoot(serviceRoot);
				ODataResponse odataResponse = null;
				if (response.getStatusCode() != HttpStatusCodes.NO_CONTENT.getStatusCode()) {
					EdmFunctionImport functionImport = uriInfo.getFunctionImport();
					EdmType returnType = functionImport.getReturnType().getType();
					Map<String, ODataCallback> callbacks = new HashMap<String, ODataCallback>();
					List<Map<String, Object>> data = ResultSetProcessor.postProcessResponsePayload(returnType,
							response.getResult(), callbacks, serviceRoot);

					odataResponse = writeFunctionImport(contentType, uriInfo.getFunctionImport(),
							ResultSetProcessor.getReturnValueOfFunction(functionImport, data),
							propertiesBuilder.callbacks(callbacks).build());
				}
				return buildODataResponse(cdsRequest, odataResponse, response, contentType);
			} catch (EntityProviderException e) {
				throw new ErrorStatusException(CdsErrorStatuses.RESPONSE_SERIALIZATION_FAILED, e);
			} catch (ODataException e) {
				throw new ServiceException(e);
			}
		});
	}

	@Override
	public ODataResponse readMetadata(final GetMetadataUriInfo uriInfo, final String contentType)
			throws ODataException {
		boolean isNotModified = false;
		ODataResponseBuilder odataResponseBuilder = ODataResponse.newBuilder();
		if (metadataInfo.getEtag() != null) {
			String etag = ETagUtils.createETagHeaderValue(metadataInfo.getEtag());
			// Set application etag at response
			odataResponseBuilder.header(HttpHeaders.ETAG, etag);
			// Check if metadata document has been modified
			isNotModified = ETagHelper.checkReadPreconditions(etag,
					getContext().getRequestHeader(HttpHeaders.IF_MATCH), getContext().getRequestHeader(HttpHeaders.IF_NONE_MATCH));
			// allow caching, but enforce revalidation using etags
			odataResponseBuilder.header(HttpHeaders.CACHE_CONTROL, "max-age=0");
		}
		// Send the correct response
		if (isNotModified) {
			odataResponseBuilder.status(HttpStatusCodes.NOT_MODIFIED);
		} else {
			// HTTP HEAD requires no payload but a 200 OK response
			if (getContext().getHttpMethod().equalsIgnoreCase("HEAD")) {
				odataResponseBuilder.status(HttpStatusCodes.OK);
			} else {
				EdmServiceMetadata edmServiceMetadata = getContext().getService().getEntityDataModel().getServiceMetadata();
				odataResponseBuilder.entity(new ByteArrayInputStream(metadataInfo.getEdmxBytes()))
						.header(HttpHeaders.CONTENT_TYPE, contentType)
						.header(ODataHttpHeaders.DATASERVICEVERSION, edmServiceMetadata.getDataServiceVersion());
			}
		}
		return odataResponseBuilder.build();
	}

	@Override
	public ODataResponse readServiceDocument(GetServiceDocumentUriInfo uriInfo, String contentType) throws ODataException {
		boolean isNotModified = false;
		ODataResponseBuilder odataResponseBuilder = ODataResponse.newBuilder();
		if (metadataInfo.getEtag() != null) {
			String etag = ETagUtils.createETagHeaderValue(metadataInfo.getEtag());
			// Set application etag at response
			odataResponseBuilder.header(HttpHeaders.ETAG, etag);
			// Check if metadata document has been modified
			isNotModified = ETagHelper.checkReadPreconditions(etag,
					getContext().getRequestHeader(HttpHeaders.IF_MATCH), getContext().getRequestHeader(HttpHeaders.IF_NONE_MATCH));
			// allow caching, but enforce revalidation using etags
			odataResponseBuilder.header(HttpHeaders.CACHE_CONTROL, "max-age=0");
		}
		// Send the correct response
		if (isNotModified) {
			odataResponseBuilder.status(HttpStatusCodes.NOT_MODIFIED);
		} else {
			// HTTP HEAD requires no payload but a 200 OK response
			if (getContext().getHttpMethod().equalsIgnoreCase("HEAD")) {
				odataResponseBuilder.status(HttpStatusCodes.OK);
			} else {
				EdmServiceMetadata edmServiceMetadata = getContext().getService().getEntityDataModel().getServiceMetadata();
				ODataResponse serviceDocument = super.readServiceDocument(uriInfo, contentType);
				odataResponseBuilder.entity(serviceDocument.getEntity())
						.header(HttpHeaders.CONTENT_TYPE, contentType)
						.header(ODataHttpHeaders.DATASERVICEVERSION, edmServiceMetadata.getDataServiceVersion());
			}
		}
		return odataResponseBuilder.build();
	}

	@Override
	public ODataResponse executeBatch(final BatchHandler handler, final String contentType, final InputStream content) throws ODataException {
		PathInfo pathInfo = getContext().getPathInfo();
		EntityProviderBatchProperties batchProperties = EntityProviderBatchProperties.init().pathInfo(pathInfo).build();

		List<BatchRequestPart> parts = EntityProvider.parseBatchRequest(contentType, content, batchProperties);
		long requestLimit = globals.getRuntime().getEnvironment().getCdsProperties().getOdataV2().getBatch().getMaxRequests();
		if (requestLimit >= 0 && requestLimit < parts.stream().mapToLong(p -> p.getRequests().size()).sum()) {
			throw new ErrorStatusException(CdsErrorStatuses.BATCH_TOO_MANY_REQUESTS);
		}
		List<BatchResponsePart> responseParts = new ArrayList<>();

		Set<String> allowedParentHeaders = Sets.filter(getContext().getRequestHeaders().keySet(),
				h -> batchHeaderPropagationBlacklist.stream().noneMatch(b -> h.equalsIgnoreCase(b)));
		for (BatchRequestPart part : parts) {
			for (ODataRequest req : part.getRequests()) {
				for (String header : allowedParentHeaders) {
					if (StringUtils.isEmpty(req.getRequestHeaderValue(header))) {
						String parentHeader = getContext().getRequestHeader(header);
						if (!StringUtils.isEmpty(parentHeader)) {
							req.getRequestHeaders().put(header, new ArrayList<>(Arrays.asList(parentHeader)));
						}
					}
				}
			}

			// Execute each batch request
			BatchResponsePart responsePart = handler.handleBatchPart(part);
			logBatchRequest(part, responsePart);
			responseParts.add(responsePart);
		}

		return EntityProvider.writeBatchResponse(responseParts);
	}

	@Override
	public BatchResponsePart executeChangeSet(final BatchHandler handler, final List<ODataRequest> requests) throws ODataException {
		try {
			return globals.getRuntime().changeSetContext().run((context) -> {
				List<ODataResponse> responses = new ArrayList<>();
				for (ODataRequest request : requests) {
					ODataResponse response;
					try {
						response = handler.handleRequest(request);
					} catch (ODataException e) {
						context.markForCancel();
						throw new ErrorStatusException(ErrorStatuses.SERVER_ERROR, e);
					}

					if(response.getStatus().getStatusCode() >= 200 && response.getStatus().getStatusCode() < 400) {
						// only collect success responses
						responses.add(response);
					} else {
						context.markForCancel();
						return BatchResponsePart.responses(Arrays.asList(response)).changeSet(false).build();
					}
				}
				return BatchResponsePart.responses(responses).changeSet(true).build();
			});
		} catch (Exception e) { // NOSONAR
			ODataErrorContext context = new ODataErrorContext();
			context.setException(e);
			// TODO can we do better here in guessing the content type?
			context.setContentType(ContentType.APPLICATION_XML.toContentTypeString());
			ODataResponse response = new ErrorCallback(globals.getRuntime()).handleError(context);
			return BatchResponsePart.responses(Arrays.asList(response)).changeSet(false).build();
		}
	}

	@Override
	public ODataResponse readEntityMedia(final GetMediaResourceUriInfo getMediaResourceUriInfo, final String contentType) {
		CdsODataRequest request = new CdsODataRequest(getContext(), (UriInfo) getMediaResourceUriInfo, contentType, globals);
		return cdsProcessor.processRequest(request, cdsProcessor::get, (response) -> {
			try {
				String mimeType = (response.getMimeType() == null) ? ContentType.APPLICATION_OCTET_STREAM.toContentTypeString() : response.getMimeType();
				InputStream is = toInputStream(response);
				if (null != is) {
					ODataResponse odataResponse = writeBinary(mimeType, IOUtils.toByteArray(is));
					return buildODataResponse(request, odataResponse, response, contentType);
				}

				throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_FOUND, this.uriUtils.getTargetEntityName(request.getUriInfo()));
			} catch (EntityProviderException | IOException e) {
				throw new ErrorStatusException(CdsErrorStatuses.RESPONSE_SERIALIZATION_FAILED, e);
			}
		});
	}

	@Override
	public ODataResponse deleteEntityMedia(final DeleteUriInfo uriInfo, final String contentType) {
		CdsODataRequest request = new CdsODataRequest(getContext(), (UriInfo) uriInfo, contentType, globals);
		return cdsProcessor.processRequest(request, cdsProcessor::delete, (response) -> {
			return buildODataResponse(request, null, response, contentType);
		});
	}

	@Override
	public ODataResponse updateEntityMedia(final PutMergePatchUriInfo uriInfo, final InputStream content, final String requestContentType, final String contentType) {
		CdsODataRequest cdsRequest = new CdsODataRequest(getContext(), (UriInfo) uriInfo, contentType, globals);
		try {
			final byte[] binaryContent = EntityProvider.readBinary(content);
			Map<String, Object> data = new HashMap<String, Object>();
			data.put(CdsProcessor.VALUE_KEY, binaryContent);
			cdsRequest.setBodyMap(data);
		} catch (EntityProviderException e) {
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_PAYLOAD, ExceptionUtils.getRootCause(e).getLocalizedMessage());
		}

		return cdsProcessor.processRequest(cdsRequest, cdsProcessor::patch, (response) -> {
			return buildODataResponse(cdsRequest, null, response, contentType);
		});
	}

	private ODataEntityProviderPropertiesBuilder getPropertiesBuilder(ODataContext oDataContext, UriInfo uriInfo) throws ODataException {
		URI serviceRoot = oDataContext.getPathInfo().getServiceRoot();
		ExpandSelectTreeNode expandSelectTree = UriParser.createExpandSelectTree(uriInfo.getSelect(), uriInfo.getExpand());
		return EntityProviderWriteProperties.serviceRoot(serviceRoot).expandSelectTree(expandSelectTree);
	}

	private ODataResponse buildODataResponse(CdsODataRequest cdsRequest, ODataResponse odataResponse, CdsODataResponse cdsResponse, String contentType) {
		return buildODataResponse(cdsRequest, odataResponse, cdsResponse, contentType, null);
	}

	private ODataResponse buildODataResponse(CdsODataRequest cdsRequest, ODataResponse odataResponse, CdsODataResponse cdsResponse, String contentType, String etag) {
		ODataResponseBuilder odataResponseBuilder = null;
		if (odataResponse != null) {
			odataResponseBuilder = ODataResponse.fromResponse(odataResponse);
		} else {
			odataResponseBuilder = ODataResponse.newBuilder();
		}
		odataResponseBuilder = odataResponseBuilder.status(HttpStatusCodes.fromStatusCode(cdsResponse.getStatusCode()))
				.header(HttpHeaders.CONTENT_TYPE, contentType);

		// SAP Messages
		String sapMessageHeader = MessagesUtils.getSapMessagesHeader(RequestContext.getCurrent(globals.getRuntime()).getMessages(), contentType);
		if(!StringUtils.isEmpty(sapMessageHeader)) {
			odataResponseBuilder.header("sap-message", sapMessageHeader);
		}
		if (!StringUtils.isEmpty(etag)) {
			odataResponseBuilder.header("ETag", etag);
		}
		return odataResponseBuilder.build();
	}

	private static String toText(CdsODataResponse response) {
		return ResultSetProcessor.resultToString(response.getResult());
	}

	private static Object toSingleValue(CdsODataResponse response, EdmProperty property) {
		try {
			return response.getResult().single().get(property.getName());
		} catch (EdmException e) {
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_PAYLOAD, ExceptionUtils.getRootCause(e).getLocalizedMessage());
		}
	}

	private static InputStream toInputStream(CdsODataResponse response) {
		return ResultSetProcessor.rowToSingleValue(response.getResult().first(), InputStream.class);
	}

	private static void logBatchRequest(BatchRequestPart part, BatchResponsePart responsePart) {
		if (accessLogger.isInfoEnabled()) {
			List<ODataRequest> requests = part.getRequests();
			List<ODataResponse> responses = responsePart.getResponses();

			for (int i = 0; i < requests.size(); i++) {
				StringBuilder builder = new StringBuilder();

				ODataRequest request = requests.get(i);
				builder.append("$batch ");
				builder.append(request.getMethod()).append(" ");
				builder.append(request.getPathInfo().getODataSegments().stream().map(PathSegment::getPath).collect(Collectors.joining("/")));

				ODataResponse response;
				if (responses.size() > i) {
					response = responses.get(i);
				} else {
					response = responses.get(0);
				}
				builder.append(" ").append(response.getStatus().getStatusCode());

				accessLogger.info(builder.toString());
			}
		}
	}
}
