/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v2.processors.request;

import static java.util.Optional.ofNullable;

import java.time.Instant;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.olingo.odata2.api.processor.ODataContext;
import org.apache.olingo.odata2.api.processor.ODataRequest;
import org.apache.olingo.odata2.api.uri.UriInfo;

import com.sap.cds.adapter.odata.v2.CdsRequestGlobals;
import com.sap.cds.adapter.odata.v2.utils.UriInfoUtils;
import com.sap.cds.services.request.ParameterInfo;
import com.sap.cds.services.utils.CorrelationIdUtils;
import com.sap.cds.services.utils.HttpHeaders;
import com.sap.cds.services.utils.LocaleUtils;
import com.sap.cds.services.utils.QueryParameters;
import com.sap.cds.services.utils.TemporalRangeUtils;
import org.apache.olingo.odata2.api.exception.ODataException;

public class CdsODataRequest implements ParameterInfo {

	private final ODataContext odataContext;
	private final UriInfo uriInfo;

	private final Map<String, String> queryParameters = new HashMap<>();
	private final String contentType;
	private Map<String, Object> bodyMap;

	// required to be able to pass the value from one thread to another
	private String correlationId;

	private Locale locale;
	private Instant validFrom;
	private Instant validTo;
	private final CdsRequestGlobals globals;
	private final UriInfoUtils uriUtils;

	public CdsODataRequest(ODataContext odataContext, UriInfo uriInfo, String contentType, CdsRequestGlobals globals) {
		this.odataContext = odataContext;
		this.uriInfo = uriInfo;
		this.contentType = contentType;
		this.globals = globals;
		this.uriUtils = new UriInfoUtils(this.globals);
		ODataRequest odataRequest = ((ODataRequest) odataContext.getParameter("~odataRequest"));
		queryParameters.putAll(odataRequest.getQueryParameters().entrySet().stream().collect(
				Collectors.toMap(Map.Entry::getKey, e -> ofNullable(getHeader(e.getKey())).orElse(e.getValue()))));
	}

	/**
	 * Get the fully qualified name of the last entity
	 * @return fully qualified name of the last entity
	 */
	public String getLastResourceName() {
		String resourceName = this.uriUtils.getTargetEntityName(uriInfo);
		String cdsName = this.globals.getCdsEntityNames().get(resourceName);
		if (cdsName != null) {
			resourceName = cdsName;
		}
		return resourceName;
	}

	public boolean isRawValueResourceRequest() {

		try {
			return odataContext.getPathInfo().getRequestUri().getPath().endsWith("$value");
		} catch (ODataException exception) {
			return false;
		}
	}

	public UriInfo getUriInfo() {
		return uriInfo;
	}

	public String getContentType() {
		return contentType;
	}

	public void setBodyMap(Map<String, Object> data) {
		this.bodyMap = data;
	}

	public Map<String, Object> getBodyMap() {
		return bodyMap;
	}

	// Request Parameters API

	@Override
	public String getCorrelationId() {
		if (correlationId == null) {
			correlationId = CorrelationIdUtils.getOrGenerateCorrelationId(this);
		}
		return correlationId;
	}

	@Override
	public String getHeader(String id) {
		return odataContext.getRequestHeader(id);
	}

	@Override
	public Map<String, String> getHeaders() {
		return odataContext.getRequestHeaders().entrySet().stream().collect(
				Collectors.toMap(Map.Entry::getKey, e -> getHeader(e.getKey()) )) ; // projects List<String> to String
	}

	@Override
	public String getQueryParameter(String key) {
		return queryParameters.get(key);
	}

	@Override
	public Map<String, String> getQueryParams() {
		return queryParameters;
	}

	@Override
	public Locale getLocale() {
		if(this.locale == null) {
			this.locale = new LocaleUtils(globals.getRuntime().getEnvironment().getCdsProperties()).getLocale(
					getQueryParameter(QueryParameters.SAP_LOCALE),
					getHeader(HttpHeaders.ACCEPT_LANGUAGE),
					getQueryParameter(QueryParameters.SAP_LANGUAGE),
					getHeader(HttpHeaders.X_SAP_REQUEST_LANGUAGE_HEADER));
		}
		return this.locale;
	}

	@Override
	public Instant getValidFrom() {
		if (this.validFrom == null) {
			initTemporalRange();
		}
		return this.validFrom;
	}

	@Override
	public Instant getValidTo() {
		if (this.validTo == null) {
			initTemporalRange();
		}
		return this.validTo;
	}

	private void initTemporalRange() {
		Instant[] range = TemporalRangeUtils.getTemporalRanges(
				getQueryParameter(QueryParameters.VALID_FROM),
				getQueryParameter(QueryParameters.VALID_TO),
				getQueryParameter(QueryParameters.VALID_AT));

		this.validFrom = range[0];
		this.validTo = range[1];
	}

}
