/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v2.processors.request;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.olingo.odata2.api.edm.EdmAnnotatable;
import org.apache.olingo.odata2.api.edm.EdmEntityType;
import org.apache.olingo.odata2.api.edm.EdmException;
import org.apache.olingo.odata2.api.edm.EdmNavigationProperty;
import org.apache.olingo.odata2.api.edm.EdmProperty;
import org.apache.olingo.odata2.api.edm.EdmType;
import org.apache.olingo.odata2.api.ep.entry.ODataEntry;
import org.apache.olingo.odata2.api.ep.feed.ODataFeed;

import com.sap.cds.adapter.odata.v2.utils.TypeConverterUtils;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

public class PayloadProcessor {

	/**
	 * Certain primitive types like UUID, Time, DateTime, DateTimeOffset and Date
	 * have to be converted to String, Instant
	 * since Olingo V2 does not support Java8 date and Time data types.
	 * Also, the conversion should happen for parent entity + all levels of child entity
	 * @param edmType EdmType
	 * @param odataEntry ODataEntry
	 * @return Map which includes converted values
	 */
	@SuppressWarnings("unchecked")
	public static Map<String, Object> processRequestPayload(EdmType edmType, ODataEntry odataEntry) {
		return (Map<String, Object>) processRow(edmType, odataEntry);
	}

	private static Map<String, Object> processProperty(EdmType edmType, EdmAnnotatable property, String key, Object value) {
		Map<String, Object> finalResult = new HashMap<>();
		if (value instanceof ODataEntry entry) {
			finalResult.put(key, processRow(edmType, entry));
		} else if (value instanceof ODataFeed feed) {
			List<Map<String, Object>> list = new ArrayList<>();
			for (ODataEntry entry : feed.getEntries()) {
				list.add(processRequestPayload(edmType, entry));
			}
			finalResult.put(key, list);
		} else {
			finalResult.put(key, TypeConverterUtils.getValueBasedOnTypeOfRequestPayload(edmType, property, value));
		}
		return finalResult;
	}

	@SuppressWarnings({ "unchecked" })
	private static Object processRow(EdmType edmType, ODataEntry odataEntry) {
		Object resValue = null;
		try {
			switch(edmType.getKind()) {
			case ENTITY:
				EdmEntityType edmEntityType = (EdmEntityType) edmType;
				List<String> properties = edmEntityType.getPropertyNames();
				List<String> navProperties = edmEntityType.getNavigationPropertyNames();
				resValue = new HashMap<>();
				for (Entry<String, Object> entry : odataEntry.getProperties().entrySet()) {
					String key = entry.getKey();
					Object value = entry.getValue();
					EdmAnnotatable property = null;
					EdmType type = null;
					if (properties.contains(key)) {
						try {
							EdmProperty edmProperty = (EdmProperty) edmEntityType.getProperty(key);
							property = edmProperty;
							type = edmProperty.getType();
						} catch (EdmException e) {
							throw new ErrorStatusException(CdsErrorStatuses.MISSING_EDM_PROPERTY, key,
									edmEntityType.getName(), e);
						}
					}
					if (navProperties.contains(key)) {
						try {
							EdmNavigationProperty edmNavProperty = (EdmNavigationProperty) edmEntityType.getProperty(key);
							property = edmNavProperty;
							type = edmNavProperty.getType();
						} catch (EdmException e) {
							throw new ErrorStatusException(CdsErrorStatuses.MISSING_EDM_PROPERTY, key,
									edmEntityType.getName(), e);
						}
					}
					if(type != null) {
						((Map<String, Object>)resValue).putAll(processProperty(type, property, key, value));
					}
				}
				break;
			default:
				throw new ErrorStatusException(CdsErrorStatuses.UNSUPPORTED_ACTION_FUNCTION_RETURN_TYPE, edmType.getKind().name());
			}
		} catch (EdmException ex) {
			// Never thrown by Olingo, as methods getPropertyNames() and
			// getNavigationPropertyNames() simply return a List.
			throw new IllegalStateException(ex);
		}

		return resValue;
	}

}
