/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v2.processors.response;

import static com.sap.cds.adapter.odata.v2.utils.AggregateTransformation.AGGREGATE_ID;
import static com.sap.cds.adapter.odata.v2.utils.TypeConverterUtils.getValueBasedOnTypeOfResultSet;
import static java.util.Collections.emptyMap;

import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;

import org.apache.olingo.odata2.api.ODataCallback;
import org.apache.olingo.odata2.api.edm.EdmComplexType;
import org.apache.olingo.odata2.api.edm.EdmEntityType;
import org.apache.olingo.odata2.api.edm.EdmException;
import org.apache.olingo.odata2.api.edm.EdmFunctionImport;
import org.apache.olingo.odata2.api.edm.EdmMultiplicity;
import org.apache.olingo.odata2.api.edm.EdmNavigationProperty;
import org.apache.olingo.odata2.api.edm.EdmProperty;
import org.apache.olingo.odata2.api.edm.EdmType;
import org.apache.olingo.odata2.api.edm.EdmTypeKind;
import org.apache.olingo.odata2.api.edm.EdmTyped;

import com.sap.cds.Result;
import com.sap.cds.Row;
import com.sap.cds.adapter.odata.v2.processors.ExpandWriteCallback;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

public class ResultSetProcessor {
	/**
	 * Certain primitive types like string UUID, Time, DateTime, DateTimeOffset and Date
	 * have to be converted to UUID, java.sql.Time, java.util.GregorianCalendar, java.sql.Date
	 * since Olingo V2 does not support Java8 date and Time data types.
	 * Also, the conversion should happen for parent entity + all levels of child entity
	 * @param edmType EdmType
	 * @param result Result
	 * @param callbacks Map of String, ODataCallback
	 * @param baseUri URI
	 * @return List of Maps or List of Objects
	 */

	public static List<Map<String, Object>> postProcessResponsePayload(EdmType edmType, Result result, Map<String, ODataCallback> callbacks, URI baseUri) {
		return postProcessResponsePayload( edmType,  result,  callbacks,  baseUri, false, false);
	}

	/**
	 * Certain primitive types like string UUID, Time, DateTime, DateTimeOffset and Date
	 * have to be converted to UUID, java.sql.Time, java.util.GregorianCalendar, java.sql.Date
	 * since Olingo V2 does not support Java8 date and Time data types.
	 * Also, the conversion should happen for parent entity + all levels of child entity
	 * @param edmType EdmType
	 * @param result Result
	 * @param callbacks Map of String, ODataCallback
	 * @param baseUri URI
	 * @param generateID in case of aggregate entities, if true, generate corresponding response ID
	 * @param deferredExpand defer loading of navigation properties which are not in expand list
	 * @return List of Maps or List of Objects
	 */
	public static List<Map<String, Object>> postProcessResponsePayload(EdmType edmType, Result result,
			Map<String, ODataCallback> callbacks, URI baseUri, boolean generateID, boolean deferredExpand) {
		if (null == result || result.first().isEmpty()) {
			return Collections.emptyList();
		}
		return processResultList(edmType, result.stream().collect(Collectors.toList()), callbacks, baseUri, generateID, deferredExpand);
	}

	private static List<Map<String, Object>> processResultList(EdmType edmType, List<Map<String, Object>> list,
			Map<String, ODataCallback> callbacks, URI baseUri, boolean generateID, boolean deferredExpand) {
		List<Map<String, Object>> processedResultList = new ArrayList<>();
		for (Map<String, Object> row : list) {
			if (generateID) {
				row.put(AGGREGATE_ID, UUID.randomUUID().toString());
			}
			processedResultList.add(processRow(edmType, row, callbacks, baseUri, generateID, deferredExpand));
		}
		return processedResultList;
	}

	@SuppressWarnings({ "unchecked" })
	private static Map<String, Object> processProperty(EdmType edmType, String key, Object value,
			Map<String, ODataCallback> callbacks, URI baseUri, boolean generateID, boolean deferredExpand) {
		Map<String, Object> finalResult = new HashMap<>();
		if (value instanceof Map) {
			finalResult.put(key, processRow(edmType, (Map<String, Object>) value, callbacks, baseUri, generateID, deferredExpand));
		} else if (value instanceof List) {
			List<Map<String, Object>> list = processResultList(edmType, (List<Map<String, Object>>) value, callbacks,
					baseUri, generateID, deferredExpand);
			finalResult.put(key, list);
		} else {
			finalResult.put(key, getValueBasedOnTypeOfResultSet(edmType, value));
		}
		return finalResult;
	}

	@SuppressWarnings({ "unchecked", "rawtypes" })
	private static Map<String, Object> processRow(EdmType edmType, Map<String, Object> row,
			Map<String, ODataCallback> callbacks, URI baseUri, boolean generateID, boolean deferredExpand) {
		Map<String, Object> resValue = null;
		try {
			switch(edmType.getKind()) {
			case ENTITY:
				EdmEntityType edmEntityType = (EdmEntityType) edmType;
				List<String> properties = edmEntityType.getPropertyNames();
				List<String> navProperties = edmEntityType.getNavigationPropertyNames();
				resValue = new HashMap<>();
				for (Entry<String, Object> entry : row.entrySet()) {
					String key = entry.getKey();
					Object value = entry.getValue();
					EdmType type = null;
					if (properties.contains(key)) {
						try {
							EdmProperty edmProperty = (EdmProperty) edmEntityType.getProperty(key);
							type = edmProperty.getType();
						} catch (EdmException e) {
							throw new ErrorStatusException(CdsErrorStatuses.MISSING_EDM_PROPERTY, key,
									edmEntityType.getName(), e);
						}
					}
					if (navProperties.contains(key)) {
						try {
							EdmNavigationProperty edmNavProperty = (EdmNavigationProperty) edmEntityType.getProperty(key);
							type = edmNavProperty.getType();
							String navigationKey = edmEntityType.getName().concat(".").concat(key);
							callbacks.put(navigationKey, new ExpandWriteCallback(baseUri, deferredExpand));
						} catch (EdmException e) {
							throw new ErrorStatusException(CdsErrorStatuses.MISSING_EDM_PROPERTY, key,
									edmEntityType.getName(), e);
						}
					}
					if(type != null) {
						resValue.putAll(processProperty(type, key, value, callbacks, baseUri, generateID, deferredExpand));
					}
				}
				break;
			case COMPLEX:
				EdmComplexType edmComplexType = (EdmComplexType) edmType;
				properties = edmComplexType.getPropertyNames();
				resValue = new HashMap<>();
				for (Entry<String, Object> entry : row.entrySet()) {
					String key = entry.getKey();
					Object value = entry.getValue();
					EdmType type = null;
					if (properties.contains(key)) {
						try {
							EdmProperty edmProperty = (EdmProperty)edmComplexType.getProperty(key);
							type = edmProperty.getType();
						} catch (EdmException e) {
							throw new ErrorStatusException(CdsErrorStatuses.MISSING_EDM_PROPERTY, key,
									edmComplexType.getName(), e);
						}
					}
					if(type != null) {
						resValue.putAll(processProperty(type, key, value, callbacks, baseUri, generateID, deferredExpand));
					}
				}
				break;
			case SIMPLE:
				for (Entry<String, Object> entry : row.entrySet()) {
					String key = entry.getKey();
					Object value = entry.getValue();
					resValue = new HashMap<>();
					if (value instanceof List list) {
						List values = new ArrayList<>();
						list.forEach(val -> values.add(getValueBasedOnTypeOfResultSet(edmType, val)));
						resValue.put(key, values);
					} else {
						resValue.put(key, getValueBasedOnTypeOfResultSet(edmType, value));
					}

				}
				break;
			default:
				throw new ErrorStatusException(CdsErrorStatuses.UNSUPPORTED_ACTION_FUNCTION_RETURN_TYPE, edmType.getKind().name());
			}
		} catch (EdmException ex) {
			// Never thrown by Olingo, as methods getPropertyNames() and
			// getNavigationPropertyNames() simply return a List.
			throw new IllegalStateException(ex);
		}

		return resValue;
	}

	public static String resultToString(Result result) {
		return findFirstValue(rowToMap(result.first())).toString();
	}

	public static Map<String, Object> rowToMap(Optional<Row> row) {
		if (row.isPresent()) {
			return row.get();
		}
		return emptyMap();
	}

	@SuppressWarnings("unchecked")
	public static <T> T rowToSingleValue(Optional<Row> row, Class<T> type) {
		if (row.isPresent()) {
			return (T) findFirstValue(row.get());
		}
		return null;
	}

	public static Object getReturnValueOfFunction(EdmFunctionImport functionImport, List<Map<String, Object>> data)
			throws EdmException {
		EdmTyped returnType = functionImport.getReturnType();
		if (returnType.getMultiplicity() == EdmMultiplicity.ONE && data.isEmpty()) {
			throw new ErrorStatusException(CdsErrorStatuses.FUNCTION_RETURN_VALUE_NOT_FOUND, functionImport.getName());
		}
		if (returnType.getType().getKind() == EdmTypeKind.SIMPLE) {
			return findFirstValue(findFirstRow(data));
		}
		return returnType.getMultiplicity() == EdmMultiplicity.MANY ? data : findFirstRow(data);
	}

	private static Object findFirstValue(Map<String, Object> row) {
		return row.values().stream().findFirst().get();
	}

	private static Map<String, Object> findFirstRow(List<Map<String, Object>> list) {
		if (null == list || list.isEmpty()) {
			return Collections.emptyMap();
		}
		return list.get(0);
	}
}
