/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v2.utils;

import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import com.sap.cds.adapter.odata.v2.processors.request.CdsODataRequest;
import com.sap.cds.impl.builder.model.Conjunction;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.CdsDataException;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ETagUtils;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.HttpHeaders;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.util.CdsTypeUtils;

public class ETagHelper {

	public static Optional<CdsElement> getETagElement(String entityName, CdsModel cdsModel) {
		Optional<CdsEntity> entity = cdsModel.findEntity(entityName);
		if(entity.isPresent()) {
			return getETagElement(entity.get());
		}
		return Optional.empty();
	}

	public static Optional<CdsElement> getETagElement(CdsEntity entity) {
		return entity.elements().filter(e -> CdsAnnotations.ETAG.isTrue(e)).findFirst();
	}

	public static boolean isETagHeaderInRequest(CdsODataRequest odataRequest) {
		return odataRequest.getHeader(HttpHeaders.IF_MATCH) != null || odataRequest.getHeader(HttpHeaders.IF_NONE_MATCH) != null;
	}

	public static boolean hasIfNoneMatchHeaderWithAsteriskValue(CdsODataRequest request) {
		List<String> headerValueList = ETagUtils.parseHeader(request.getHeader(HttpHeaders.IF_NONE_MATCH));
		if (headerValueList != null && headerValueList.contains("*")) {
			return true;
		}
		return false;
	}

	public static CqnPredicate getETagPredicate(CdsODataRequest request, CdsElement etagElement) {
		CqnPredicate predicate = null;
		List<String> ifMatchValues = ETagUtils.parseHeader(request.getHeader(HttpHeaders.IF_MATCH));
		if (ifMatchValues != null && !ifMatchValues.contains("*")) {
			List<Object> convertedValues = ifMatchValues.stream().map(v -> convertETag(v, etagElement)).collect(Collectors.toList());
			predicate = CQL.get(etagElement.getName()).in(convertedValues);
		}

		List<String> ifNoneMatchValues = ETagUtils.parseHeader(request.getHeader(HttpHeaders.IF_NONE_MATCH));
		if (ifNoneMatchValues != null) {
			ifNoneMatchValues.remove("*"); // remove * from IF-None-Match header. It's handled separately
			if(!ifNoneMatchValues.isEmpty()) {
				List<Object> convertedValues = ifNoneMatchValues.stream().map(v -> convertETag(v, etagElement)).collect(Collectors.toList());
				CqnPredicate predicateIfNone = CQL.not(CQL.get(etagElement.getName()).in(convertedValues));
				if (predicate != null) {
					predicate = Conjunction.and(predicate, predicateIfNone);
				} else {
					return predicateIfNone;
				}
			}
		}

		return predicate;
	}

	private static Object convertETag(String value, CdsElement element) {
		CdsBaseType type = element.getType().as(CdsSimpleType.class).getType();
		try {
			return CdsTypeUtils.parse(type, value);
		} catch (CdsDataException e) {
			throw new ErrorStatusException(CdsErrorStatuses.ETAG_VALUE_INVALID, type.cdsName(), e);
		}
	}

	public static boolean checkReadPreconditions(final String eTag,
			final String ifMatchHeaders, final String ifNoneMatchHeaders)
					throws ErrorStatusException {
		List<String> ifMatchHeaderList = ifMatchHeaders != null ? Arrays.asList(ifMatchHeaders.split(",")) : null;
		List<String> ifNoneMatchHeaderList = ifNoneMatchHeaders != null ? Arrays.asList(ifNoneMatchHeaders.split(",")) : null;
		if (eTag != null) {
			final ETagInformation ifMatch = createETagInformation(ifMatchHeaderList);
			if (!ifMatch.isMatchedBy(eTag) && !ifMatch.getETags().isEmpty()) {
				throw new ErrorStatusException(CdsErrorStatuses.ETAG_FAILED);
			}
			return createETagInformation(ifNoneMatchHeaderList).isMatchedBy(eTag);
		}
		return false;
	}

	/**
	 * Creates ETag information from the values of a HTTP header
	 * containing a list of entity tags or a single star character, i.e.,
	 * <code>If-Match</code> and <code>If-None-Match</code>.
	 * @param values the collection of header values
	 * @return an {@link ETagInformation} instance
	 */
	private static ETagInformation createETagInformation(final Collection<String> values) {
		final Collection<String> eTags = ETagParser.parse(values);
		final boolean isAll = eTags.size() == 1 && "*".equals(eTags.iterator().next());
		return new ETagInformation(isAll,
				isAll ? Collections.<String> emptySet() : Collections.unmodifiableCollection(eTags));
	}

	/**
	 * @param data the data
	 * @param entity the {@link CdsEntity}
	 * @return The etag value retrieved from {@code data} or {@code null}
	 * if no value could be retrieved or no etag is specified for the entity
	 */
	public static String getEtag(Map<String, Object> data, CdsEntity entity) {
		Optional<CdsElement> etagElement = ETagHelper.getETagElement(entity);
		if (etagElement.isPresent()) {
			Object value = data.get(etagElement.get().getName());
			if (value != null) {
				String strValue;
				if (value instanceof Instant instant) {
					// to be consistent with the olingo serialization
					DateTimeFormatter formatter = new DateTimeFormatterBuilder().appendInstant(7).toFormatter(Locale.US);
					strValue = formatter.format(instant);
				} else {
					strValue = value.toString();
				}
				return ETagUtils.createETagHeaderValue(strValue);
			}
		}
		return null;
	}
}
