/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v2.utils;

import java.util.Collection;
import java.util.Collections;

public class ETagInformation {
	private final boolean all;
	private final Collection<String> eTags;

	public ETagInformation(final boolean all, final Collection<String> eTags) {
		this.all = all;
		this.eTags = Collections.unmodifiableCollection(eTags);
	}

	/**
	 * Gets the information whether the values contain "*".
	 * @return boolean
	 */
	public boolean isAll() {
		return all;
	}

	/**
	 * Gets the collection of ETag values found.
	 * @return collection of etags.
	 */
	public Collection<String> getETags() {
		return eTags;
	}

	/**
	 * <p>Checks whether a given ETag value is matched by this ETag information,
	 * using weak comparison as described in
	 * <a href="https://www.ietf.org/rfc/rfc7232.txt">RFC 7232</a>, section 2.3.2.</p>
	 * <p>If the given value is <code>null</code>, or if this ETag information
	 * does not contain anything, the result is <code>false</code>.</p>
	 * @param eTag the ETag value to match
	 * @return a boolean match result
	 */
	public boolean isMatchedBy(final String eTag) {
		if (eTag == null) {
			return false;
		} else if (all) {
			return true;
		} else {
			for (final String candidate : eTags) {
				if ((eTag.startsWith("W/") ? eTag.substring(2) : eTag)
						.equals(candidate.startsWith("W/") ? candidate.substring(2) : candidate)) {
					return true;
				}
			}
			return false;
		}
	}
}
