/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v2.utils;

import java.math.BigDecimal;
import java.nio.charset.Charset;
import java.sql.Time;
import java.sql.Timestamp;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoField;
import java.time.temporal.ChronoUnit;
import java.util.Calendar;
import java.util.Calendar.Builder;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Objects;
import java.util.UUID;

import org.apache.olingo.odata2.api.edm.EdmAnnotatable;
import org.apache.olingo.odata2.api.edm.EdmAnnotationAttribute;
import org.apache.olingo.odata2.api.edm.EdmException;
import org.apache.olingo.odata2.api.edm.EdmType;
import org.apache.olingo.odata2.core.edm.Bit;
import org.apache.olingo.odata2.core.edm.EdmBinary;
import org.apache.olingo.odata2.core.edm.EdmBoolean;
import org.apache.olingo.odata2.core.edm.EdmByte;
import org.apache.olingo.odata2.core.edm.EdmDateTime;
import org.apache.olingo.odata2.core.edm.EdmDateTimeOffset;
import org.apache.olingo.odata2.core.edm.EdmDecimal;
import org.apache.olingo.odata2.core.edm.EdmDouble;
import org.apache.olingo.odata2.core.edm.EdmGuid;
import org.apache.olingo.odata2.core.edm.EdmInt16;
import org.apache.olingo.odata2.core.edm.EdmInt32;
import org.apache.olingo.odata2.core.edm.EdmInt64;
import org.apache.olingo.odata2.core.edm.EdmSByte;
import org.apache.olingo.odata2.core.edm.EdmSingle;
import org.apache.olingo.odata2.core.edm.EdmString;
import org.apache.olingo.odata2.core.edm.EdmTime;
import org.apache.olingo.odata2.core.edm.Uint7;

import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.util.CdsTypeUtils;

public class TypeConverterUtils {

	private TypeConverterUtils() {

	}

	// For Edm.DateTime seconds are optional and fraction of a second also optional (up to six digits)
	// Spec datetime'yyyy-mm-ddThh:mm[:ss[.fffffff]]
	private static final DateTimeFormatter DATE_FORMATTER = new DateTimeFormatterBuilder()
		.append(DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm[:ss]"))
		.optionalStart().appendFraction(ChronoField.NANO_OF_SECOND, 0, 6, true).optionalEnd().toFormatter();

	/**
	 * Convert string value of EdmType to a corresponding java type. Used for conversion of
	 * filter values.
	 *
	 * @param type  Edm Type
	 * @param value string representation of a value
	 * @return Object
	 */
	public static Object convertToType(EdmType type, String value) {
		if (value == null) {
			return null;
		}

		Object typedValue;
		if (type instanceof EdmDecimal) {
			typedValue = new BigDecimal(value);
		} else if (type instanceof EdmDouble) {
			typedValue = Double.parseDouble(value);
		} else if (type instanceof EdmSingle) {
			typedValue = Float.parseFloat(value);
		} else if (type instanceof EdmInt64) {
			typedValue = Long.parseLong(value);
		} else if (type instanceof EdmInt32 || type instanceof EdmInt16 || type instanceof EdmSByte
				|| type instanceof EdmByte || type instanceof Uint7 || type instanceof Bit) {
			typedValue = Integer.parseInt(value);
		} else if (type instanceof EdmBinary) {
			typedValue = value.getBytes(Charset.defaultCharset());
		} else if (type instanceof EdmString) {
			if (value.length() >= 2 && value.startsWith("'") && value.endsWith("'")) {
				typedValue = value.substring(1, value.length() - 1);
			} else {
				typedValue = value;
			}
		} else if (type instanceof EdmTime) {
			if (value.startsWith("PT")) {
				typedValue = LocalTime.ofNanoOfDay(Duration.parse(value).toNanos());
			} else {
				typedValue = LocalTime.parse(value);
			}
		} else if (type instanceof EdmDateTime) {
			typedValue = LocalDate.from(DATE_FORMATTER.parse(value));
		} else if (type instanceof EdmDateTimeOffset) {
			typedValue = ZonedDateTime.parse(value).toInstant();
		} else if (type instanceof EdmBoolean) {
			typedValue = Boolean.parseBoolean(value);
		} else if (type instanceof EdmGuid) {
			typedValue = CdsTypeUtils.parseUuid(value);
		} else {
			typedValue = value;
		}

		return typedValue;
	}

	/**
	 * Convert value of EdmType to a corresponding java type supported by Olingo
	 * library. This is used for type conversion of a response.
	 *
	 * @param type  Edm Type
	 * @param value string representation of a value
	 * @return Object
	 */
	public static Object getValueBasedOnTypeOfResultSet(EdmType type, Object value) {
		if (value == null) {
			return null;
		}

		try {
			if (type instanceof EdmTime) {
				return convertToEdmTimeAsCalendar(value);
			} else if (type instanceof EdmDateTime) {
				return convertToEdmDateTimeAsSqlTimestamp(value);
			} else if (type instanceof EdmDateTimeOffset) {
				return convertToEdmDateTimeOffsetAsSqlTimestamp(value);
			} else if (type instanceof EdmGuid) {
				return UUID.fromString(value.toString());
			}
		} catch (IllegalArgumentException | DateTimeParseException | ErrorStatusException e) {
			ErrorStatusException conversionError = conversionError(value, getTypeName(type));
			conversionError.initCause(e);

			throw conversionError;
		}

		return value;
	}

	private static String getTypeName(EdmType type) {
		return type.getClass().getSimpleName().substring(3);
	}

	private static long convertToEdmDateTimeAsSqlTimestamp(Object value) {
		if (value instanceof LocalDate date) {
			LocalDateTime ldt = date.atStartOfDay();
			return utcMillisOfLocalDateTimeUTC(ldt);
		} else if (value instanceof String dateTimeString) {
			try {
				LocalDateTime ldt = LocalDateTime.parse(dateTimeString);
				return utcMillisOfLocalDateTimeUTC(ldt);
			} catch (DateTimeParseException ex) {
				// NOSONAR
			}

			try {
				LocalDate ld = LocalDate.parse(dateTimeString);
				return utcMillisOfLocalDateTimeUTC(ld.atStartOfDay());
			} catch (DateTimeParseException ex) {
				// NOSONAR
			}

			ZonedDateTime zdt = ZonedDateTime.parse(dateTimeString);
			return zdt.toInstant().truncatedTo(ChronoUnit.MILLIS).toEpochMilli();
		} else if (value instanceof java.sql.Date sqlDate) {
			LocalDateTime ldt = sqlDate.toLocalDate().atStartOfDay();
			return utcMillisOfLocalDateTimeUTC(ldt);
		}

		Timestamp ts = convertToEdmDateTimeOffsetAsSqlTimestamp(value);
		return ts.toInstant().truncatedTo(ChronoUnit.MILLIS).toEpochMilli();
	}

	private static Timestamp timestampOfLocalDateTimeUTC(LocalDateTime ldt) {
		return Timestamp.from(ldt.toInstant(ZoneOffset.UTC));
	}

	private static long utcMillisOfLocalDateTimeUTC(LocalDateTime ldt) {
		return ldt.toInstant(ZoneOffset.UTC).toEpochMilli();
	}

	private static Calendar convertToEdmTimeAsCalendar(Object value) {
		if (value instanceof LocalTime time) {
			return timeToCalendar(time);
		} else if (value instanceof String string) {
			return timeToCalendar(LocalTime.parse(string));
		} else if (value instanceof java.sql.Time time) {
			return timeToCalendar(time.toLocalTime());
		} else if (value instanceof Calendar calendar) {
			return calendar;
		}
		throw conversionError(value, "Time");
	}

	private static ErrorStatusException conversionError(Object value, String type) {
		return new ErrorStatusException(CdsErrorStatuses.VALUE_CONVERSION_FAILED,
				value == null ? null : value.toString(), value == null ? null : value.getClass().getCanonicalName(),
				type);
	}

	private static Timestamp convertToEdmDateTimeOffsetAsSqlTimestamp(Object value) {
		if (value instanceof Instant instant) {
			return Timestamp.from(instant);
		} else if (value instanceof ZonedDateTime time) {
			return Timestamp.from(time.toInstant());
		} else if (value instanceof LocalDateTime ldt) {
			return timestampOfLocalDateTimeUTC(ldt);
		} else if (value instanceof Timestamp timestamp) {
			return timestamp;
		} else if (value instanceof Long long1) {
			return new Timestamp(long1);
		} else if (value instanceof String string) {
			ZonedDateTime zdt = ZonedDateTime.parse(string);
			return Timestamp.from(zdt.toInstant());
		}

		throw conversionError(value, "DateTimeOffset");
	}

	private static Calendar timeToCalendar(LocalTime time) {
		return new Builder().set(Calendar.HOUR_OF_DAY, time.getHour()).set(Calendar.MINUTE, time.getMinute())
				.set(Calendar.SECOND, time.getSecond()).set(Calendar.MILLISECOND, time.getNano() / 1000000).build();
	}

	/**
	 * Convert value of EdmType to a corresponding java type supported by CDS4J
	 * library. This is used for type conversion of a request payload.
	 *
	 * @param type  Edm Type
	 * @param value string representation of a value
	 * @return Object
	 */
	public static Object getValueBasedOnTypeOfRequestPayload(EdmType type, EdmAnnotatable property, Object value) {
		if (value == null) {
			return null;
		}

		Object typedValue = null;

		if (type instanceof EdmTime) {
			if (value instanceof GregorianCalendar calendar) {
				typedValue = calendar.toZonedDateTime().toLocalTime();
			} else if (value instanceof Time time) {
				typedValue = time.toLocalTime();
			}
		} else if (type instanceof EdmDateTime) {
			boolean isDate = hasDateAnnotation(property);
			if (value instanceof GregorianCalendar calendar) {
				// cds.Date is mapped to Edm.DateTime in V2
				if (isDate) {
					typedValue = calendar.toZonedDateTime().toLocalDate();
				} else {
					typedValue = calendar.toZonedDateTime().toInstant();
				}
			} else if (value instanceof Timestamp timestamp) {
				if (isDate) {
					typedValue = timestamp.toLocalDateTime().toLocalDate();
				} else {
					typedValue = timestamp.toInstant();
				}
			}
		} else if (type instanceof EdmDateTimeOffset) {
			if (value instanceof GregorianCalendar calendar) {
				typedValue = calendar.toInstant();
			} else if (value instanceof Timestamp timestamp) {
				typedValue = timestamp.toInstant();
			}
		} else if (type instanceof EdmGuid) {
			typedValue = CdsTypeUtils.parseUuid(value);
		} else {
			typedValue = value;
		}

		return typedValue;
	}

	private static boolean hasDateAnnotation(EdmAnnotatable property) {
		try {
			if (property == null) {
				return true; // assume Edm.DateTime with Date annotation by default, as CAP always generates it as Date by default
			}

			List<EdmAnnotationAttribute> attributes = property.getAnnotations().getAnnotationAttributes();
			if(attributes == null) {
				return false;
			}

			EdmAnnotationAttribute attribute = attributes.stream()
				.filter(a -> Objects.equals(a.getPrefix(), "sap") && Objects.equals(a.getName(), "display-format")).findFirst().orElse(null);
			return attribute != null && Objects.equals(attribute.getText(), "Date");
		} catch (EdmException e) {
			return true; // assume Edm.DateTime with Date annotation by default, as CAP always generates it as Date by default
		}
	}

}
