/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.metadata.provider;

import java.io.InputStream;
import java.util.Collections;
import java.util.Locale;
import java.util.Map;

import org.apache.olingo.commons.api.edm.provider.CsdlEdmProvider;

import com.google.common.hash.Hashing;
import com.sap.cds.adapter.edmx.EdmxI18nProvider;
import com.sap.cds.adapter.edmx.EdmxV4Provider;

public class LocalizingEdmxProviderWrapper implements OlingoAwareEdmxV4Provider {

	private static final EdmxI18nProvider NOOP = new EdmxI18nProvider() {
		@Override
		public String getETag(Locale locale) {
			return null;
		}

		@Override
		public Map<String, String> getTexts(Locale locale) {
			return Collections.emptyMap();
		}
	};

	private final EdmxV4Provider provider;
	private final EdmxI18nProvider i18nProvider;
	private final Locale locale;

	public LocalizingEdmxProviderWrapper(EdmxV4Provider provider, EdmxI18nProvider i18nProvider, Locale locale) {
		this.provider = provider;
		this.i18nProvider = i18nProvider != null ? i18nProvider : NOOP;
		this.locale = locale;
	}

	@Override
	public String getETag(String serviceName) {
		String eTag = provider.getETag(serviceName);
		String i18nETag = i18nProvider.getETag(locale);
		if (i18nETag == null) {
			return eTag;
		}
		return Hashing.sha256().hashUnencodedChars(eTag + i18nETag).toString();
	}

	@Override
	public InputStream getEdmx(String serviceName) {
		InputStream original = provider.getEdmx(serviceName);
		Map<String, String> i18n = i18nProvider.getTexts(locale);
		if (!i18n.isEmpty()) {
			return new EdmxI18nInputStream(original, i18n);
		}
		return original;
	}

	@Override
	public CsdlEdmProvider getEdmProvider(String serviceName) {
		return provider instanceof OlingoAwareEdmxV4Provider withEdm ? withEdm.getEdmProvider(serviceName) : null;
	}

}
