/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.processors;

import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;
import java.util.Optional;
import java.util.function.BiConsumer;

import org.apache.http.HttpStatus;
import org.apache.olingo.commons.api.data.ContextURL;
import org.apache.olingo.commons.api.data.ContextURL.Builder;
import org.apache.olingo.commons.api.data.ContextURL.Suffix;
import org.apache.olingo.commons.api.data.Entity;
import org.apache.olingo.commons.api.data.EntityCollection;
import org.apache.olingo.commons.api.edm.EdmBindingTarget;
import org.apache.olingo.commons.api.edm.EdmEntityType;
import org.apache.olingo.commons.api.edm.EdmProperty;
import org.apache.olingo.commons.api.edm.EdmSingleton;
import org.apache.olingo.commons.api.edm.EdmStructuredType;
import org.apache.olingo.commons.api.edm.EdmType;
import org.apache.olingo.commons.api.edm.constants.EdmTypeKind;
import org.apache.olingo.commons.api.format.ContentType;
import org.apache.olingo.commons.api.http.HttpHeader;
import org.apache.olingo.commons.api.http.HttpMethod;
import org.apache.olingo.commons.core.edm.primitivetype.EdmStream;
import org.apache.olingo.server.api.ODataContent;
import org.apache.olingo.server.api.ODataRequest;
import org.apache.olingo.server.api.ODataResponse;
import org.apache.olingo.server.api.prefer.Preferences.Return;
import org.apache.olingo.server.api.prefer.PreferencesApplied;
import org.apache.olingo.server.api.serializer.EdmAssistedSerializer;
import org.apache.olingo.server.api.serializer.ODataSerializer;
import org.apache.olingo.server.api.serializer.SerializerException;
import org.apache.olingo.server.api.uri.UriHelper;
import org.apache.olingo.server.api.uri.UriInfo;
import org.apache.olingo.server.api.uri.UriResource;
import org.apache.olingo.server.api.uri.UriResourceComplexProperty;
import org.apache.olingo.server.api.uri.UriResourceEntitySet;
import org.apache.olingo.server.api.uri.UriResourceKind;
import org.apache.olingo.server.api.uri.UriResourceNavigation;
import org.apache.olingo.server.api.uri.UriResourcePartTyped;
import org.apache.olingo.server.api.uri.UriResourcePrimitiveProperty;
import org.apache.olingo.server.api.uri.UriResourceProperty;
import org.apache.olingo.server.api.uri.queryoption.ExpandOption;
import org.apache.olingo.server.api.uri.queryoption.SelectOption;
import org.apache.olingo.server.core.serializer.utils.CircleStreamBuffer;

import com.sap.cds.Result;
import com.sap.cds.adapter.odata.v4.CdsRequestGlobals;
import com.sap.cds.adapter.odata.v4.processors.request.CdsODataRequest;
import com.sap.cds.adapter.odata.v4.processors.response.CdsODataResponse;
import com.sap.cds.adapter.odata.v4.processors.response.ResultSetProcessor;
import com.sap.cds.adapter.odata.v4.utils.EdmUtils;
import com.sap.cds.adapter.odata.v4.utils.MessagesUtils;
import com.sap.cds.adapter.odata.v4.utils.ODataUtils;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.ErrorStatuses;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.StringUtils;

public abstract class AbstractODataProcessor {

	protected final CdsRequestGlobals globals;
	protected final CdsProcessor cdsProcessor;
	protected final EdmUtils edmUtils;
	protected final boolean isBuffered;

	protected AbstractODataProcessor(CdsRequestGlobals globals) {
		this.globals = globals;
		this.cdsProcessor = new CdsProcessor(globals);
		this.edmUtils = new EdmUtils(globals);
		this.isBuffered = globals.getRuntime().getEnvironment().getCdsProperties().getOdataV4().getSerializer().isBuffered();
	}

	public abstract void processEntity(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo,
			ContentType requestFormat, ContentType contentType);

	public abstract void processEntities(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo,
			ContentType requestFormat, ContentType contentType);

	public abstract void processSingleComplex(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo,
			ContentType requestFormat, ContentType responseFormat);

	public abstract void processCollectionComplex(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat, ContentType responseFormat);

	public abstract void processSinglePrimitive(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo,
			ContentType requestFormat, ContentType responseFormat);

	public abstract void processSinglePrimitiveValue(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat, ContentType responseFormat);

	public abstract void processCollectionPrimitive(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo,
			ContentType requestFormat, ContentType responseFormat);

	public void processNoContentRequest(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat) {
		processRequest(odataRequest, odataResponse, uriInfo, requestFormat, (cdsRequest, cdsResponse) -> {
			if (isValueRequestOnStreamProperty(cdsRequest)) {
				UriResourcePartTyped lastTypedResource = cdsRequest.getLastTypedResource();
				throw new ErrorStatusException(CdsErrorStatuses.VALUE_ACCESS_NOT_ALLOWED, lastTypedResource.getSegmentValue());
			}
			setODataResponse(cdsRequest, cdsResponse, odataResponse, (InputStream)null, null);
		});
	}

	public void processCountRequest(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo) {
		processRequest(odataRequest, odataResponse, uriInfo, null, (cdsRequest, cdsResponse) -> {
			int count = ResultSetProcessor.toCount(cdsResponse.getResult());
			InputStream content;
			try {
				content = globals.getOData().createFixedFormatSerializer().count(count);
			} catch (SerializerException e) {
				throw new ErrorStatusException(CdsErrorStatuses.RESPONSE_SERIALIZATION_FAILED, e);
			}
			setODataResponse(cdsRequest, cdsResponse, odataResponse, content, ContentType.TEXT_PLAIN);
		});
	}

	/* UTILS */

	static void setLocationHeader(ODataResponse odataResponse, String locationPrefix, String id) {
		if (locationPrefix != null) {
			String location = locationPrefix + (id == null ? "" : id);
			// set respective headers
			odataResponse.setHeader(HttpHeader.LOCATION, location);
			odataResponse.setHeader(HttpHeader.ODATA_ENTITY_ID, location);
		}
	}

	static void setCacheControlHeader(ODataResponse odataResponse, CdsEntity entity, String property) {
		Object maxAge = ODataUtils.getMaxAgeValue(entity, property);
		if (maxAge != null) {
			odataResponse.setHeader(HttpHeader.CACHE_CONTROL, "max-age=" + maxAge);
		}
	}

	static void setInlineCount(UriInfo uriInfo, Result result, EntityCollection entityCollection) {
		if (uriInfo.getCountOption() != null && uriInfo.getCountOption().getValue()) {
			int inlineCount = (int)result.inlineCount();
			if (inlineCount == -1) {
				throw new ErrorStatusException(CdsErrorStatuses.MISSING_VALUE_FOR_COUNT);
			}
			entityCollection.setCount(inlineCount);
		}
	}

	protected boolean isValueRequestOnStreamProperty(CdsODataRequest request) {
		UriResource lastResource = request.getLastResource();
		UriResourcePartTyped lastTypedResource = request.getLastTypedResource();
		return (lastResource.getKind().equals(UriResourceKind.value)
				&& lastTypedResource.getType() instanceof EdmStream);
	}

	protected void processRequest(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo,
			ContentType requestFormat, BiConsumer<CdsODataRequest, CdsODataResponse> processor) {

		CdsODataRequest cdsRequest = new CdsODataRequest(odataRequest, uriInfo, requestFormat, globals);

		cdsProcessor.processRequest(cdsRequest, cdsResponse -> {
			if (cdsResponse.isSuccess()) {
				processor.accept(cdsRequest, cdsResponse);
			} else {
				ODataUtils.setODataErrorResponse(globals.getOData(), odataRequest, odataResponse, cdsResponse,
						ODataUtils.getBindingParameter(globals.getModel(), uriInfo),
						ContentType.APPLICATION_JSON);
			}
		});
	}

	protected void setODataResponse(CdsODataRequest cdsRequest, CdsODataResponse cdsResponse, ODataResponse odataResponse,
			InputStream content, ContentType contentType) {
		setODataResponse(cdsRequest, cdsResponse, odataResponse, content, null, contentType);
	}

	protected void setODataResponse(CdsODataRequest cdsRequest, CdsODataResponse cdsResponse,
			ODataResponse odataResponse, ODataContent content, ContentType contentType) {
		InputStream bufferContent = isBuffered ? toInputStream(content) : null;
		setODataResponse(cdsRequest, cdsResponse, odataResponse, bufferContent, content, contentType);
	}

	protected void setODataResponse(CdsODataRequest cdsRequest, CdsODataResponse cdsResponse, ODataResponse odataResponse,
			InputStream bufferContent, ODataContent odataContent, ContentType contentType) {

		if( (bufferContent != null || odataContent != null) && contentType != null) {
			odataResponse.setStatusCode(cdsResponse.getStatusCode());
			odataResponse.setHeader(HttpHeader.CONTENT_TYPE, contentType.toContentTypeString());

			if (bufferContent != null) {
				odataResponse.setContent(bufferContent);
			} else {
				odataResponse.setODataContent(odataContent);
			}
		} else {
			odataResponse.setStatusCode(cdsResponse.getStatusCode() < HttpStatus.SC_NO_CONTENT ? HttpStatus.SC_NO_CONTENT : cdsResponse.getStatusCode());
		}

		// Preference Applied
		Return returnPreference = cdsRequest.getReturnPreference();
		if(returnPreference != null) {
			String applied = PreferencesApplied.with().returnRepresentation(returnPreference).build().toValueString();
			odataResponse.setHeader(HttpHeader.PREFERENCE_APPLIED, applied);
		}

		// OData Version
		odataResponse.setHeader(HttpHeader.ODATA_VERSION, ODataUtils.getODataVersion(cdsRequest.getODataRequest()));

		// SAP Messages
		String sapMessageHeader = MessagesUtils.getSapMessagesHeader(
				ODataUtils.getBindingParameter(globals.getModel(), cdsRequest.getUriInfo()),
				RequestContext.getCurrent(globals.getRuntime()).getMessages());

		if(!StringUtils.isEmpty(sapMessageHeader)) {
			odataResponse.setHeader("sap-messages", sapMessageHeader);
		}
	}

	private InputStream toInputStream(ODataContent odataContent) {
		if (null != odataContent) {
			CircleStreamBuffer buffer = new CircleStreamBuffer(1024);
			odataContent.write(buffer.getOutputStream());
			return buffer.getInputStream();
		}
		return null;
	}

	protected ODataSerializer createSerializer(ODataRequest odataRequest, ContentType responseFormat) {
		return ODataUtils.createSerializer(globals.getOData(), odataRequest, responseFormat);
	}

	protected EdmAssistedSerializer createSerializerForApply(ODataRequest odataRequest, ContentType responseFormat) {
		return ODataUtils.createSerializerForApply(globals.getOData(), odataRequest, responseFormat);
	}

	protected boolean isActionOrFunction(CdsODataRequest request) {
		UriResource lastResource = request.getLastResource();
		return lastResource.getKind() == UriResourceKind.action || lastResource.getKind() == UriResourceKind.function;
	}

	protected ContextURL getContextUrl(CdsODataRequest request, boolean isCollection, SelectOption select, ExpandOption expand) {
		try {
			Builder builder = ContextURL.with();
			UriHelper uriHelper = globals.getOData().createUriHelper();

			if(globals.getRuntime().getEnvironment().getCdsProperties().getOdataV4().isContextAbsoluteUrl()) {
				try {
					builder.serviceRoot(new URI(request.getODataRequest().getRawBaseUri() + "/"));
				} catch (URISyntaxException e) {
					throw new ErrorStatusException(ErrorStatuses.SERVER_ERROR, e);
				}
			}

			boolean isSingleton = false;
			EdmType responseType = request.getResponseType();
			if (isActionOrFunction(request)) {
				if(isCollection) {
					builder.asCollection();
				}
				builder.type(responseType);
			} else {
				UriResourcePartTyped entityResource = request.getLastEntityResource(false);
				EdmEntityType entityType = (EdmEntityType) entityResource.getType();
				EdmBindingTarget bindingTarget = edmUtils.getEdmBindingTarget(entityType);

				List<UriResource> uriResources = request.getUriInfo().getUriResourceParts();
				int entitySetIndex = uriResources.indexOf(entityResource);

				// navigation properties available
				if(entitySetIndex + 1 < uriResources.size()) {
					// root entity set
					String keys = "";
					if(entityResource.getKind() == UriResourceKind.entitySet) {
						keys = "(" + uriHelper.buildContextURLKeyPredicate(((UriResourceEntitySet) entityResource).getKeyPredicates()) + ")";
					} else if (entityResource.getKind() == UriResourceKind.navigationProperty) {
						keys = "(" + uriHelper.buildContextURLKeyPredicate(((UriResourceNavigation) entityResource).getKeyPredicates()) + ")";
					}
					builder.entitySetOrSingletonOrType(bindingTarget.getName() + keys);

					StringBuilder navigationPathBuilder = new StringBuilder();
					UriResourceNavigation previousUriResourceNavigation = null;

					// navigation resources and final property
					for(UriResource uriResource : uriResources.subList(entitySetIndex + 1, uriResources.size())) {
						if(previousUriResourceNavigation != null && !previousUriResourceNavigation.isCollection()) {
							String navKeys = uriHelper.buildContextURLKeyPredicate(previousUriResourceNavigation.getKeyPredicates());
							if(navKeys != null) {
								navigationPathBuilder.append("(").append(navKeys).append(")");
							}
						}
						if(navigationPathBuilder.length() > 0) {
							navigationPathBuilder.append("/");
						}
						if(uriResource.getKind() == UriResourceKind.navigationProperty) {
							UriResourceNavigation uriResourceNavigation = (UriResourceNavigation) uriResource;
							navigationPathBuilder.append(uriResourceNavigation.getProperty().getName());
							previousUriResourceNavigation = uriResourceNavigation;
						} else if (uriResource.getKind() == UriResourceKind.complexProperty) {
							navigationPathBuilder.append(((UriResourceComplexProperty) uriResource).getProperty().getName());
						} else if (uriResource.getKind() == UriResourceKind.primitiveProperty) {
							navigationPathBuilder.append(((UriResourcePrimitiveProperty) uriResource).getProperty().getName());
						} else {
							throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_URI_RESOURCE, uriResource.getKind());
						}
					}
					builder.navOrPropertyPath(navigationPathBuilder.toString());
					// no navigation properties available
				} else {
					// type entity
					if(responseType.getKind() == EdmTypeKind.ENTITY) {
						isSingleton = bindingTarget instanceof EdmSingleton;
						builder.entitySetOrSingletonOrType(bindingTarget.getName());
						// primitive or complex values
					} else {
						if(isCollection) {
							builder.asCollection();
						}
						builder.type(responseType);
					}
				}

				if(responseType.getKind() == EdmTypeKind.ENTITY && !isCollection && !isSingleton) {
					builder.suffix(Suffix.ENTITY);
				}
			}

			if((expand != null || select != null) && responseType instanceof EdmStructuredType type) {
				String selectList = uriHelper.buildContextURLSelectList(type, expand, select);
				builder.selectList(selectList);
			}

			return builder.build();
		} catch(SerializerException e) {
			throw new ErrorStatusException(CdsErrorStatuses.RESPONSE_SERIALIZATION_FAILED, e);
		}
	}

	protected void checkCountOptionsOnExpand(Entity entity, ExpandOption expandOption) {
		if (expandOption != null) {
			expandOption.getExpandItems().forEach(expandItem -> {
				if (expandItem.getCountOption() != null && expandItem.getCountOption().getValue()) {
					String expandedItemName = !(expandItem.getResourcePath().getUriResourceParts().isEmpty()) ? expandItem.getResourcePath().getUriResourceParts().get(0).getSegmentValue() : "";
					EntityCollection inlineEntityCollection = entity.getNavigationLink(expandedItemName) != null ? entity.getNavigationLink(expandedItemName).getInlineEntitySet() : null;
					if (inlineEntityCollection != null) {
						inlineEntityCollection.setCount(inlineEntityCollection.getEntities().size());
						if (expandItem.getExpandOption() != null) {
							inlineEntityCollection.getEntities().forEach(inlineEntity ->
							checkCountOptionsOnExpand(inlineEntity, expandItem.getExpandOption()));
						}
					}
				}
			});
		}
	}

	protected Optional<EdmProperty> getEdmProperty(CdsODataRequest cdsRequest) {
		return Optional.ofNullable(cdsRequest.getLastTypedResource())
				.filter(UriResourcePrimitiveProperty.class::isInstance).map(UriResourcePrimitiveProperty.class::cast)
				.map(UriResourceProperty::getProperty);
	}

	protected boolean isGetStreamContext(ODataRequest odataRequest, UriInfo uriInfo, ContentType requestFormat) {
		HttpMethod method = odataRequest.getMethod();
		CdsODataRequest cdsODataRequest = new CdsODataRequest(odataRequest, uriInfo, requestFormat, this.globals);

		return (method.equals(HttpMethod.GET)
				&& (cdsODataRequest.getLastTypedResource().getType() instanceof EdmStream));
	}

	protected void setContentDispositionHeaderIfNotNull(ODataResponse response, String filename) {
		if (!StringUtils.isEmpty(filename)) {
			response.setHeader("Content-Disposition", "attachment; filename=\"%s\"".formatted(filename));
		}
	}
}
