/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.processors;

import java.io.InputStream;
import java.io.Reader;
import java.net.URI;
import java.nio.charset.Charset;

import org.apache.commons.io.input.ReaderInputStream;
import org.apache.http.HttpStatus;
import org.apache.olingo.commons.api.data.ContextURL;
import org.apache.olingo.commons.api.data.Entity;
import org.apache.olingo.commons.api.data.EntityCollection;
import org.apache.olingo.commons.api.data.Property;
import org.apache.olingo.commons.api.edm.EdmComplexType;
import org.apache.olingo.commons.api.edm.EdmEntityType;
import org.apache.olingo.commons.api.edm.EdmPrimitiveType;
import org.apache.olingo.commons.api.format.ContentType;
import org.apache.olingo.commons.api.http.HttpHeader;
import org.apache.olingo.commons.api.http.HttpMethod;
import org.apache.olingo.commons.core.edm.primitivetype.EdmStream;
import org.apache.olingo.server.api.ODataRequest;
import org.apache.olingo.server.api.ODataResponse;
import org.apache.olingo.server.api.prefer.Preferences.Return;
import org.apache.olingo.server.api.serializer.ComplexSerializerOptions;
import org.apache.olingo.server.api.serializer.EdmAssistedSerializerOptions;
import org.apache.olingo.server.api.serializer.EntityCollectionSerializerOptions;
import org.apache.olingo.server.api.serializer.EntitySerializerOptions;
import org.apache.olingo.server.api.serializer.PrimitiveSerializerOptions;
import org.apache.olingo.server.api.serializer.PrimitiveValueSerializerOptions;
import org.apache.olingo.server.api.serializer.SerializerException;
import org.apache.olingo.server.api.serializer.SerializerResult;
import org.apache.olingo.server.api.uri.UriInfo;
import org.apache.olingo.server.api.uri.UriResource;
import org.apache.olingo.server.api.uri.UriResourceKind;
import org.apache.olingo.server.api.uri.UriResourceNavigation;
import org.apache.olingo.server.api.uri.UriResourcePartTyped;
import org.apache.olingo.server.api.uri.queryoption.ExpandOption;
import org.apache.olingo.server.core.deserializer.helper.ExpandTreeBuilder;
import org.apache.olingo.server.core.deserializer.helper.ExpandTreeBuilderImpl;

import com.sap.cds.adapter.odata.v4.CdsRequestGlobals;
import com.sap.cds.adapter.odata.v4.processors.request.CdsODataRequest;
import com.sap.cds.adapter.odata.v4.processors.response.CdsODataResponse;
import com.sap.cds.adapter.odata.v4.processors.response.ResultSetProcessor;
import com.sap.cds.adapter.odata.v4.query.NextLinkInfo;
import com.sap.cds.adapter.odata.v4.utils.ChangeSetContextAwareInputStream;
import com.sap.cds.adapter.odata.v4.utils.ODataUtils;
import com.sap.cds.services.changeset.ChangeSetContextSPI;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.StringUtils;

/**
 * First turns an {@link ODataRequest} into a {@link CdsODataRequest} and passes it to the {@link ODataProcessor}.
 * The {@link CdsODataResponse} returned from the {@link ODataProcessor} is transformed into an {@link ODataResponse} by this class.
 */
public class ODataProcessor extends AbstractODataProcessor {

	private final ResultSetProcessor resultSetProcessor;

	public ODataProcessor(CdsRequestGlobals globals) {
		super(globals);
		this.resultSetProcessor = new ResultSetProcessor(globals);
	}

	@Override
	public void processEntity(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat, ContentType responseFormat) {
		processRequest(odataRequest, odataResponse, uriInfo, requestFormat, (cdsRequest, cdsResponse) -> {
			ExpandTreeBuilder expandBuilder = ExpandTreeBuilderImpl.create();
			Entity entity = resultSetProcessor.toEntity(cdsRequest, cdsResponse.getResult(), expandBuilder);
			ExpandOption expand = uriInfo.getExpandOption() != null ? uriInfo.getExpandOption() : expandBuilder.build();

			// not a GET request or an action or function request
			if(entity != null && odataRequest.getMethod() != HttpMethod.GET && cdsRequest.getLastResource() == cdsRequest.getLastEntityResource(true)) {
				// calculate entity location URL
				String location = odataRequest.getRawBaseUri() + odataRequest.getRawODataPath();
				UriResource lastResource = cdsRequest.getLastResource();
				if(!location.endsWith(")") &&
						(lastResource.getKind() == UriResourceKind.entitySet ||
						(lastResource.getKind() == UriResourceKind.navigationProperty && ((UriResourceNavigation) lastResource).getProperty().isCollection()))) {
					try {
						String keyPredicate = globals.getOData().createUriHelper().buildKeyPredicate((EdmEntityType) cdsRequest.getResponseType(), entity);
						if(!StringUtils.isEmpty(keyPredicate)) {
							location += '(' + keyPredicate + ')';
						}
					} catch (SerializerException e) {
						throw new ErrorStatusException(CdsErrorStatuses.RESPONSE_SERIALIZATION_FAILED, e);
					}
				}

				// set respective headers
				odataResponse.setHeader(HttpHeader.LOCATION, location);
				odataResponse.setHeader(HttpHeader.ODATA_ENTITY_ID, location);
			}

			SerializerResult serializerResult = null;
			if (entity != null) {
				String etag = entity.getETag();
				if(etag != null) {
					odataResponse.setHeader(HttpHeader.ETAG, etag);
				}
				if(cdsResponse.getStatusCode() < HttpStatus.SC_NO_CONTENT && cdsRequest.getReturnPreference() != Return.MINIMAL) {
					checkCountOptionsOnExpand(entity, uriInfo.getExpandOption());


					ContextURL contextUrl = getContextUrl(cdsRequest, false, uriInfo.getSelectOption(), expand);
					EntitySerializerOptions options = EntitySerializerOptions.with()
							.contextURL(contextUrl)
							.select(uriInfo.getSelectOption())
							.expand(expand)
							.build();

					try {
						serializerResult = createSerializer(odataRequest, responseFormat).entity(globals.getServiceMetadata(), (EdmEntityType) cdsRequest.getResponseType(), entity, options);
					} catch (SerializerException e) {
						throw new ErrorStatusException(CdsErrorStatuses.RESPONSE_SERIALIZATION_FAILED, e);
					}
				}
			}

			InputStream content = serializerResult != null ? serializerResult.getContent() : null;
			setODataResponse(cdsRequest, cdsResponse, odataResponse, content, responseFormat);
		});
	}

	@Override
	public void processEntities(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat, ContentType responseFormat) {
		processRequest(odataRequest, odataResponse, uriInfo, requestFormat, (cdsRequest, cdsResponse) -> {
			ExpandTreeBuilder expandBuilder = ExpandTreeBuilderImpl.create();
			EntityCollection entityCollection = resultSetProcessor.toEntityCollection(cdsRequest, cdsResponse.getResult(), expandBuilder);
			ExpandOption expand = uriInfo.getExpandOption() != null ? uriInfo.getExpandOption() : expandBuilder.build();

			entityCollection.forEach(entity -> checkCountOptionsOnExpand(entity, uriInfo.getExpandOption()));
			// server-driven paging applied and (probably) not on last page
			NextLinkInfo nextLinkInfo = cdsResponse.getNextLinkInfo();
			if(nextLinkInfo != null) {
				URI nextLink = nextLinkInfo.getNextLink(odataRequest);
				entityCollection.setNext(nextLink);
			}

			// Set Count for outer entity if ?$count = true
			setInlineCount(uriInfo, cdsResponse.getResult(), entityCollection);

			SerializerResult serializerResult;
			try {
				ContextURL contextUrl = getContextUrl(cdsRequest, true, uriInfo.getSelectOption(), expand);
				if (uriInfo.getApplyOption() != null) {
					EdmAssistedSerializerOptions optionsForApply = EdmAssistedSerializerOptions.with().contextURL(contextUrl).build();
					serializerResult = createSerializerForApply(odataRequest, responseFormat).entityCollection(
							globals.getServiceMetadata(), (EdmEntityType) cdsRequest.getResponseType(),
							entityCollection, optionsForApply);
				} else {
					EntityCollectionSerializerOptions options = EntityCollectionSerializerOptions.with()
							.contextURL(contextUrl)
							.select(uriInfo.getSelectOption())
							.expand(expand)
							.count(uriInfo.getCountOption())
							.build();
					serializerResult = createSerializer(odataRequest, responseFormat).entityCollection(globals.getServiceMetadata(), (EdmEntityType) cdsRequest.getResponseType(), entityCollection, options);
				}
			} catch (SerializerException e) {
				throw new ErrorStatusException(CdsErrorStatuses.RESPONSE_SERIALIZATION_FAILED, e);
			}
			setODataResponse(cdsRequest, cdsResponse, odataResponse, serializerResult.getContent(), responseFormat);
		});
	}

	@Override
	public void processSingleComplex(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat, ContentType responseFormat) {
		processRequest(odataRequest, odataResponse, uriInfo, requestFormat, (cdsRequest, cdsResponse) -> {
			ExpandTreeBuilder expandBuilder = ExpandTreeBuilderImpl.create();
			Property complexProperty = resultSetProcessor.toComplex(cdsRequest, cdsResponse.getResult(), expandBuilder);
			ExpandOption expand = uriInfo.getExpandOption() != null ? uriInfo.getExpandOption() : expandBuilder.build();

			SerializerResult serializerResult = null;
			if(cdsRequest.getReturnPreference() != Return.MINIMAL) {
				ContextURL contextURL = getContextUrl(cdsRequest, false, uriInfo.getSelectOption(), expand);
				ComplexSerializerOptions options = ComplexSerializerOptions.with()
						.contextURL(contextURL)
						.select(uriInfo.getSelectOption())
						.expand(expand)
						.build();

				try {
					serializerResult = createSerializer(odataRequest, responseFormat).complex(globals.getServiceMetadata(), (EdmComplexType) cdsRequest.getResponseType(), complexProperty, options);
				} catch (SerializerException e) {
					throw new ErrorStatusException(CdsErrorStatuses.RESPONSE_SERIALIZATION_FAILED, e);
				}
			}

			InputStream content = serializerResult != null ? serializerResult.getContent() : null;
			setODataResponse(cdsRequest, cdsResponse, odataResponse, content, responseFormat);
		});
	}

	@Override
	public void processCollectionComplex(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat, ContentType responseFormat) {
		processRequest(odataRequest, odataResponse, uriInfo, requestFormat, (cdsRequest, cdsResponse) -> {
			ExpandTreeBuilder expandBuilder = ExpandTreeBuilderImpl.create();
			Property complexProperty = resultSetProcessor.toComplexCollection(cdsRequest, cdsResponse.getResult(), expandBuilder);
			ExpandOption expand = uriInfo.getExpandOption() != null ? uriInfo.getExpandOption() : expandBuilder.build();

			SerializerResult serializerResult = null;
			if (cdsRequest.getReturnPreference() != Return.MINIMAL) {
				ContextURL contextURL = getContextUrl(cdsRequest, true, uriInfo.getSelectOption(), expand);
				ComplexSerializerOptions options = ComplexSerializerOptions.with()
						.contextURL(contextURL)
						.select(uriInfo.getSelectOption())
						.expand(expand)
						.build();

				try {
					serializerResult = createSerializer(odataRequest, responseFormat).complexCollection(globals.getServiceMetadata(), (EdmComplexType) cdsRequest.getResponseType(), complexProperty, options);
				} catch (SerializerException e) {
					throw new ErrorStatusException(CdsErrorStatuses.RESPONSE_SERIALIZATION_FAILED, e);
				}
			}

			InputStream content = serializerResult != null ? serializerResult.getContent() : null;
			setODataResponse(cdsRequest, cdsResponse, odataResponse, content, responseFormat);
		});
	}

	@SuppressWarnings("deprecation")
	public void processSinglePrimitive(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat, ContentType responseFormat) {
		final ChangeSetContextSPI changeSetContext;
		if (isGetStreamContext(odataRequest, uriInfo, requestFormat)) {
			changeSetContext = ChangeSetContextSPI.open(); // NOSONAR
			globals.getUnclosedChangeSetTracker().set(true);
		} else {
			changeSetContext = null;
		}

		processRequest(odataRequest, odataResponse, uriInfo, requestFormat, (cdsRequest, cdsResponse) -> {
			SerializerResult serializerResult = null;
			Property property = resultSetProcessor.toPrimitive(cdsRequest, cdsResponse.getResult());
			boolean isStream = (cdsRequest.getLastTypedResource().getType() instanceof EdmStream);

			if (cdsRequest.getReturnPreference() != Return.MINIMAL && !isStream) {
				ContextURL contextURL = getContextUrl(cdsRequest, false, null, null);
				PrimitiveSerializerOptions.Builder builder = PrimitiveSerializerOptions.with().contextURL(contextURL);

				getEdmProperty(cdsRequest).ifPresent(edmProp -> builder.facetsFrom(edmProp));
				PrimitiveSerializerOptions options = builder.build();

				try {
					serializerResult = createSerializer(odataRequest, responseFormat).primitive(
							globals.getServiceMetadata(), (EdmPrimitiveType) cdsRequest.getResponseType(), property,
							options);
				} catch (SerializerException e) {
					throw new ErrorStatusException(CdsErrorStatuses.RESPONSE_SERIALIZATION_FAILED, e);
				}
			}

			ContentType contentType;
			if (!StringUtils.isEmpty(cdsResponse.getContentType())) {
				contentType = ContentType.create(cdsResponse.getContentType());
			} else {
				contentType = responseFormat;
			}

			InputStream content;
			if (changeSetContext != null && property.getValue() != null) {
				InputStream stream;
				if (property.getValue() instanceof Reader) {
					Charset charset = ODataUtils.getCharset(contentType);
					stream = new ReaderInputStream((Reader) property.getValue(), charset);
				} else {
					stream = (InputStream) property.getValue();
				}
				content = new ChangeSetContextAwareInputStream(stream, changeSetContext, globals.getUnclosedChangeSetTracker());
			} else {
				content = serializerResult != null ? serializerResult.getContent() : null;
			}

			setContentDispositionHeaderIfNotNull(odataResponse, cdsResponse.getContentDispositionFilename());
			setODataResponse(cdsRequest, cdsResponse, odataResponse, content, contentType);
		});
	}

	@Override
	public void processSinglePrimitiveValue(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat, ContentType responseFormat) {
		processRequest(odataRequest, odataResponse, uriInfo, requestFormat, (cdsRequest, cdsResponse) -> {
			if (isValueRequestOnStreamProperty(cdsRequest)) {
				UriResourcePartTyped lastTypedResource = cdsRequest.getLastTypedResource();
				throw new ErrorStatusException(CdsErrorStatuses.VALUE_ACCESS_NOT_ALLOWED, lastTypedResource.getSegmentValue());
			}

			Property property = resultSetProcessor.toPrimitive(cdsRequest, cdsResponse.getResult());
			Object value = property.getValue();

			InputStream content = null;
			if (value != null && cdsRequest.getReturnPreference() != Return.MINIMAL) {
				org.apache.olingo.server.api.serializer.PrimitiveValueSerializerOptions.Builder builder = PrimitiveValueSerializerOptions
						.with();

				getEdmProperty(cdsRequest).ifPresent(edmProp -> builder.facetsFrom(edmProp));
				PrimitiveValueSerializerOptions options = builder.build();

				try {
					content = globals.getOData().createFixedFormatSerializer().primitiveValue((EdmPrimitiveType) cdsRequest.getResponseType(), value, options);
				} catch (SerializerException e) {
					throw new ErrorStatusException(CdsErrorStatuses.RESPONSE_SERIALIZATION_FAILED, e);
				}
			}

			setODataResponse(cdsRequest, cdsResponse, odataResponse, content, responseFormat);
		});
	}

	@Override
	public void processCollectionPrimitive(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat, ContentType responseFormat) {
		processRequest(odataRequest, odataResponse, uriInfo, requestFormat, (cdsRequest, cdsResponse) -> {
			Property property = resultSetProcessor.toPrimitiveCollection(cdsRequest, cdsResponse.getResult());

			SerializerResult serializerResult = null;
			if (cdsRequest.getReturnPreference() != Return.MINIMAL) {
				ContextURL contextURL = getContextUrl(cdsRequest, true, null, null);
				PrimitiveSerializerOptions options = PrimitiveSerializerOptions.with().contextURL(contextURL).build();

				try {
					serializerResult = createSerializer(odataRequest, responseFormat).primitiveCollection(globals.getServiceMetadata(), (EdmPrimitiveType) cdsRequest.getResponseType(), property, options);
				} catch (SerializerException e) {
					throw new ErrorStatusException(CdsErrorStatuses.RESPONSE_SERIALIZATION_FAILED, e);
				}
			}

			InputStream content = serializerResult != null ? serializerResult.getContent() : null;
			setODataResponse(cdsRequest, cdsResponse, odataResponse, content, responseFormat);
		});
	}
}
