/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.processors.request;


import java.io.Reader;
import java.nio.charset.Charset;
import java.time.Instant;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.olingo.commons.api.edm.EdmEntityType;
import org.apache.olingo.commons.api.edm.EdmProperty;
import org.apache.olingo.commons.api.edm.EdmType;
import org.apache.olingo.commons.api.format.ContentType;
import org.apache.olingo.commons.api.http.HttpHeader;
import org.apache.olingo.commons.api.http.HttpMethod;
import org.apache.olingo.server.api.ODataRequest;
import org.apache.olingo.server.api.prefer.Preferences.Return;
import org.apache.olingo.server.api.uri.UriInfo;
import org.apache.olingo.server.api.uri.UriResource;
import org.apache.olingo.server.api.uri.UriResourceEntitySet;
import org.apache.olingo.server.api.uri.UriResourceKind;
import org.apache.olingo.server.api.uri.UriResourceNavigation;
import org.apache.olingo.server.api.uri.UriResourcePartTyped;

import com.sap.cds.adapter.odata.v4.CdsRequestGlobals;
import com.sap.cds.adapter.odata.v4.utils.EdmUtils;
import com.sap.cds.adapter.odata.v4.utils.ODataUtils;
import com.sap.cds.adapter.odata.v4.utils.mapper.EdmxFlavourMapper;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsDefinition;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.services.request.ParameterInfo;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.CorrelationIdUtils;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.HttpHeaders;
import com.sap.cds.services.utils.LocaleUtils;
import com.sap.cds.services.utils.QueryParameters;
import com.sap.cds.services.utils.TemporalRangeUtils;
import com.sap.cds.services.utils.model.CdsAnnotations;
public class CdsODataRequest implements ParameterInfo {

	public static final String DELTA = "cds.delta";
	public static final String DELTA_DELETE = "cds.delete";
	public static final String DELTA_UPSERT = "cds.upsert";

	private final CdsRequestGlobals globals;
	private final EdmxFlavourMapper edmxFlavourMapper;
	private final EdmUtils edmUtils;

	private final ODataRequest odataRequest;
	private final UriInfo uriInfo;

	private final ContentType requestFormat;
	private final Map<String, String> queryParameters = new HashMap<>();
	private Map<String, Object> bodyMap;

	// required to be able to pass the value from one thread to another
	private String correlationId;

	private Locale locale; // calculated according to given request parameters - might change within a batch request
	private Instant validFrom;
	private Instant validTo;

	private Return returnPreference;

	public CdsODataRequest(ODataRequest odataRequest, UriInfo uriInfo, ContentType requestFormat, CdsRequestGlobals globals) {
		this.odataRequest = odataRequest;
		this.uriInfo = uriInfo;
		this.requestFormat = requestFormat;
		this.globals = globals;
		this.edmxFlavourMapper = EdmxFlavourMapper.create(globals.getEdmxFlavour(), true);
		this.edmUtils = new EdmUtils(globals);

		Stream.concat(uriInfo.getSystemQueryOptions().stream(), uriInfo.getCustomQueryOptions().stream()).forEach(query -> 
			queryParameters.put(query.getName(), query.getText())
		);
	}

	public ODataRequest getODataRequest() {
		return this.odataRequest;
	}

	public UriInfo getUriInfo() {
		return this.uriInfo;
	}

	public UriResource getLastResource() {
		List<UriResource> resources = uriInfo.getUriResourceParts();
		return resources.get(resources.size() - 1);
	}

	public UriResourcePartTyped getLastTypedResource() {
		List<UriResource> resources = uriInfo.getUriResourceParts();
		for(int i=resources.size() - 1; i >= 0; --i) {
			UriResource resource = resources.get(i);
			if(resource instanceof UriResourcePartTyped typed) {
				return typed;
			}
		}
		return null;
	}

	public UriResourcePartTyped getLastEntityResource(boolean includeContainment) {
		UriResource lastEntityResource = null;
		for(UriResource resource : uriInfo.getUriResourceParts()) {
			if(resource.getKind() == UriResourceKind.entitySet || resource.getKind() == UriResourceKind.singleton) {
				lastEntityResource = resource;
			} else if (resource.getKind() == UriResourceKind.navigationProperty) {
				if(includeContainment || !((UriResourceNavigation) resource).getProperty().containsTarget()) {
					lastEntityResource = resource;
				}
			} else {
				break;
			}
		}
		return (UriResourcePartTyped) lastEntityResource;
	}

	public CdsEntity getLastEntity() {
		String qualifiedName = edmUtils.getCdsEntityName((EdmEntityType) getLastEntityResource(true).getType());
		return globals.getModel().getEntity(qualifiedName);
	}

	public Map<String, Object> getBodyMap() {
		if(bodyMap == null) {
			bodyMap = remapBodyMap(extractBodyMap());
		}
		return bodyMap;
	}

	public boolean isDeltaCollection() {
		return getBodyMap().containsKey(DELTA);
	}

	private Map<String, Object> extractBodyMap() {
		UriResourcePartTyped lastResource = getLastTypedResource();

		RequestBodyExtractor bodyExtractor = new RequestBodyExtractor(globals, odataRequest, requestFormat);
		if (lastResource.getKind() == UriResourceKind.function) {
			return bodyExtractor.extractBodyFromFunctionParameters(lastResource);
		} else {
			Optional<EdmProperty> edmProperty = edmUtils.getEdmProperty(lastResource);

			if (odataRequest.getMethod().equals(HttpMethod.DELETE)) {
				return bodyExtractor.extractBodyFromProperty(edmProperty, lastResource);
			} else if (isPatchEntitySet(lastResource)) {
				String qualifiedName = edmUtils.getCdsEntityName((EdmEntityType) getLastEntityResource(true).getType());
				CdsEntity entity = globals.getModel().getEntity(qualifiedName);
				if (CdsAnnotations.UPDATABLE_DELTA.isTrue(entity)) {
					return bodyExtractor.extractDeltaCollectionFromValue(lastResource, getResponseType());
				}
				throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_DELTA_UPDATABLE);
			} else if (requestFormat == null) {
				throw new ErrorStatusException(CdsErrorStatuses.MISSING_CONTENT_TYPE);
			} else if (requestFormat.isCompatible(ContentType.TEXT_PLAIN) && !edmUtils.isEdmStream(edmProperty)) {
				return bodyExtractor.extractBodyFromPrimitiveValue(edmProperty, lastResource);
			} else if (requestFormat.isCompatible(ContentType.APPLICATION_JSON) && !edmUtils.isEdmStream(edmProperty)) {
				return bodyExtractor.extractBodyFromJson(edmProperty, lastResource, getResponseType());
			} else {
				Charset charset = null;
				if (edmProperty.isPresent()) {
					CdsEntity entity = getLastEntity();
					String elementName = edmxFlavourMapper.remap(edmProperty.get().getName(), entity);
					CdsType type = entity.getElement(elementName).getType();
					if (type.isSimple() && CdsBaseType.cdsJavaMediaType(type.as(CdsSimpleType.class).getType()) == Reader.class) {
						charset = ODataUtils.getCharset(requestFormat);
					}
				}

				return bodyExtractor.extractBodyFromBinaryValue(edmProperty, lastResource, charset);
			}
		}
	}

	private boolean isPatchEntitySet(UriResourcePartTyped lastResource) {
		return odataRequest.getMethod().equals(HttpMethod.PATCH)
				&& lastResource.getKind() == UriResourceKind.entitySet
				&& ((UriResourceEntitySet) lastResource).getKeyPredicates().isEmpty();
	}

	@SuppressWarnings("unchecked")
	private Map<String, Object> remapBodyMap(Map<String, Object> bodyMap) {
		UriResource lastResource = getLastTypedResource();
		if(lastResource.getKind() == UriResourceKind.function || lastResource.getKind() == UriResourceKind.action) {
			CdsDefinition operation = edmUtils.getCdsOperation(this);
			Map<String, CdsType> parameters = edmUtils.getCdsOperationParameters(operation);
			for(String key : bodyMap.keySet()) {
				CdsType parameterType = parameters.get(key);
				if(parameterType instanceof CdsStructuredType) {
					Object value = bodyMap.get(key);
					if(value instanceof Map) {
						edmxFlavourMapper.remap((Map<String, Object>) value, parameterType.as(CdsStructuredType.class));
					} else if (value instanceof List) {
						edmxFlavourMapper.remap((List<Map<String, Object>>) value, parameterType.as(CdsStructuredType.class));
					}
				}
			}
			return bodyMap;
		} else {
			return edmxFlavourMapper.remap(bodyMap, getLastEntity());
		}
	}

	public EdmType getResponseType() {
		UriResourcePartTyped lastResource = getLastTypedResource();
		if(lastResource != null) {
			return lastResource.getType();
		}
		return null;
	}

	public Return getReturnPreference() {
		if (returnPreference == null) {
			returnPreference = globals.getOData().createPreferences(odataRequest.getHeaders(HttpHeader.PREFER)).getReturn();
		}
		return returnPreference;
	}

	public boolean hasApply() {
		return getUriInfo() != null && getUriInfo().getApplyOption() != null;
	}

	// Request Parameters API

	@Override
	public String getCorrelationId() {
		if (correlationId == null) {
			correlationId = CorrelationIdUtils.getOrGenerateCorrelationId(this);
		}
		return correlationId;
	}

	@Override
	public String getHeader(String id) {
		return odataRequest.getHeader(id);
	}

	@Override
	public Map<String, String> getHeaders() {
		return odataRequest.getAllHeaders().entrySet().stream().collect(
				Collectors.toMap(Map.Entry::getKey, e -> getHeader(e.getKey()) )) ; // projects List<String> to String
	}

	@Override
	public String getQueryParameter(String key) {
		return queryParameters.get(key);
	}

	@Override
	public Map<String, String> getQueryParams() {
		return queryParameters;
	}

	@Override
	public Locale getLocale() {
		if(this.locale == null) {
			this.locale = new LocaleUtils(globals.getRuntime().getEnvironment().getCdsProperties()).getLocale(
					getQueryParameter(QueryParameters.SAP_LOCALE),
					getHeader(HttpHeaders.ACCEPT_LANGUAGE),
					getQueryParameter(QueryParameters.SAP_LANGUAGE),
					getHeader(HttpHeaders.X_SAP_REQUEST_LANGUAGE_HEADER));
		}
		return this.locale;
	}

	@Override
	public Instant getValidFrom() {
		if (this.validFrom == null) {
			initTemporalRange();
		}
		return this.validFrom;
	}

	@Override
	public Instant getValidTo() {
		if (this.validTo == null) {
			initTemporalRange();
		}
		return this.validTo;
	}

	private void initTemporalRange() {
		Instant[] range = TemporalRangeUtils.getTemporalRanges(
				getQueryParameter(QueryParameters.VALID_FROM),
				getQueryParameter(QueryParameters.VALID_TO),
				getQueryParameter(QueryParameters.VALID_AT));

		this.validFrom = range[0];
		this.validTo = range[1];
	}
}
