/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.query.apply;

import java.util.HashMap;
import java.util.List;
import java.util.Optional;

import org.apache.olingo.server.api.uri.UriResource;
import org.apache.olingo.server.api.uri.queryoption.apply.AggregateExpression.StandardMethod;
import org.apache.olingo.server.api.uri.queryoption.expression.Expression;

import com.sap.cds.adapter.odata.v4.query.ExpressionParser;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.ElementRef;
import com.sap.cds.ql.Value;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.util.CdsModelUtils;

public class ElementAggregator {
	private static final String SINGLE_VALUE = "singleValue";

	private final ExpressionParser expressionParser;

	public ElementAggregator(ExpressionParser expressionParser) {
		this.expressionParser = expressionParser;
	}

	public Value<?> genericAggregate(Expression expr, StandardMethod standardMethod) {
		Value<?> value = (Value<?>) expressionParser.parseValue(expr);
		return toFunctionCall(value, standardMethod);
	}

	public Value<?> customAggregate(List<UriResource> path) {
		ElementRef<Object> ref = CQL.get(expressionParser.toSegmentList(path));

		return customAggregate(ref);
	}

	public Value<?> customAggregate(ElementRef<Object> ref) {
		if (ref.lastSegment().equals("$count")) {
			return CQL.count();
		}

		CdsElement element = CdsModelUtils.element(expressionParser.getRootType(), ref);
		String methodName = getAggregationMethodName(element);

		return toFunctionCall(ref, methodName);
	}

	private String getAggregationMethodName(CdsElement element) {
		HashMap<String, String> valueMap = CdsAnnotations.AGGREGATION_DEFAULT.getOrValue(element,
				new HashMap<String, String>());
		String value = valueMap.get("#");
		if (value != null) {
			return value;
		}

		boolean isCurrencyCode = CdsAnnotations.SEMANTICS_CURRENCY_CODE.isTrue(element);
		boolean isUnitOfMeasure = CdsAnnotations.SEMANTICS_UNIT_OF_MEASURE.isTrue(element);

		if (isCurrencyCode || isUnitOfMeasure) {
			return SINGLE_VALUE;
		}

		throw new ErrorStatusException(CdsErrorStatuses.NO_CUSTOM_AGGREGATE_DEFINED, element);
	}

	Value<?> toFunctionCall(ElementRef<Object> ref, String methodName) {
		Value<?> functionCall;
		switch (methodName) {
			case "AVG":
			case "AVERAGE":
				functionCall = ref.average();
				break;
			case "COUNT":
				functionCall = CQL.func(methodName, ref).type(Long.class);
				break;
			case "COUNT_DISTINCT":
				functionCall = ref.countDistinct();
				break;
			case "MIN":
			case "MAX":
			case "SUM":
			case SINGLE_VALUE:
				functionCall = CQL.func(methodName, ref);
				type(ref).ifPresent(functionCall::type);
				break;
			default:
				throw new ErrorStatusException(CdsErrorStatuses.UNKONWN_AGGREGATION_METHOD, methodName);
		}
		return functionCall;
	}

	public Value<?> toFunctionCall(Value<?> value, StandardMethod standardMethod) {
		Value<?> functionCall;
		switch (standardMethod) {
			case AVERAGE:
				functionCall = value.average();
				break;
			case COUNT_DISTINCT:
				functionCall = value.countDistinct();
				break;
			case MIN:
				functionCall = value.min();
				type(value).ifPresent(functionCall::type);
				break;
			case MAX:
				functionCall = value.max();
				type(value).ifPresent(functionCall::type);
				break;
			default:
				functionCall = CQL.func(standardMethod.name(), value);
		}

		return functionCall;
	}

	private Optional<CdsBaseType> type(Value<?> value) {
		if (value.isRef()) {
			CdsElement element = CdsModelUtils.element(expressionParser.getRootType(), value.asRef());
			CdsType t = element.getType();
			if (t.isSimple()) {
				return Optional.ofNullable(t.as(CdsSimpleType.class).getType());
			}
		}
		return Optional.empty();
	}

}
