/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.query.apply;

import java.util.List;

import org.apache.olingo.server.api.uri.queryoption.ApplyItem;
import org.apache.olingo.server.api.uri.queryoption.apply.Ancestors;
import org.apache.olingo.server.api.uri.queryoption.apply.Descendants;
import org.apache.olingo.server.api.uri.queryoption.apply.HierarchicalSubsetTransformation;

import com.sap.cds.adapter.odata.v4.query.ExpressionParser;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.StructuredTypeRef;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.transformation.CqnAncestorsTransformation;
import com.sap.cds.ql.cqn.transformation.CqnTransformation;

public abstract class HierarchySubsetTransformation {

	private final CqnStructuredTypeRef hierarchyReference;
	private final String hierarchyQualifier;
	private final CqnElementRef nodeProperty;
	private final List<CqnTransformation> transformations;
	private final int distanceFromStart;
	private final boolean keepStart;

	public HierarchySubsetTransformation(CqnStructuredTypeRef hierarchyReference, String hierarchyQualifier,
			CqnElementRef nodeProperty, List<CqnTransformation> transformations, int distanceFromStart,
			boolean keepStart) {
		this.hierarchyReference = hierarchyReference;
		this.hierarchyQualifier = hierarchyQualifier;
		this.nodeProperty = nodeProperty;
		this.transformations = transformations;
		this.distanceFromStart = distanceFromStart;
		this.keepStart = keepStart;
	}

	public static CqnAncestorsTransformation ancestors(Ancestors ancestors, ExpressionParser expressionParser) {
		ParamConverter converter = new ParamConverter(ancestors, expressionParser);
		return new AncestorsTransformation(converter.hierarchyReference(), converter.hierarchyQualifier(),
				converter.nodeProperty(), converter.transformations(), converter.distanctFromStart(),
				converter.keepStart());
	}

	public static CqnTransformation descendants(Descendants descendants, ExpressionParser expressionParser) {
		ParamConverter converter = new ParamConverter(descendants, expressionParser);
		return new DescendantsTransformation(converter.hierarchyReference(), converter.hierarchyQualifier(),
				converter.nodeProperty(), converter.transformations(), converter.distanctFromStart(),
				converter.keepStart());
	}

	private static class ParamConverter {
		private ExpressionParser parser;
		private HierarchicalSubsetTransformation transformation;

		public ParamConverter(HierarchicalSubsetTransformation transformation, ExpressionParser expressionParser) {
			this.transformation = transformation;
			this.parser = expressionParser;
		}

		StructuredTypeRef hierarchyReference() {
			return CQL.to(parser.toSegmentList(transformation.getHierarchyReference())).asRef();
		}

		String hierarchyQualifier() {
			return transformation.getHierarchyQualifier();
		}

		CqnElementRef nodeProperty() {
			return CQL.get(parser.toSegmentList(transformation.getNodeProperty()));
		}

		List<CqnTransformation> transformations() {
			List<ApplyItem> applies = transformation.getTransformations();
			List<CqnTransformation> transformations;
			ApplyHandler applyHandler = new ApplyHandler(applies, parser);
			transformations = applyHandler.getTransformations(0);
			return transformations;
		}

		int distanctFromStart() {
			return transformation.getDistanceFromStart() != null ? transformation.getDistanceFromStart() : -1;
		}

		boolean keepStart() {
			return transformation.isKeepStart() == Boolean.TRUE;// TODO check default
		}
	}

	public CqnStructuredTypeRef hierarchyReference() {
		return hierarchyReference;
	}

	public String hierarchyQualifier() {
		return hierarchyQualifier;
	}

	public CqnElementRef nodeProperty() {
		return nodeProperty;
	}

	public List<CqnTransformation> transformations() {
		return transformations;
	}

	public int distanceFromStart() {
		return distanceFromStart;
	}

	public boolean keepStart() {
		return keepStart;
	}

}
