/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.query.apply;

import java.util.Map;
import java.util.stream.Collectors;

import org.apache.olingo.server.api.uri.UriParameter;
import org.apache.olingo.server.api.uri.queryoption.apply.CustomFunction;

import com.sap.cds.adapter.odata.v4.metadata.extension.CustomVocabularyCatalog;
import com.sap.cds.adapter.odata.v4.query.ExpressionParser;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.transformation.CqnTopLevelsTransformation;

public class TopLevelsTransformation implements CqnTopLevelsTransformation {

	private static final String NAME = CustomVocabularyCatalog.COM_SAP_VOCABULARY_HIERARCHY + "."
			+ CustomVocabularyCatalog.TOP_LEVELS;

	private final CqnStructuredTypeRef hierarchyReference;
	private final String hierarchyQualifier;
	private final CqnElementRef nodeProperty;
	private final long levels;

	public TopLevelsTransformation(CqnStructuredTypeRef hierarchyReference, String hierarchyQualifier,
			CqnElementRef nodeProperty, long levels) {
		this.hierarchyReference = hierarchyReference;
		this.hierarchyQualifier = hierarchyQualifier;
		this.nodeProperty = nodeProperty;
		this.levels = levels;
	}

	public static CqnTopLevelsTransformation topLevels(CustomFunction custom, ExpressionParser expressionParser) {
		ParamConverter converter = new ParamConverter(custom, expressionParser);
		return new TopLevelsTransformation(converter.hierarchyReference(), converter.hierarchyQualifier(),
				converter.nodeProperty(), converter.levels());
	}

	private static class ParamConverter {
		private final ExpressionParser parser;
		private final Map<String, UriParameter> params;

		public ParamConverter(CustomFunction transformation, ExpressionParser expressionParser) {
			params = transformation.getParameters().stream().collect(Collectors.toMap(UriParameter::getName, p -> p));
			this.parser = expressionParser;
		}

		CqnStructuredTypeRef hierarchyReference() {
			UriParameter ref = params.get("HierarchyNodes");

			return parser.parseStructuredTypeRef(ref.getExpression());
		}

		String hierarchyQualifier() {
			return unquote(params.get("HierarchyQualifier").getText());
		}

		CqnElementRef nodeProperty() {
			UriParameter node = params.get("NodeProperty");
			return CQL.get(node.getText());
		}

		long levels() {
			return params.containsKey("Levels") ? Long.parseLong(params.get("Levels").getText()) : -1l;
		}
	}

	private static String unquote(String text) {
		if (text.startsWith("'")) {
			text = text.substring(1, text.length() - 2);
		}
		return text;
	}
	
	@Override
	public CqnStructuredTypeRef hierarchyReference() {
		return hierarchyReference;
	}

	@Override
	public String hierarchyQualifier() {
		return hierarchyQualifier;
	}

	@Override
	public CqnElementRef nodeProperty() {
		return nodeProperty;
	}

	@Override
	public long levels() {
		return levels;
	}
	
	public static boolean handles(CustomFunction function) {
		return NAME.equals(function.getFunction().getFullQualifiedName().getFullQualifiedNameAsString());
	}

}
