/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.utils;

import java.io.IOException;
import java.io.InputStream;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;

import org.apache.commons.io.IOUtils;
import org.apache.commons.io.input.ProxyInputStream;

import com.sap.cds.services.changeset.ChangeSetContextSPI;

public class ChangeSetContextAwareInputStream extends ProxyInputStream {

	private final ChangeSetContextSPI changeSetContext;
	private final AtomicBoolean unclosedChangeSetTracker;

	public ChangeSetContextAwareInputStream(InputStream in, ChangeSetContextSPI changeSetContext, AtomicBoolean unclosedChangeSetTracker) {
		super(Objects.requireNonNull(in));
		this.changeSetContext = Objects.requireNonNull(changeSetContext);
		this.unclosedChangeSetTracker = Objects.requireNonNull(unclosedChangeSetTracker);
	}

	@Override
	public void close() throws IOException {
		try {
			this.changeSetContext.close();
		} catch (Exception e) {
			throw new IOException(e);
		} finally {
			unclosedChangeSetTracker.set(false);
		}
		super.close();
	}

	@Override
	protected void afterRead(int n) throws IOException {
		if (n == IOUtils.EOF) {
			try {
				this.changeSetContext.close();
			} catch (Exception e) {
				throw new IOException(e);
			} finally {
				unclosedChangeSetTracker.set(false);
			}
		}
	}

	@Override
	protected void handleIOException(IOException e) throws IOException {
		try {
			this.changeSetContext.markForCancel();
			this.changeSetContext.close();
		} catch (Exception e1) {
			throw new IOException(e1);
		} finally {
			unclosedChangeSetTracker.set(false);
		}
		throw e;
	}

}
