/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.metadata;

import org.apache.olingo.commons.api.format.ContentType;
import org.apache.olingo.commons.api.http.HttpHeader;
import org.apache.olingo.commons.api.http.HttpMethod;
import org.apache.olingo.commons.api.http.HttpStatusCode;
import org.apache.olingo.server.api.OData;
import org.apache.olingo.server.api.ODataApplicationException;
import org.apache.olingo.server.api.ODataLibraryException;
import org.apache.olingo.server.api.ODataRequest;
import org.apache.olingo.server.api.ODataResponse;
import org.apache.olingo.server.api.ServiceMetadata;
import org.apache.olingo.server.api.etag.ETagHelper;
import org.apache.olingo.server.api.etag.ServiceMetadataETagSupport;
import org.apache.olingo.server.api.processor.ServiceDocumentProcessor;
import org.apache.olingo.server.api.serializer.ODataSerializer;
import org.apache.olingo.server.api.uri.UriInfo;

import com.sap.cds.adapter.odata.v4.utils.ODataUtils;
import com.sap.cds.services.runtime.CdsRuntime;

public class CustomServiceDocumentProcessor implements ServiceDocumentProcessor {

	private final CdsRuntime runtime;
	private OData odata;
	private ServiceMetadata serviceMetadata;

	public CustomServiceDocumentProcessor(CdsRuntime runtime) {
		this.runtime = runtime;
	}

	@Override
	public void init(OData odata, ServiceMetadata serviceMetadata) {
		this.odata = odata;
		this.serviceMetadata = serviceMetadata;
	}

	@Override
	public void readServiceDocument(ODataRequest request, ODataResponse response, UriInfo uriInfo, ContentType responseFormat) throws ODataApplicationException, ODataLibraryException {
		boolean isNotModified = false;
		ServiceMetadataETagSupport eTagSupport = serviceMetadata.getServiceMetadataETagSupport();
		if (eTagSupport != null && eTagSupport.getServiceDocumentETag() != null) {
			// Set application etag at response
			response.setHeader(HttpHeader.ETAG, eTagSupport.getServiceDocumentETag());
			// Check if service document has been modified
			ETagHelper eTagHelper = odata.createETagHelper();
			isNotModified = eTagHelper.checkReadPreconditions(eTagSupport.getServiceDocumentETag(), request.getHeaders(HttpHeader.IF_MATCH), request.getHeaders(HttpHeader.IF_NONE_MATCH));
			// allow caching, but enforce revalidation using etags
			response.setHeader(HttpHeader.CACHE_CONTROL, "max-age=0");
		}

		// Send the correct response
		if (isNotModified) {
		  	response.setStatusCode(HttpStatusCode.NOT_MODIFIED.getStatusCode());
		} else {
			// HTTP HEAD requires no payload but a 200 OK response
			if (HttpMethod.HEAD == request.getMethod()) {
				response.setStatusCode(HttpStatusCode.OK.getStatusCode());
			} else {
				ODataSerializer serializer = ODataUtils.createSerializer(odata, request, responseFormat);
				// add custom URL handling
				String baseUri = runtime.getEnvironment().getCdsProperties().getOdataV4().isContextAbsoluteUrl() ? request.getRawBaseUri() : null;
				response.setContent(serializer.serviceDocument(serviceMetadata, baseUri).getContent());
				response.setStatusCode(HttpStatusCode.OK.getStatusCode());
				response.setHeader(HttpHeader.CONTENT_TYPE, responseFormat.toContentTypeString());
			}
		}
	}
}
