/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.metadata;

import java.io.ByteArrayInputStream;
import java.io.InputStream;

import org.apache.olingo.commons.api.edm.provider.CsdlEdmProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.hash.Hashing;
import com.sap.cds.services.runtime.CdsRuntime;

public class MetadataInfo {

	private static Logger log = LoggerFactory.getLogger(MetadataInfo.class);

	private final byte[] edmxBytes;
	private final CsdlEdmProvider provider;
	private final String etag;

	public static MetadataInfo create(String serviceName, byte[] edmxBytes, CdsRuntime runtime) throws Exception {
		if (!isEmpty(edmxBytes)) {
			log.info("Loading OData V4 metadata for service '{}'", serviceName);
		}
		return new MetadataInfo(edmxBytes, runtime.getEnvironment().getCdsProperties().getOdataV4().getCdsToEdm().isEnabled());
	}

	private MetadataInfo(byte[] edmxBytes, boolean cdsToEdm) throws Exception {
		this.edmxBytes = edmxBytes;
		this.provider = calculateProvider(edmxBytes, cdsToEdm);
		this.etag = calculateMetadataEtag(edmxBytes);
	}

	private static CsdlEdmProvider calculateProvider(byte[] edmxBytes, boolean cdsToEdm) throws Exception {
		if (cdsToEdm || isEmpty(edmxBytes)) {
			return null;
		}
		try (ByteArrayInputStream stream = new ByteArrayInputStream(edmxBytes)) {
			return new ODataEdmProvider(stream);
		}
	}

	private static String calculateMetadataEtag(byte[] edmxBytes) {
		if (isEmpty(edmxBytes)) {
			return null;
		}

		return Hashing.sha256().hashBytes(edmxBytes).toString();
	}

	private static boolean isEmpty(byte[] edmxBytes) {
		return edmxBytes == null || edmxBytes.length == 0;
	}

	public boolean isEmpty() {
		return isEmpty(edmxBytes);
	}

	public InputStream getEdmx() {
		return new ByteArrayInputStream(edmxBytes);
	}

	public CsdlEdmProvider getEdmProvider() {
		return provider;
	}

	public String getETag() {
		return etag;
	}
}
