/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.metadata;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import org.apache.olingo.commons.api.edm.FullQualifiedName;
import org.apache.olingo.commons.api.edm.provider.CsdlAction;
import org.apache.olingo.commons.api.edm.provider.CsdlActionImport;
import org.apache.olingo.commons.api.edm.provider.CsdlAliasInfo;
import org.apache.olingo.commons.api.edm.provider.CsdlAnnotations;
import org.apache.olingo.commons.api.edm.provider.CsdlComplexType;
import org.apache.olingo.commons.api.edm.provider.CsdlEdmProvider;
import org.apache.olingo.commons.api.edm.provider.CsdlEntityContainer;
import org.apache.olingo.commons.api.edm.provider.CsdlEntityContainerInfo;
import org.apache.olingo.commons.api.edm.provider.CsdlEntitySet;
import org.apache.olingo.commons.api.edm.provider.CsdlEntityType;
import org.apache.olingo.commons.api.edm.provider.CsdlEnumType;
import org.apache.olingo.commons.api.edm.provider.CsdlFunction;
import org.apache.olingo.commons.api.edm.provider.CsdlFunctionImport;
import org.apache.olingo.commons.api.edm.provider.CsdlSchema;
import org.apache.olingo.commons.api.edm.provider.CsdlSingleton;
import org.apache.olingo.commons.api.edm.provider.CsdlTerm;
import org.apache.olingo.commons.api.edm.provider.CsdlTypeDefinition;
import org.apache.olingo.commons.api.ex.ODataException;

import com.sap.cds.adapter.odata.v4.metadata.extension.CustomVocabularyCatalog;

/*
 * Wrapper class that extends the base EdmProvider by adding 
 * SAP specific schemata and functions.
 */
public class ODataExtendedEdmProvider implements CsdlEdmProvider {

	private static final CsdlSchema customSchema = CustomVocabularyCatalog.getHierarchySchema();

	private final CsdlEdmProvider baseProvider;
	private List<CsdlSchema> extendedSchemata = null; // lazy

	private ODataExtendedEdmProvider(CsdlEdmProvider baseProvider) {
		this.baseProvider = baseProvider;
	}

	public static ODataExtendedEdmProvider wrap(CsdlEdmProvider baseProvider) {
		return new ODataExtendedEdmProvider(baseProvider);
	}

	@Override
	public List<CsdlSchema> getSchemas() throws ODataException {
		if (extendedSchemata == null) {
			List<CsdlSchema> baseSchemata = baseProvider.getSchemas();
			extendedSchemata = new ArrayList<>(baseSchemata.size() + 1);
			extendedSchemata.addAll(baseSchemata);
			extendedSchemata.add(customSchema);
		}
		return extendedSchemata;
	}

	@Override
	public CsdlEnumType getEnumType(FullQualifiedName enumTypeName) throws ODataException {
		return baseProvider.getEnumType(enumTypeName);
	}

	@Override
	public CsdlTypeDefinition getTypeDefinition(FullQualifiedName typeDefinitionName) throws ODataException {
		return baseProvider.getTypeDefinition(typeDefinitionName);
	}

	@Override
	public CsdlEntityType getEntityType(FullQualifiedName entityTypeName) throws ODataException {
		return baseProvider.getEntityType(entityTypeName);
	}

	@Override
	public CsdlComplexType getComplexType(FullQualifiedName complexTypeName) throws ODataException {
		return baseProvider.getComplexType(complexTypeName);
	}

	@Override
	public List<CsdlAction> getActions(FullQualifiedName actionName) throws ODataException {
		return baseProvider.getActions(actionName);
	}

	@Override
	public List<CsdlFunction> getFunctions(FullQualifiedName functionName) throws ODataException {
		List<CsdlSchema> csdlSchemas = getSchemas();
		for (CsdlSchema csdlSchema : csdlSchemas) {
			if (Objects.equals(csdlSchema.getNamespace(), functionName.getNamespace()) ||
					Objects.equals(csdlSchema.getAlias(), functionName.getNamespace())) {
				return csdlSchema.getFunctions(functionName.getName());
			}
		}
		return null;
	}

	@Override
	public CsdlTerm getTerm(FullQualifiedName termName) throws ODataException {
		return baseProvider.getTerm(termName);
	}

	@Override
	public CsdlEntitySet getEntitySet(FullQualifiedName entityContainer, String entitySetName) throws ODataException {
		return baseProvider.getEntitySet(entityContainer, entitySetName);
	}

	@Override
	public CsdlSingleton getSingleton(FullQualifiedName entityContainer, String singletonName) throws ODataException {
		return baseProvider.getSingleton(entityContainer, singletonName);
	}

	@Override
	public CsdlActionImport getActionImport(FullQualifiedName entityContainer, String actionImportName)
			throws ODataException {
		return baseProvider.getActionImport(entityContainer, actionImportName);
	}

	@Override
	public CsdlFunctionImport getFunctionImport(FullQualifiedName entityContainer, String functionImportName)
			throws ODataException {
		return baseProvider.getFunctionImport(entityContainer, functionImportName);
	}

	@Override
	public CsdlEntityContainerInfo getEntityContainerInfo(FullQualifiedName entityContainerName) throws ODataException {
		return baseProvider.getEntityContainerInfo(entityContainerName);
	}

	@Override
	public List<CsdlAliasInfo> getAliasInfos() throws ODataException {
		return baseProvider.getAliasInfos();
	}

	@Override
	public CsdlEntityContainer getEntityContainer() throws ODataException {
		return baseProvider.getEntityContainer();
	}

	@Override
	public CsdlAnnotations getAnnotationsGroup(FullQualifiedName targetName, String qualifier) throws ODataException {
		return baseProvider.getAnnotationsGroup(targetName, qualifier);
	}
}
