/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.processors.request;

import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.olingo.commons.api.data.ComplexValue;
import org.apache.olingo.commons.api.data.DeletedEntity;
import org.apache.olingo.commons.api.data.Delta;
import org.apache.olingo.commons.api.data.Entity;
import org.apache.olingo.commons.api.data.EntityCollection;
import org.apache.olingo.commons.api.data.Link;
import org.apache.olingo.commons.api.data.Property;
import org.apache.olingo.commons.api.data.Valuable;
import org.apache.olingo.commons.api.edm.EdmEntityType;
import org.apache.olingo.commons.api.edm.EdmNavigationProperty;
import org.apache.olingo.commons.api.edm.FullQualifiedName;

import com.sap.cds.CdsData;
import com.sap.cds.CdsList;
import com.sap.cds.adapter.odata.v4.CdsRequestGlobals;
import com.sap.cds.adapter.odata.v4.utils.TypeConverterUtils;
import com.sap.cds.services.ErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

/**
 * Turns EDMX entities or properties into Maps or Objects, according to the EDMX-defined structure.
 * Types are already mapped to the one expected by CDS by leveraging {@link TypeConverterUtils}.
 */
public class PayloadProcessor {

	private final CdsRequestGlobals globals;

	public PayloadProcessor(CdsRequestGlobals globals) {
		this.globals = globals;
	}

	/**
	 * This method takes the output of Olingos deserializer and convert it to the payload representation understood by CDS4j (Map)
	 * @param entity the entity
	 * @return Map
	 */
	public Map<String, Object> getEntity(Entity entity) {
		Map<String, Object> propertyMap = new HashMap<>();

		if (entity != null) {
			EdmEntityType entityType = null;
			// Fetch properties of entities
			for (Property property : entity.getProperties()) {
				propertyMap.put(property.getName(), getValue(property));
			}

			// Fetch inline properties of entity
			for (Link link : entity.getNavigationLinks()) {
				String key = link.getTitle();
				if (link.getInlineEntity() != null) {
					propertyMap.put(key, getEntity(link.getInlineEntity()));

				} else if (link.getInlineEntitySet() instanceof Delta) {
					// Deep Update Delta handling
					CdsList<CdsData> deltaList = CdsList.<CdsData>create().delta();
					propertyMap.put(key, deltaList);
					Delta delta = (Delta) link.getInlineEntitySet();
					for (Entity upsertedEntity : delta.getEntities()) {
						deltaList.add(CdsData.create(getEntity(upsertedEntity)));
					}
					entityType = entityType != null ? entityType :
							globals.getServiceMetadata().getEdm().getEntityType(new FullQualifiedName(entity.getType()));
					EdmEntityType targetType = !delta.getDeletedEntities().isEmpty()
							? entityType.getNavigationProperty(link.getTitle()).getType()
							: null;
					for (DeletedEntity deletedEntity : delta.getDeletedEntities()) {
						Map<String, Object> removedEntity = getEntity(deletedEntity);
						URI uri = deletedEntity.getId();
						if (uri != null) {
							String path = deletedEntity.getId().getPath();
							Map<String, Object> keyValues = extractProperties(targetType, path);
							removedEntity.putAll(keyValues);
						}
						deltaList.add(CdsData.create(removedEntity).forRemoval());
					}
				} else if (link.getInlineEntitySet() != null) {
					List<Map<String, Object>> entityCollection = new ArrayList<>();
					for (Entity collectionEntity : link.getInlineEntitySet().getEntities()) {
						entityCollection.add(getEntity(collectionEntity));
					}
					propertyMap.put(key, entityCollection);
				} else {
					propertyMap.put(key, null);
				}
			}

			for (Link bindingLink : entity.getNavigationBindings()) {
				entityType = entityType != null ? entityType :
						globals.getServiceMetadata().getEdm().getEntityType(new FullQualifiedName(entity.getType()));
				EdmNavigationProperty navigationProperty = entityType.getNavigationProperty(bindingLink.getTitle());
				if(navigationProperty.isCollection()) {
					throw new ErrorStatusException(ErrorStatuses.NOT_IMPLEMENTED);
				}

				String link = bindingLink.getBindingLink(); // TODO toMany
				Map<String, Object> refProperties = extractProperties(navigationProperty.getType(), link);
				propertyMap.put(navigationProperty.getName(), refProperties);
			}
		}

		return propertyMap;
	}

	private static Map<String, Object> extractProperties(EdmEntityType entityType, String link) {
		Map<String, Object> refProperties = new HashMap<>();
		String entityId = link.substring(link.indexOf('(') + 1, link.lastIndexOf(')'));
		for (String keyValuePair : entityId.split(",")) {
			String[] pair = keyValuePair.split("=");
			String key;
			String value;
			if (pair.length == 1) {
				key = entityType.getKeyPredicateNames().get(0);
				value = pair[0];
			} else {
				key = pair[0];
				value = pair[1];
			}
			refProperties.put(key, TypeConverterUtils.convertToType(entityType.getKeyPropertyRef(key).getProperty().getType(), value));
		}
		return refProperties;
	}

	@SuppressWarnings("unchecked")
	public Object getValue(Valuable property) {
		Object value = property.getValue();
		if(value == null) {
			return null;
		}

		if (property.isCollection()) {
			List<Object> collection = new ArrayList<>();

			if (value instanceof EntityCollection entityCollection) {
				for (Entity collectionEntity : entityCollection.getEntities()) {
					collection.add(getEntity(collectionEntity));
				}

			} else if (property.isComplex()) {
				for(ComplexValue complexValue : (List<ComplexValue>) property.asCollection()) {
					Map<String, Object> complex = new HashMap<>();
					for(Property complexProperty : complexValue.getValue()) {
						complex.put(complexProperty.getName(), getValue(complexProperty));
					}
					collection.add(complex);
				}
			} else if (property.isPrimitive()) {
				for(Object primitiveValue : property.asCollection()) {
					collection.add(TypeConverterUtils.getValueBasedOnTypeOfRequestPayload(property.getType(), primitiveValue));
				}

			}

			return collection;
		} else {
			if(value instanceof Entity entity) {
				return getEntity(entity);

			} else if (property.isComplex()) {
				Map<String, Object> complex = new HashMap<>();
				for(Property complexProperty : property.asComplex().getValue()) {
					complex.put(complexProperty.getName(), getValue(complexProperty));
				}
				return complex;

			} else if (property.isPrimitive()) {
				return TypeConverterUtils.getValueBasedOnTypeOfRequestPayload(property.getType(), property.getValue());
			} else {
				return null;
			}
		}
	}
}
