/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.processors.response;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.lang3.StringUtils;
import org.apache.olingo.commons.api.Constants;
import org.apache.olingo.commons.api.data.Annotation;
import org.apache.olingo.commons.api.data.ComplexValue;
import org.apache.olingo.commons.api.data.Entity;
import org.apache.olingo.commons.api.data.EntityCollection;
import org.apache.olingo.commons.api.data.Link;
import org.apache.olingo.commons.api.data.Property;
import org.apache.olingo.commons.api.data.ValueType;
import org.apache.olingo.commons.api.edm.EdmBindingTarget;
import org.apache.olingo.commons.api.edm.EdmComplexType;
import org.apache.olingo.commons.api.edm.EdmEntityType;
import org.apache.olingo.commons.api.edm.EdmKeyPropertyRef;
import org.apache.olingo.commons.api.edm.EdmNavigationProperty;
import org.apache.olingo.commons.api.edm.EdmPrimitiveType;
import org.apache.olingo.commons.api.edm.EdmProperty;
import org.apache.olingo.commons.api.edm.EdmType;
import org.apache.olingo.commons.api.edm.constants.EdmTypeKind;
import org.apache.olingo.commons.api.edm.constants.ODataServiceVersion;
import org.apache.olingo.commons.core.Encoder;
import org.apache.olingo.commons.core.edm.primitivetype.AbstractGeospatialType;
import org.apache.olingo.commons.core.edm.primitivetype.EdmDateTimeOffset;
import org.apache.olingo.commons.core.edm.primitivetype.EdmStream;
import org.apache.olingo.commons.core.edm.primitivetype.EdmString;
import org.apache.olingo.commons.core.edm.primitivetype.EdmTimeOfDay;
import org.apache.olingo.server.api.uri.UriResource;
import org.apache.olingo.server.api.uri.UriResourceAction;
import org.apache.olingo.server.api.uri.UriResourceComplexProperty;
import org.apache.olingo.server.api.uri.UriResourceFunction;
import org.apache.olingo.server.api.uri.UriResourceKind;
import org.apache.olingo.server.api.uri.UriResourcePrimitiveProperty;
import org.apache.olingo.server.core.deserializer.helper.ExpandTreeBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.Result;
import com.sap.cds.Row;
import com.sap.cds.adapter.odata.v4.CdsRequestGlobals;
import com.sap.cds.adapter.odata.v4.processors.request.CdsODataRequest;
import com.sap.cds.adapter.odata.v4.utils.ETagHelper;
import com.sap.cds.adapter.odata.v4.utils.EdmUtils;
import com.sap.cds.adapter.odata.v4.utils.ODataUtils;
import com.sap.cds.adapter.odata.v4.utils.StreamUtils;
import com.sap.cds.adapter.odata.v4.utils.TypeConverterUtils;
import com.sap.cds.adapter.odata.v4.utils.mapper.EdmxFlavourMapper;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.ErrorStatuses;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ETagUtils;
import com.sap.cds.services.utils.ErrorStatusException;

public class ResultSetProcessor {

	final static Logger logger = LoggerFactory.getLogger(ResultSetProcessor.class);

	private final CdsRequestGlobals globals;
	private final EdmxFlavourMapper toEdmxMapper;
	private final EdmxFlavourMapper toCsnMapper;
	private final EdmUtils edmUtils;

	public ResultSetProcessor(CdsRequestGlobals globals) {
		this.globals = globals;
		this.toEdmxMapper = EdmxFlavourMapper.create(globals.getEdmxFlavour(), false);
		this.toCsnMapper = EdmxFlavourMapper.create(globals.getEdmxFlavour(), true);
		this.edmUtils = new EdmUtils(globals);
	}

	/*
	 * ENTITIES
	 */

	public Entity toEntity(CdsODataRequest request, Result result, ExpandTreeBuilder expand) {
		if(result.first().isEmpty()) {
			return null;
		}
		EdmEntityType type = checkEntityType(request.getResponseType());
		EdmBindingTarget bindingTarget = edmUtils.getEdmBindingTarget(type);
		String eTagProperty = getETagProperty(type);

		return toEntity(bindingTarget, type, eTagProperty, result.single(), request, expand);
	}

	public EntityCollection toEntityCollection(CdsODataRequest request, Result result, ExpandTreeBuilder expand) {

		EdmEntityType type = checkEntityType(request.getResponseType());
		EdmBindingTarget bindingTarget = edmUtils.getEdmBindingTarget(type);
		EntityCollection entitySet = new EntityCollection();
		toEntities(bindingTarget, type, result.list(), request, expand, entitySet.getEntities()::add);
		return entitySet;
	}

	private void toEntities(EdmBindingTarget bindingTarget, EdmEntityType type, List<?> result, CdsODataRequest request, ExpandTreeBuilder expand, Consumer<Entity> consumer) {

		String eTagProperty = getETagProperty(type);
		result.stream().map(row -> toEntity(bindingTarget, type, eTagProperty, (Map<?, ?>) row, request, expand)).forEach(consumer);
	}

	private Entity toEntity(EdmBindingTarget bindingTarget, EdmEntityType type, String eTagProperty, Map<?, ?> entityRow, CdsODataRequest request, ExpandTreeBuilder expand) {
		Entity entity = new Entity();
		entity.setType(type.getFullQualifiedName().getFullQualifiedNameAsString());

		Map<?, ?> applyProperties = null;
		if (request.hasApply()) {
			applyProperties = new HashMap<>(entityRow);
		}

		// TODO this actually needs to build the correct link if no binding target is available
		if(bindingTarget != null) {
			try {
				entity.setId(new URI(bindingTarget.getName() + getEntityId(type, entityRow)));
			} catch (URISyntaxException e) {
				throw new ErrorStatusException(ErrorStatuses.SERVER_ERROR, e);
			}
		}

		// entity properties
		List<String> propertyNames = type.getPropertyNames();
		for (String propertyName : propertyNames) {
			EdmProperty edmProperty = (EdmProperty) type.getProperty(propertyName);
			Object value = entityRow.get(propertyName);
			if (value != null || entityRow.containsKey(propertyName) || edmProperty.getType() == EdmStream.getInstance()) {
				if (applyProperties != null) {
					applyProperties.remove(propertyName);
				}
				Property property = consumeProperty(edmProperty, value, request, expand);
				if (property.getType().equals("Edm.Stream")) {
					boolean isV40 = ODataServiceVersion.V40.toString().equals(ODataUtils.getODataVersion(request.getODataRequest()));
					boolean exists = property.getValue() != null;
					// don't represent stream value in entity responses
					property.setValue(property.getValueType(), null);
					// add [odata.]mediaContentType annotation
					Annotation mediaContentType = new Annotation();
					mediaContentType.setTerm((isV40 ? "odata." : "") + "mediaContentType");
					mediaContentType.setType("String");
					mediaContentType.setValue(ValueType.PRIMITIVE, exists ? getContentType(type, propertyName, entityRow) : null);
					property.getAnnotations().add(mediaContentType);
				}
				entity.addProperty(property);
				if(Objects.equals(propertyName, eTagProperty) && property.getValue() != null) {
					String etag = ETagUtils.createETagHeaderValue(property.getValue().toString()); // TODO check type handling
					entity.setETag(etag);
				}
			}
		}

		// navigation properties
		List<String> navigationPropertyNames = type.getNavigationPropertyNames();
		for (String navigationPropertyName : navigationPropertyNames) {
			if (applyProperties != null) {
				applyProperties.remove(navigationPropertyName);
			}
			Object navigationValue = entityRow.get(navigationPropertyName);
			Link link;
			if (navigationValue != null) {
				EdmNavigationProperty navigationProperty = type.getNavigationProperty(navigationPropertyName);
				link = createLink(navigationProperty, navigationValue, request, expand);
			} else {
				link = new Link();
				link.setTitle(navigationPropertyName);
			}

			if(entity.getId() != null) {
				link.setHref(entity.getId().toASCIIString() + "/" + navigationPropertyName);
			}
			entity.getNavigationLinks().add(link);
		}

		if (applyProperties != null) {
			applyProperties.forEach((key, value) -> {
				Property property = new Property();
				property.setName(key.toString());
				property.setValue(ValueType.PRIMITIVE, value);
				entity.getProperties().add(property);
			});
		}

		return entity;
	}

	private String getETagProperty(EdmEntityType edmEntityType) {
		String edmEntityName = edmEntityType.getFullQualifiedName().getFullQualifiedNameAsString();
		String cdsEntityName = globals.getCdsEntityNames().getOrDefault(edmEntityName, edmEntityName);
		Optional<CdsEntity> cdsEntity = globals.getModel().findEntity(cdsEntityName);
		if(cdsEntity.isPresent()) {
			String eTagElement = ETagHelper.getETagElementName(cdsEntity.get());
			if(eTagElement != null) {
				return toEdmxMapper.remap(eTagElement, cdsEntity.get());
			}
		}
		return null;
	}

	private String getContentType(EdmEntityType edmEntityType, String edmStreamProperty, Map<?, ?> entityRow) {
		String edmEntityName = edmEntityType.getFullQualifiedName().getFullQualifiedNameAsString();
		String cdsEntityName = globals.getCdsEntityNames().getOrDefault(edmEntityName, edmEntityName);
		CdsEntity cdsEntity = globals.getModel().findEntity(cdsEntityName).orElse(null);
		if(cdsEntity != null) {
			String cdsStreamProperty = toCsnMapper.remap(edmStreamProperty, cdsEntity);
			String mediaTypeElement = StreamUtils.getCoreMediaTypeElement(cdsEntity, cdsStreamProperty);
			if (!StringUtils.isEmpty(mediaTypeElement)) {
				return (String) entityRow.get(toEdmxMapper.remap(mediaTypeElement, cdsEntity));
			} else {
				return StreamUtils.getCoreMediaTypeValue(cdsEntity, cdsStreamProperty);
			}
		}
		return null;
	}

	public static String getEntityId(EdmEntityType type, Map<?, ?> row) {
		StringBuilder result = new StringBuilder();
		List<String> keyNames = type.getKeyPredicateNames();
		boolean first = true;
		for (String keyName : keyNames) {
			EdmKeyPropertyRef keyRef = type.getKeyPropertyRef(keyName);
			if (first) {
				result.append('(');
				first = false;
			} else {
				result.append(',');
			}
			if (keyNames.size() > 1) {
				result.append(Encoder.encode(keyName)).append('=');
			}

			EdmType propertyType = keyRef.getProperty().getType();
			String keyValue;
			if (propertyType instanceof EdmString ||
				propertyType instanceof EdmDateTimeOffset ||
				propertyType instanceof EdmTimeOfDay ||
				propertyType instanceof AbstractGeospatialType
			) {
				keyValue = Encoder.encode(String.valueOf(row.get(keyName)));
			} else {
				keyValue = String.valueOf(row.get(keyName));
			}
			if (propertyType instanceof EdmString) {
				result.append("'").append(keyValue).append("'");
			} else {
				result.append(keyValue);
			}
		}
		if(result.length() > 0) {
			result.append(')');
		}
		return result.toString();
	}

	private Property consumeProperty(EdmProperty edmProperty, Object propertyValue, CdsODataRequest request, ExpandTreeBuilder expand) {
		return consumeProperty(edmProperty.getName(), edmProperty.getType(), edmProperty.isCollection(), propertyValue, request, expand);
	}

	private Property consumeProperty(String propertyName, EdmType propertyType, boolean isCollection, Object propertyValue, CdsODataRequest request, ExpandTreeBuilder expand) {
		Property property = new Property();
		property.setName(propertyName);
		property.setType(propertyType.getFullQualifiedName().getFullQualifiedNameAsString());

		Object convertedValue;
		ValueType valueType;
		switch (propertyType.getKind()) {
		case PRIMITIVE:
		case DEFINITION:
		case ENUM:
			if(isCollection) {
				Stream<?> stream;
				if(propertyValue instanceof List<?> list) {
					stream = list.stream();
				} else {
					// also support single values
					stream = Stream.of(propertyValue);
				}
				convertedValue = stream.map(v -> TypeConverterUtils.getValueBasedOnTypeOfResultSet(propertyType, v)).collect(Collectors.toList());
				valueType = propertyType.getKind() == EdmTypeKind.ENUM ? ValueType.COLLECTION_ENUM : ValueType.COLLECTION_PRIMITIVE;
			} else {
				convertedValue = TypeConverterUtils.getValueBasedOnTypeOfResultSet(propertyType, propertyValue);
				valueType = propertyType.getKind() == EdmTypeKind.ENUM ? ValueType.ENUM : ValueType.PRIMITIVE;
			}
			break;
		case COMPLEX:
			if(isCollection) {
				Stream<?> stream;
				if(propertyValue instanceof List<?> list) {
					stream = list.stream();
				} else {
					// also support single values
					stream = Stream.of(propertyValue);
				}
				convertedValue = stream.map(v -> consumeComplexValue((EdmComplexType) propertyType, v, request, expand)).collect(Collectors.toList());
				valueType = ValueType.COLLECTION_COMPLEX;
			} else {
				convertedValue = consumeComplexValue((EdmComplexType) propertyType, propertyValue, request, expand);
				valueType = ValueType.COMPLEX;
			}
			break;
		default:
			throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_EDM_TYPE, propertyType.getKind());
		}

		property.setValue(valueType, convertedValue);
		return property;
	}

	private Link createLink(EdmNavigationProperty navigationProperty, Object navigationValue, CdsODataRequest request, ExpandTreeBuilder expand) {
		Link link = new Link();
		link.setTitle(navigationProperty.getName());

		ExpandTreeBuilder childExpand = (expand != null) ? expand.expand(navigationProperty) : null;
		EdmEntityType edmEntityType = navigationProperty.getType();
		EdmBindingTarget bindingTarget = edmUtils.getEdmBindingTarget(edmEntityType);

		if (navigationProperty.isCollection()) {
			List<?> navigationValueList;
			if (navigationValue instanceof List<?> list) {
				navigationValueList = list;
			} else {
				// also support single values
				navigationValueList = Arrays.asList(navigationValue);
			}

			EntityCollection inlineEntitySet = new EntityCollection();
			toEntities(bindingTarget, edmEntityType, navigationValueList, request, childExpand, inlineEntitySet.getEntities()::add);
			link.setType(Constants.ENTITY_SET_NAVIGATION_LINK_TYPE);
			link.setInlineEntitySet(inlineEntitySet);
		} else {
			String eTagProperty = getETagProperty(edmEntityType);
			Entity inlineEntity = toEntity(bindingTarget, edmEntityType, eTagProperty, (Map<?, ?>) navigationValue, request, childExpand);
			link.setType(Constants.ENTITY_NAVIGATION_LINK_TYPE);
			link.setInlineEntity(inlineEntity);
		}
		return link;
	}

	private ComplexValue consumeComplexValue(EdmComplexType type, Object value, CdsODataRequest request, ExpandTreeBuilder expand) {
		ComplexValue complexValue = new ComplexValue();
		complexValue.setTypeName(type.getFullQualifiedName().getFullQualifiedNameAsString());
		Map<?, ?> valueMap = (Map<?, ?>) value;

		// properties
		for (String propertyName : type.getPropertyNames()) {
			Object propertyValue = valueMap.get(propertyName);
			if (propertyValue != null) {
				EdmProperty edmProperty = (EdmProperty) type.getProperty(propertyName);
				Property property = consumeProperty(edmProperty, propertyValue, request, expand);
				complexValue.getValue().add(property);
			}
		}

		// navigation properties
		for (String navigationPropertyName : type.getNavigationPropertyNames()) {
			Object navigationValue = valueMap.get(navigationPropertyName);
			if (navigationValue != null) {
				EdmNavigationProperty navigationProperty = type.getNavigationProperty(navigationPropertyName);
				Link link = createLink(navigationProperty, navigationValue, request, expand);
				complexValue.getNavigationLinks().add(link);
			}
		}
		return complexValue;
	}

	private EdmEntityType checkEntityType(EdmType type) {
		if(type.getKind() != EdmTypeKind.ENTITY) {
			throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_EDM_TYPE, type.getKind());
		}
		return (EdmEntityType) type;
	}

	/*
	 * COMPLEX
	 */

	public Property toComplex(CdsODataRequest request, Result result, ExpandTreeBuilder expand) {
		EdmComplexType type = checkComplexType(request.getResponseType());
		String name = retrieveComplexName(request.getLastTypedResource());

		return consumeProperty(name, type, false, result.single(), request, expand);
	}

	public Property toComplexCollection(CdsODataRequest request, Result result, ExpandTreeBuilder expand) {
		EdmComplexType type = checkComplexType(request.getResponseType());
		UriResource resource = request.getLastTypedResource();
		String name = retrieveComplexName(resource);
		List<?> results;
		if (resource.getKind().equals(UriResourceKind.complexProperty)) {
			Row row = result.single();
			results = (List<?>) row.get(name);
		} else {
			results = result.list();
		}
		return consumeProperty(name, type, true, results, request, expand);
	}

	private EdmComplexType checkComplexType(EdmType type) {
		if(type.getKind() != EdmTypeKind.COMPLEX) {
			throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_EDM_TYPE, type.getKind());
		}
		return (EdmComplexType) type;
	}

	private String retrieveComplexName(UriResource resource) {
		String name;
		switch(resource.getKind()) {
		case complexProperty:
			name = ((UriResourceComplexProperty) resource).getProperty().getName();
			break;
		case action:
			name = ((UriResourceAction) resource).getAction().getName();
			break;
		case function:
			name = ((UriResourceFunction) resource).getFunction().getName();
			break;
		default:
			throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_URI_RESOURCE, resource.getKind());
		}
		return name;
	}

	/*
	 * PRIMITIVES
	 */

	public Property toPrimitive(CdsODataRequest request, Result result) {
		EdmPrimitiveType type = checkPrimitiveType(request.getResponseType());
		String name = retrievePrimitiveName(request.getLastTypedResource());
		Object primitive = result.single().get(name);

		return consumeProperty(name, type, false, primitive, request, null);
	}

	public Property toPrimitiveCollection(CdsODataRequest request, Result result) {
		EdmPrimitiveType type = checkPrimitiveType(request.getResponseType());
		String name = retrievePrimitiveName(request.getLastTypedResource());
		Object primitive = result.single().get(name);

		return consumeProperty(name, type, true, primitive, request, null);
	}

	private EdmPrimitiveType checkPrimitiveType(EdmType type) {
		if(type.getKind() != EdmTypeKind.PRIMITIVE) {
			throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_EDM_TYPE, type.getKind());
		}
		return (EdmPrimitiveType) type;
	}

	private String retrievePrimitiveName(UriResource resource) {
		String name;
		switch(resource.getKind()) {
		case primitiveProperty:
			name = ((UriResourcePrimitiveProperty) resource).getProperty().getName();
			break;
		case action:
			name = ((UriResourceAction) resource).getAction().getName();
			break;
		case function:
			name = ((UriResourceFunction) resource).getFunction().getName();
			break;
		default:
			throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_URI_RESOURCE, resource.getKind());
		}
		return name;
	}

	/*
	 * COUNT
	 */

	public static int toCount(Result result) {
		int count = 0;
		for (Map<String, Object> row : result.list()) {
			count += Integer.parseInt(Optional.ofNullable(row.get("count"))
					.orElseThrow(() -> new ErrorStatusException(CdsErrorStatuses.MISSING_VALUE_FOR_COUNT)).toString());
		}
		return count;
	}
}
