/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.serializer.json;

import java.io.IOException;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.fasterxml.jackson.core.JsonGenerator;
import com.sap.cds.adapter.odata.v4.serializer.json.api.Data2Json;
import com.sap.cds.adapter.odata.v4.serializer.json.primitive.String2Json;
import com.sap.cds.impl.parser.token.Jsonizer;

public abstract class Struct2Json implements Data2Json<Map<String, Object>> {

	// prepared serializers and options:
	String2Json<Void> context;
	String2Json<Void> metadataEtag;
	String2Json<Map<String, Object>> etag;
	String2Json<Void> type;
	List<Data2Json<Map<String, Object>>> primitiveProperties;
	List<Data2Json<Map<String, Object>>> navigationLinkProperties;
	List<Data2Json<Map<String, Object>>> navigationProperties;
	// Contains a subtree for dynamic expands
	List<Data2Json<Map<String, Object>>> dynamicProperties;
	// Set of dynamic properties that shouldn't be serialized
	Set<String> excludedProperties;
	boolean isOpenType;

	boolean isRootLevel = true;
	boolean isAutoExpand = false;

	Struct2JsonBuilder builder;

	Struct2Json() {
	}

	public String2Json<Void> getContext() {
		return context;
	}

	public String2Json<Void> getMetadataEtag() {
		return metadataEtag;
	}

	protected void writeProperties(Map<String, Object> row, JsonGenerator json) throws IOException {
		// Collect all property names from data payload to be serialized
		Set<String> rowProperties = isOpenType || isAutoExpand ? new HashSet<>(row.keySet()) : null;

		if (primitiveProperties != null) {
			for (Data2Json<Map<String, Object>> p : primitiveProperties) {
				p.toJson(row, json);
				consumeProperty(rowProperties, p.getName());
			}
		}

		if (navigationLinkProperties != null) {
			for (Data2Json<Map<String, Object>> n : navigationLinkProperties) {
				n.toJson(row, json);
			}
		}

		if (navigationProperties != null) {
			for (Data2Json<Map<String, Object>> n : navigationProperties) {
				n.toJson(row, json);
				consumeProperty(rowProperties, n.getName());
			}
		}
		handleDynamicProperties(row, rowProperties, json);
	}

	private void handleDynamicProperties(Map<String, Object> row, Set<String> properties, JsonGenerator json)
			throws IOException {
		if (isOpenType) {
			for (String p : properties) {
				Object pojo = row.get(p);
				json.writeFieldName(p);
				if (pojo == null) {
					json.writeNull();
				} else {
					json.writeRawValue(Jsonizer.json(pojo));
				}
			}
		} else if (properties != null) {
			// Lazy load navigation properties only, the rest (Draft_UUID, ...) is ignored
			if (excludedProperties != null) {
				properties.removeAll(excludedProperties);
			}
			if (properties.size() > 0 && builder != null) {
				builder.addDynamicExpandProperties(this, properties);
				for (Data2Json<Map<String, Object>> n : dynamicProperties) {
					n.toJson(row, json);
				}
			}
		}
	}

	private void consumeProperty(Set<String> rowProperties, String propertyName) {
		if (isAutoExpand) {
			rowProperties.remove(propertyName);
		}
	}
}
