/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.utils.mapper;

import java.util.Arrays;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Stream;

import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsStructuredType;

/**
 * Maps data which is in an EDMX-defined structure to CSN-defined structure and vice versa.
 */
public interface EdmxFlavourMapper {

	/**
	 * Describes the EDMX flavour used together with Universal CSN. The special value {@link #NOOP}
	 * indicates, that CSN is not universal, but simply matches the EDMX flavour.
	 */
	public static enum EdmxFlavour {
		NOOP, V4;
	}

	public static EdmxFlavourMapper create(EdmxFlavour flavour, boolean toCsn) {
		switch (flavour) {
			case V4:
				return new V4EdmxFlavourMapper(toCsn);
			default:
				return new NoopEdmxFlavourMapper();
		}
	}

	/**
	 * Remaps a simple element name of a given structured type
	 *
	 * @param element the element name
	 * @param type the structured type
	 * @return the mapped element name
	 */
	public String remap(String element, CdsStructuredType type);

	/**
	 * Remaps a data map of a given structured type
	 *
	 * @param <T> the map type
	 * @param entry the data map
	 * @param entryType the structured type of the entry
	 * @return the mapped data map
	 */
	public default <T extends Map<String, Object>> T remap(T entry, CdsStructuredType entryType) {
		return remap(Arrays.asList(entry), entryType).get(0);
	}

	/**
	 * Remaps an Iterable of data maps of a given structured type
	 *
	 * @param <T> the Iterable type
	 * @param entries an Iterable of data maps
	 * @param entryType the structured type of the entries
	 * @return the mapped Iterable of data maps
	 */
	public default <T extends Iterable<? extends Map<String, Object>>> T remap(T entries,
			CdsStructuredType entryType) {
		return remap(entries, entryType, (path) -> true);
	}

	/**
	 * Remaps an Iterable of data maps of a given structured type
	 *
	 * @param <T> the Iterable type
	 * @param entries an Iterable of data maps
	 * @param entryType the structured type of the entries
	 * @param isExpanded a function to determine if the given path is expanded
	 * @return the mapped Iterable of data maps
	 */
	public <T extends Iterable<? extends Map<String, Object>>> T remap(T entries,
			CdsStructuredType entryType, Function<String, Boolean> isExpanded);

	/**
	 * A stream of {@link Mapping} instances for each element of a {@link CdsStructuredType}.
	 *
	 * @param type the {@link CdsStructuredType}
	 * @return the stream of all {@link Mapping} instances
	 */
	public Stream<? extends Mapping> createMappings(CdsStructuredType type);

	/**
	 * The {@link CdsElement} with it's name in the EDMX and CSN.
	 */
	public interface Mapping {

		/**
		 * @return the {@link CdsElement} associated with the mapped names
		 */
		public CdsElement getTargetElement();

		/**
		 * @return the originating root {@link CdsElement} from which the target element was derived or the target element if it was not derived.
		 */
		public CdsElement getRootElement();

		/**
		 * @return the EDMX name
		 */
		public String getEdmxName();

		/**
		 * @return the CSN name
		 */
		public String getCsnName();

	}

}
