/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.metadata.cds;

import java.util.function.Function;

import org.apache.olingo.commons.api.edm.FullQualifiedName;

import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsDefinition;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsOperation;
import com.sap.cds.reflect.CdsParameter;
import com.sap.cds.reflect.CdsService;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.draft.DraftAdministrativeData;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.services.utils.ODataUtils;

class CdsServiceEdmUtils {

	private CdsServiceEdmUtils() {
		// hidden
	}

	static FullQualifiedName fqn(CdsDefinition def) {
		return new FullQualifiedName(def.getQualifier(), name(def));
	}

	static FullQualifiedName structuredTypeFqn(CdsService service, CdsStructuredType type, Function<CdsStructuredType, String> anonymousNameSupplier) {
		if (type.isAnonymous()) {
			return new FullQualifiedName(service.getQualifiedName(), anonymousNameSupplier.apply(type));
		} else if (!type.getQualifier().equals(service.getQualifiedName())) {
			return new FullQualifiedName(service.getQualifiedName(), ODataUtils.toODataName(type.getQualifiedName()));
		}
		return fqn(type);
	}

	static String name(CdsDefinition def) {
		return ODataUtils.toODataName(def.getName());
	}

	static String nameElement(CdsDefinition def, String parentName, String elementName) {
		return parentName + "_" + elementName;
	}

	static String nameParameter(CdsDefinition def, CdsOperation operation, CdsParameter parameter) {
		return "param_" + ODataUtils.toODataName(operation.getQualifiedName()) + "_" + parameter.getName();
	}

	static String nameReturn(CdsDefinition def, CdsOperation operation) {
		return "return_" + ODataUtils.toODataName(operation.getQualifiedName());
	}

	static CdsEntity target(CdsStructuredType entity, CdsElement element) {
		if (element.getName().equals(Drafts.SIBLING_ENTITY)) {
			return (CdsEntity) entity; // sibling entity is self link in EDMX
		}
		return element.getType().as(CdsAssociationType.class).getTarget();
	}

	static boolean isParameterized(CdsEntity entity) {
		return entity.params().count() > 0;
	}

	static boolean isIncluded(CdsEntity entity) {
		return !(DraftUtils.isDraftEnabled(entity) && entity.getName().endsWith("_drafts"));
	}

	static boolean isIncluded(CdsElement element) {
		return !(element.getAnnotationValue("cds.api.ignore", false) || element.getName().equals(Drafts.DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID));
	}

	static boolean isIncludedEntitySet(CdsEntity entity) {
		return isIncluded(entity) && !entity.getName().equals(DraftAdministrativeData.ENTITY_NAME);
	}

	static boolean isIncludedEntitySetAssociation(CdsElement element) {
		return element.getType().isAssociation() && !element.getName().equals(Drafts.DRAFT_ADMINISTRATIVE_DATA);
	}

}
