/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.processors;

import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.io.IOUtils;
import org.apache.olingo.commons.api.format.ContentType;
import org.apache.olingo.commons.api.http.HttpHeader;
import org.apache.olingo.commons.api.http.HttpStatusCode;
import org.apache.olingo.server.api.OData;
import org.apache.olingo.server.api.ODataApplicationException;
import org.apache.olingo.server.api.ODataLibraryException;
import org.apache.olingo.server.api.ODataRequest;
import org.apache.olingo.server.api.ODataResponse;
import org.apache.olingo.server.api.ODataServerError;
import org.apache.olingo.server.api.ServiceMetadata;
import org.apache.olingo.server.api.batch.BatchFacade;
import org.apache.olingo.server.api.deserializer.batch.BatchOptions;
import org.apache.olingo.server.api.deserializer.batch.BatchRequestPart;
import org.apache.olingo.server.api.deserializer.batch.ODataResponsePart;
import org.apache.olingo.server.api.prefer.PreferencesApplied;
import org.apache.olingo.server.api.processor.ActionComplexCollectionProcessor;
import org.apache.olingo.server.api.processor.ActionComplexProcessor;
import org.apache.olingo.server.api.processor.ActionEntityCollectionProcessor;
import org.apache.olingo.server.api.processor.ActionEntityProcessor;
import org.apache.olingo.server.api.processor.ActionPrimitiveCollectionProcessor;
import org.apache.olingo.server.api.processor.ActionPrimitiveProcessor;
import org.apache.olingo.server.api.processor.ActionVoidProcessor;
import org.apache.olingo.server.api.processor.BatchProcessor;
import org.apache.olingo.server.api.processor.ComplexCollectionProcessor;
import org.apache.olingo.server.api.processor.ComplexProcessor;
import org.apache.olingo.server.api.processor.CountEntityCollectionProcessor;
import org.apache.olingo.server.api.processor.DeltaProcessor;
import org.apache.olingo.server.api.processor.EntityProcessor;
import org.apache.olingo.server.api.processor.ErrorProcessor;
import org.apache.olingo.server.api.processor.PrimitiveCollectionProcessor;
import org.apache.olingo.server.api.processor.PrimitiveValueProcessor;
import org.apache.olingo.server.api.uri.UriInfo;
import org.apache.olingo.server.api.uri.UriInfoResource;
import org.apache.olingo.server.core.ODataExceptionHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Sets;
import com.sap.cds.adapter.odata.v4.CdsRequestGlobals;
import com.sap.cds.adapter.odata.v4.processors.response.CdsODataResponse;
import com.sap.cds.adapter.odata.v4.utils.ODataUtils;
import com.sap.cds.services.ErrorStatus;
import com.sap.cds.services.ErrorStatuses;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OpenTelemetryUtils;
import com.sap.cds.services.utils.StringUtils;
import io.opentelemetry.api.common.AttributeKey;
import io.opentelemetry.api.trace.Span;
import io.opentelemetry.context.Scope;

/**
 * Implements all supported Olingo Processors and delegates requests to the {@link ODataProcessor}
 */
public class OlingoProcessor implements EntityProcessor, CountEntityCollectionProcessor, ActionEntityProcessor, ActionEntityCollectionProcessor,
PrimitiveValueProcessor, PrimitiveCollectionProcessor, ActionPrimitiveProcessor, ActionPrimitiveCollectionProcessor,
ComplexProcessor, ComplexCollectionProcessor, ActionComplexProcessor, ActionComplexCollectionProcessor,
ActionVoidProcessor, BatchProcessor, ErrorProcessor, DeltaProcessor {

	private final static Logger logger = LoggerFactory.getLogger(OlingoProcessor.class);
	private final static Logger accessLogger = LoggerFactory.getLogger("com.sap.cds.adapter.odata.v4.BatchAccess");

	// these headers are not propagated from the parent batch request to its child requests
	private final static Set<String> batchHeaderPropagationBlacklist = new HashSet<>(Arrays.asList(
		HttpHeader.ACCEPT, HttpHeader.ACCEPT_ENCODING, HttpHeader.ACCEPT_CHARSET,
		HttpHeader.CONTENT_TYPE, HttpHeader.CONTENT_LENGTH, HttpHeader.CONTENT_ENCODING,
		HttpHeader.CONTENT_LANGUAGE, HttpHeader.CONTENT_LOCATION, HttpHeader.CONTENT_ID,
		"Content-Transfer-Encoding", "MIME-Version",
		HttpHeader.IF_MATCH, HttpHeader.IF_NONE_MATCH
	));

	private final CdsRequestGlobals globals;
	private final AbstractODataProcessor odataProcessor;
	private final boolean fastSerializer;

	public OlingoProcessor(CdsRequestGlobals globals) {
		this.globals = globals;
		this.fastSerializer = globals.getRuntime().getEnvironment().getCdsProperties().getOdataV4().getSerializer().isEnabled();
		this.odataProcessor = fastSerializer ? new FastODataProcessor(globals) : new ODataProcessor(globals);
	}

	@Override
	public void init(OData odata, ServiceMetadata serviceMetadata) {
		// nothing to do
	}

	/*
	 * GET REQUESTS
	 */

	@Override
	public void readEntity(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType responseFormat) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processEntity(odataRequest, odataResponse, uriInfo, null, responseFormat);
	}

	@Override
	public void readEntityCollection(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType responseFormat) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processEntities(odataRequest, odataResponse, uriInfo, null, responseFormat);
	}

	@Override
	public void countEntityCollection(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo) throws ODataApplicationException, ODataLibraryException {
		handleUnsupportedOptions(uriInfo.asUriInfoResource());
		odataProcessor.processCountRequest(odataRequest, odataResponse, uriInfo);
	}

	private void handleUnsupportedOptions(final UriInfoResource uriInfo) {
		if (uriInfo.getApplyOption() != null) {
			throw new ErrorStatusException(ErrorStatuses.NOT_IMPLEMENTED);
		}
	}

	@Override
	public void readComplex(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType responseFormat) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processSingleComplex(odataRequest, odataResponse, uriInfo, null, responseFormat);
	}

	@Override
	public void readComplexCollection(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType responseFormat) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processCollectionComplex(odataRequest, odataResponse, uriInfo, responseFormat, responseFormat);
	}

	@Override
	public void readPrimitive(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType responseFormat) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processSinglePrimitive(odataRequest, odataResponse, uriInfo, null, responseFormat);
	}

	@Override
	public void readPrimitiveValue(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType responseFormat) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processSinglePrimitiveValue(odataRequest, odataResponse, uriInfo, null, responseFormat);
	}

	@Override
	public void readPrimitiveCollection(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType responseFormat) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processCollectionPrimitive(odataRequest, odataResponse, uriInfo, null, responseFormat);
	}

	/*
	 * POST REQUESTS
	 */

	@Override
	public void createEntity(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat, ContentType responseFormat) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processEntity(odataRequest, odataResponse, uriInfo, requestFormat, responseFormat);	
	}

	@Override
	public void processActionEntity(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat, ContentType responseFormat) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processEntity(odataRequest, odataResponse, uriInfo, requestFormat, responseFormat);	
	}

	@Override
	public void processActionEntityCollection(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat, ContentType responseFormat) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processEntities(odataRequest, odataResponse, uriInfo, requestFormat, responseFormat);
	}

	@Override
	public void processActionComplex(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat, ContentType responseFormat) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processSingleComplex(odataRequest, odataResponse, uriInfo, requestFormat, responseFormat);
	}

	@Override
	public void processActionComplexCollection(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat, ContentType responseFormat) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processCollectionComplex(odataRequest, odataResponse, uriInfo, responseFormat, responseFormat);
	}

	@Override
	public void processActionPrimitive(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat, ContentType responseFormat) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processSinglePrimitive(odataRequest, odataResponse, uriInfo, requestFormat, responseFormat);
	}

	@Override
	public void processActionPrimitiveCollection(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat, ContentType responseFormat) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processCollectionPrimitive(odataRequest, odataResponse, uriInfo, requestFormat, responseFormat);
	}

	@Override
	public void processActionVoid(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processNoContentRequest(odataRequest, odataResponse, uriInfo, requestFormat);
	}

	/*
	 * PUT/PATCH REQUESTS
	 */

	@Override
	public void updateEntity(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat, ContentType responseFormat) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processEntity(odataRequest, odataResponse, uriInfo, requestFormat, responseFormat);	
	}

	@Override
	public void updateEntityCollection(ODataRequest request, ODataResponse response, UriInfo uriInfo, ContentType requestFormat, ContentType responseFormat) throws ODataApplicationException, ODataLibraryException {
	    odataProcessor.processEntities(request, response, uriInfo, requestFormat, responseFormat);
	}

	@Override
	public void updateComplex(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat, ContentType responseFormat) throws ODataApplicationException, ODataLibraryException {
		throw new ErrorStatusException(ErrorStatuses.NOT_IMPLEMENTED);
	}

	@Override
	public void updateComplexCollection(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat, ContentType responseFormat) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processCollectionComplex(odataRequest, odataResponse, uriInfo, responseFormat, responseFormat);
	}

	@Override
	public void updatePrimitive(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat, ContentType responseFormat) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processSinglePrimitive(odataRequest, odataResponse, uriInfo, requestFormat, responseFormat);
	}

	@Override
	public void updatePrimitiveValue(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat, ContentType responseFormat) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processSinglePrimitiveValue(odataRequest, odataResponse, uriInfo, requestFormat, responseFormat);
	}

	@Override
	public void updatePrimitiveCollection(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo, ContentType requestFormat, ContentType responseFormat) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processCollectionPrimitive(odataRequest, odataResponse, uriInfo, requestFormat, responseFormat);
	}

	/*
	 * DELETE REQUESTS
	 */

	@Override
	public void deleteEntity(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processNoContentRequest(odataRequest, odataResponse, uriInfo, null);
	}

	@Override
	public void deleteComplex(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo) throws ODataApplicationException, ODataLibraryException {
		throw new ErrorStatusException(ErrorStatuses.NOT_IMPLEMENTED);
	}

	@Override
	public void deleteComplexCollection(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processNoContentRequest(odataRequest, odataResponse, uriInfo, null);
	}

	@Override
	public void deletePrimitive(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processNoContentRequest(odataRequest, odataResponse, uriInfo, null);
	}

	@Override
	public void deletePrimitiveValue(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processNoContentRequest(odataRequest, odataResponse, uriInfo, null);
	}

	@Override
	public void deletePrimitiveCollection(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo) throws ODataApplicationException, ODataLibraryException {
		odataProcessor.processNoContentRequest(odataRequest, odataResponse, uriInfo, null);
	}

	/*
	 * BATCH REQUESTS
	 */

	@Override
	public void processBatch(BatchFacade facade, ODataRequest odataRequest, ODataResponse odataResponse) throws ODataApplicationException, ODataLibraryException {
		final boolean continueOnError = globals.getOData().createPreferences(odataRequest.getHeaders(HttpHeader.PREFER)).hasContinueOnError();

		// Gets the boundary from content-type header. This is used for parsing the request
		final String boundary = facade.extractBoundaryFromContentType(odataRequest.getHeader(HttpHeader.CONTENT_TYPE));
		final BatchOptions options = BatchOptions.with()
				.rawBaseUri(odataRequest.getRawBaseUri())
				.rawServiceResolutionUri(odataRequest.getRawServiceResolutionUri()).build();
		//Get the batch parts
		final List<BatchRequestPart> parts = globals.getOData().createFixedFormatDeserializer().parseBatchRequest(odataRequest.getBody(), boundary, options);

		long requestSizeLimit = globals.getRuntime().getEnvironment().getCdsProperties().getOdataV4().getBatch().getMaxRequests();
		if (requestSizeLimit >= 0 && requestSizeLimit < parts.stream().mapToLong(p -> p.getRequests().size()).sum()) {
			throw new ErrorStatusException(CdsErrorStatuses.BATCH_TOO_MANY_REQUESTS);
		}

		final List<ODataResponsePart> responseParts = new ArrayList<ODataResponsePart>();

		Set<String> allowedParentHeaders = Sets.filter(odataRequest.getAllHeaders().keySet(),
				h -> batchHeaderPropagationBlacklist.stream().noneMatch(b -> h.equalsIgnoreCase(b)));
		batchLoop: for (BatchRequestPart part : parts) {
			// Propagate OData Headers to batch requests
			for (ODataRequest req : part.getRequests()) {
				for (String header : allowedParentHeaders) {
					if (StringUtils.isEmpty(req.getHeader(header))) {
						String parentHeader = odataRequest.getHeader(header);
						if (!StringUtils.isEmpty(parentHeader)) {
							req.addHeader(header, parentHeader);
						}
					}
				}
			}

			// Execute each batch request
			ODataResponsePart responsePart;
			Optional<Span> span = (part.getRequests().size() == 1) ? OpenTelemetryUtils.createSpan(OpenTelemetryUtils.CdsSpanType.ODATA_BATCH) : Optional.empty();
			span.ifPresent(s -> {
				ODataRequest request = part.getRequests().get(0);
				updateSpan(request, s);
			});
			try (Scope scope = span.map(Span::makeCurrent).orElse(null)) {
				responsePart = facade.handleBatchRequest(part);

				span.ifPresent(s -> {
					s.setAttribute(AttributeKey.longKey("http.status_code"), responsePart.getResponses().get(0).getStatusCode());
				});
			} catch (Exception e) {
				OpenTelemetryUtils.recordException(span, e);
				throw e;
			} finally {
				OpenTelemetryUtils.endSpan(span);
			}
			logBatchRequest(part, responsePart);

			responseParts.add(responsePart); // Also add failed responses.
			if(!continueOnError) {
				for(ODataResponse response : responsePart.getResponses()) {
					int statusCode = response.getStatusCode();
					if (statusCode >= 400 && statusCode <= 600) {
						break batchLoop; // Stop processing, but serialize responses to all recent requests.
					}
				}
			}

		}

		// Form the odata response combining the response parts
		final InputStream responseContent = globals.getOData().createFixedFormatSerializer().batchResponse(responseParts, boundary);
		odataResponse.setHeader(HttpHeader.CONTENT_TYPE, ContentType.MULTIPART_MIXED + ";boundary=" + boundary);
		odataResponse.setHeader(HttpHeader.ODATA_VERSION, ODataUtils.getODataVersion(odataRequest));
		odataResponse.setContent(responseContent);
		odataResponse.setStatusCode(HttpStatusCode.OK.getStatusCode());
		if (continueOnError) {
			odataResponse.setHeader(HttpHeader.PREFERENCE_APPLIED, PreferencesApplied.with().continueOnError().build().toValueString());
		}
	}

	@Override
	public ODataResponsePart processChangeSet(BatchFacade facade, List<ODataRequest> odataRequests) throws ODataApplicationException, ODataLibraryException {
		try {
			return globals.getRuntime().changeSetContext().run((context) -> {
				List<ODataResponse> responses = new ArrayList<>();
				for (ODataRequest request : odataRequests) {
					ODataResponse response;

					Optional<Span> span = OpenTelemetryUtils.createSpan(OpenTelemetryUtils.CdsSpanType.ODATA_BATCH);
					span.ifPresent(s -> {
						updateSpan(request, s);
						s.setAttribute(OpenTelemetryUtils.CDS_ODATA_IS_CHANGESET, true);
					});
					try (Scope scope = span.map(Span::makeCurrent).orElse(null)) {
						response = facade.handleODataRequest(request);

						span.ifPresent(s -> {
							s.setAttribute(OpenTelemetryUtils.HTTP_STATUS_CODE, response.getStatusCode());
						});
					} catch (ODataApplicationException | ODataLibraryException e) {
						ErrorStatusException wrappedException = new ErrorStatusException(ErrorStatuses.SERVER_ERROR, e);
						OpenTelemetryUtils.recordException(span, wrappedException);
						throw wrappedException;
					} catch (Exception e) {
						OpenTelemetryUtils.recordException(span, e);
						throw e;
					} finally {
						OpenTelemetryUtils.endSpan(span);
					}

					if(response.getStatusCode() >= 200 && response.getStatusCode() < 400) {
						// only collect success responses
						responses.add(response);
					} else {
						// return a single response with the error for the whole changeset as specified in the specification
						// therefore the error is a simple odata error batch response and not a changeset response
						return new ODataResponsePart(response, false);
					}
				}
				return new ODataResponsePart(responses, true);
			});
		} catch (Exception e) {
			ODataResponse errorResponse = new ODataResponse();
			processError(odataRequests.get(0), errorResponse, ODataExceptionHelper.createServerErrorObject(e), ContentType.APPLICATION_JSON);
			// return a single response with the error for the whole changeset as specified in the specification
			// therefore the error is a simple odata error batch response and not a changeset response
			return new ODataResponsePart(errorResponse, false);
		}

	}

	/*
	 * ERROR HANDLING
	 */

	@Override
	public void processError(ODataRequest odataRequest, ODataResponse odataResponse, ODataServerError serverError, ContentType responseFormat) {
		Locale locale = RequestContext.getCurrent(globals.getRuntime()).getParameterInfo().getLocale();
		try {
			CdsODataResponse cdsResponse;
			Exception exception = serverError.getException();
			if(exception instanceof ServiceException serviceException) {
				cdsResponse = ODataUtils.toErrorResponse(serviceException, globals);
			} else {
				ErrorStatus errorStatus = ErrorStatuses.getByCode(serverError.getStatusCode());
				if(errorStatus == null) {
					errorStatus = ErrorStatuses.SERVER_ERROR;
				}

				String message = serverError.getMessage(); // localized by Olingo
				ServiceException serviceException;
				if(StringUtils.isEmpty(message)) {
					serviceException = new ErrorStatusException(errorStatus, exception);
				} else {
					serviceException = new ServiceException(errorStatus, message, exception);
				}

				cdsResponse = ODataUtils.toErrorResponse(serviceException, globals);
			}

			// only treat unexpected exceptions as errors
			if (cdsResponse.getStatusCode() >= 500 && cdsResponse.getStatusCode() < 600) {
				if(exception != null) {
					logger.error(exception.getMessage(), exception);
				} else {
					logger.error(serverError.getMessage());
				}
			} else {
				if(exception != null) {
					logger.debug(exception.getMessage(), exception);
				} else {
					logger.error(serverError.getMessage());
				}
			}

			// TODO better remove existing headers
			odataResponse.getAllHeaders().keySet().stream().collect(Collectors.toList()).stream().forEach((k) -> odataResponse.setHeader(k, ""));
			ODataUtils.setODataErrorResponse(globals.getOData(), odataRequest, odataResponse, cdsResponse, null, responseFormat);
		} catch (Exception e) {
			logger.error("An unexpected error occurred during error processing", e);
			String message = new ErrorStatusException(ErrorStatuses.SERVER_ERROR).getLocalizedMessage(locale);
			String responseContent = "{\"error\":{\"code\":\"500\",\"message\":\"" + message + "\"}}";
			odataResponse.setContent(IOUtils.toInputStream(responseContent, StandardCharsets.UTF_8));
			odataResponse.setStatusCode(HttpStatusCode.INTERNAL_SERVER_ERROR.getStatusCode());
			odataResponse.setHeader(HttpHeader.CONTENT_TYPE, ContentType.APPLICATION_JSON.toContentTypeString());
			odataResponse.setHeader(HttpHeader.ODATA_VERSION, ODataUtils.getODataVersion(odataRequest));
		}
	}

	private static void logBatchRequest(BatchRequestPart requestPart, ODataResponsePart responsePart) {
		if (accessLogger.isInfoEnabled()) {
			List<ODataRequest> requests = requestPart.getRequests();
			List<ODataResponse> responses = responsePart.getResponses();

			for (int i = 0; i < requests.size(); i++) {
				StringBuilder builder = new StringBuilder();

				ODataRequest request = requests.get(i);
				builder.append("$batch ");
				builder.append(request.getMethod()).append(" ");
				builder.append(request.getRawRequestUri().substring(request.getRawBaseUri().length()));

				ODataResponse response;
				if (responses.size() > i) {
					response = responses.get(i);
				} else {
					response = responses.get(0);
				}
				builder.append(" ").append(response.getStatusCode());

				accessLogger.info(builder.toString());
			}
		}
	}

	private static void updateSpan(ODataRequest request, Span s) {
		String method = request.getMethod().name();
		String path = request.getRawODataPath();
		s.updateName(method + " " + path);
		s.setAttribute(OpenTelemetryUtils.HTTP_METHOD, method);
		s.setAttribute(OpenTelemetryUtils.HTTP_SCHEME, request.getProtocol());
		s.setAttribute(OpenTelemetryUtils.HTTP_TARGET, path);
		s.setAttribute(OpenTelemetryUtils.CDS_ODATA_IS_BATCH, true);
	}
}
