/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.serializer.json.options;

import org.apache.olingo.commons.api.IConstants;
import org.apache.olingo.commons.api.constants.Constantsv00;
import org.apache.olingo.commons.api.constants.Constantsv01;
import org.apache.olingo.commons.api.data.ContextURL;
import org.apache.olingo.commons.api.format.ContentType;
import org.apache.olingo.server.core.serializer.utils.ContentTypeHelper;

import com.sap.cds.adapter.odata.v4.CdsRequestGlobals;
import com.sap.cds.adapter.odata.v4.utils.EdmUtils;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

public abstract class CdsODataOptions {

	protected ContextURL contextURL;
	private boolean isODataMetadataNone;
	private boolean isODataMetadataFull;
	private final String odataVersion;
	private final CdsRequestGlobals globals;
	private final EdmUtils edmUtils;
	private final IConstants constants;
	private final boolean isIEEE754Compatible;

	public CdsODataOptions(ContentType contentType, String odataVersion, CdsRequestGlobals globals) {
		this.odataVersion = odataVersion;
		this.globals = globals;
		this.edmUtils = new EdmUtils(globals);
		this.constants = constants(odataVersion);

		if (contentType.isCompatible(ContentType.APPLICATION_JSON) || contentType.isCompatible(ContentType.APPLICATION_PDF)) {
			String odataMetadata = contentType.getParameter(ContentType.PARAMETER_ODATA_METADATA);
			if (odataMetadata != null) {
				if( ContentType.VALUE_ODATA_METADATA_NONE.equalsIgnoreCase(odataMetadata) ) {
					isODataMetadataNone = true;
				} else if ( ContentType.VALUE_ODATA_METADATA_FULL.equalsIgnoreCase(odataMetadata) ) {
					isODataMetadataFull = true;
				} else if ( ContentType.VALUE_ODATA_METADATA_MINIMAL.equalsIgnoreCase(odataMetadata) ) {
					// hit default
				} else {
					throw new ErrorStatusException(CdsErrorStatuses.UNSUPPORTED_PARAMETER_VALUE, 
							odataMetadata, ContentType.PARAMETER_ODATA_METADATA); 
				}
			}
			// else: assuming VALUE_ODATA_METADATA_MINIMAL as default
			
		} else {
			throw new ErrorStatusException(CdsErrorStatuses.UNSUPPORTED_CONTENT_TYPE, 
					contentType.toContentTypeString());	
		}
		
		isIEEE754Compatible = ContentTypeHelper.isODataIEEE754Compatible(contentType);
	}

	private static IConstants constants(String odataVersion) {
		IConstants c = new Constantsv00();
		if (odataVersion != null && Float.valueOf(odataVersion) > 4) {
			c = new Constantsv01();
		}
		return c;
	}

	public boolean isODataMetadataNone() {
		return isODataMetadataNone;
	}

	public boolean isODataMetadataFull() {
		return isODataMetadataFull;
	}

	public IConstants getConstants() {
		return constants;
	}

	public EdmUtils getEdmUtils() {
		return edmUtils;
	}

	public String getODataVersion() {
		return odataVersion;
	}

	public CdsRequestGlobals getGlobals() {
		return globals;
	}

	public boolean isIEEE754Compatible() {
		return isIEEE754Compatible;
	}

	public ContextURL getContextURL() {
		return contextURL;
	}
}
