/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.serializer.json.options;

import org.apache.olingo.commons.api.data.ContextURL;
import org.apache.olingo.commons.api.format.ContentType;
import org.apache.olingo.server.api.uri.queryoption.CountOption;
import org.apache.olingo.server.api.uri.queryoption.ExpandOption;
import org.apache.olingo.server.api.uri.queryoption.SelectOption;

import com.sap.cds.adapter.odata.v4.CdsRequestGlobals;

/**
 * Options for the collections of complex type and entity serializers.
 */
public class StructCollection2JsonOptions extends CdsODataOptions {

	private CountOption count;
	private ExpandOption expand;
	private SelectOption select;
	private boolean writeOnlyReferences;
	private String id;
	private boolean isFullRepresentation = false;
	private boolean autoExpand;

	private boolean strictSerialization = true;

	/** Gets the $count system query option. */
	public CountOption getCount() {
		return count;
	}

	/** Gets the $expand system query option. */
	public ExpandOption getExpand() {
		return expand;
	}

	/** Gets the $select system query option. */
	public SelectOption getSelect() {
		return select;
	}

	/** only writes the references of the entities */
	public boolean getWriteOnlyReferences() {
		return writeOnlyReferences;
	}

	/** Gets the id of the entity collection */
	public String getId() {
		return id;
	}

	/** Inline entries will not have @delta if representation is full **/ 
	public boolean isFullRepresentation() {
		return isFullRepresentation;
	}

	public boolean isAutoExpand() {
		return autoExpand;
	}

	public boolean isStrictSerialization() {
		return strictSerialization;
	}

	private StructCollection2JsonOptions(ContentType contentType, String odataVersion, CdsRequestGlobals globals) {
		super(contentType, odataVersion, globals);
	}

	/** Initializes the options builder. */
	public static Builder with(ContentType contentType, String odataVersion, CdsRequestGlobals globals) {
		return new Builder(contentType, odataVersion, globals);
	}

	/** Builder of OData serializer options. */
	public static final class Builder {

		private final StructCollection2JsonOptions options;

		private Builder(ContentType contentType, String odataVersion, CdsRequestGlobals globals) {
			options = new StructCollection2JsonOptions(contentType, odataVersion, globals);
		}

		/** Sets the {@link ContextURL}. */
		public Builder contextURL(ContextURL contextURL) {
			options.contextURL = contextURL;
			return this;
		}

		/** Sets the $count system query option. */
		public Builder count(CountOption count) {
			options.count = count;
			return this;
		}

		/** Sets the $expand system query option. */
		public Builder expand(ExpandOption expand) {
			options.expand = expand;
			return this;
		}

		/** Sets the $select system query option. */
		public Builder select(SelectOption select) {
			options.select = select;
			return this;
		}

		/** Sets to serialize only references */
		public Builder writeOnlyReferences(boolean ref) {
			options.writeOnlyReferences = ref;
			return this;
		}

		/** Sets id of the collection */
		public Builder id(final String id) {
			options.id = id;
			return this;
		}

		/** sets isFullRepresentation to represent inline entries**/
		public Builder isFullRepresentation(boolean isFullRepresentation) {
			options.isFullRepresentation = isFullRepresentation;
			return this;
		}

		public Builder autoExpand(boolean autoExpand) {
			options.autoExpand = autoExpand;
			return this;
		}

		public Builder strictSerialization(boolean isStrict) {
			options.strictSerialization = isStrict;
			return this;
		}

		/** Builds the OData serializer options. */
		public StructCollection2JsonOptions build() {
			return options;
		}
	}
}
