/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4;

import java.io.PrintWriter;
import java.util.stream.Stream;

import com.sap.cds.adapter.IndexContentProvider;
import com.sap.cds.reflect.CdsDefinition;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.services.utils.ODataUtils;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.services.utils.path.CdsResourcePath;
import com.sap.cds.services.utils.path.CdsServicePath;
import com.sap.cds.services.utils.path.UrlPathUtil;

public class ODataX4IndexContentProvider implements IndexContentProvider {

	private static final String SERVICE = """
			                <h3 class="header">
			                    <a href="%s/"><span>%s</span></a><span>/</span><a href="%s/$metadata"><span class="metadata">$metadata</span></a>
			                </h3>
			""";

	private static final String ENTITY_START = """
			                <ul>
			""";

	private static final String ENTITY = """
			                    <li>
			                        <div><a href="%s"><span>%s</span></a></div>
			                    </li>
			""";

	private static final String ENTITY_END = """
			                </ul>
			""";

	private final CdsRuntime runtime;

	public ODataX4IndexContentProvider(CdsRuntime runtime) {
		this.runtime = runtime;
	}

	@Override
	public String getSectionTitle() {
		return "OData X4 endpoints";
	}

	@Override
	public int order() {
		return -5;
	}

	@Override
	public void writeContent(PrintWriter out, String contextPath) {
		Stream<CdsResourcePath> cdsServicePaths = CdsServicePath.servicePaths(runtime, CdsODataX4ServletFactory.PROTOCOL_KEY);
		String basePath = UrlPathUtil.normalizeBasePath(runtime.getEnvironment().getCdsProperties().getOdataX4().getEndpoint().getPath());
		String theBasePath = contextPath + (basePath.equals("/") ? "" : basePath);

		cdsServicePaths.filter(p -> p.getCdsDefinition() instanceof CdsService).sorted().forEach(s -> {
			String path = theBasePath + "/" + s.getPath();
			out.format(SERVICE, path, path, path);
			out.write(ENTITY_START);
			s.subPaths().filter(e -> isExposedEntity(e.getCdsDefinition())).sorted().forEach(e -> {
				String odataEntity = ODataUtils.toODataName(e.getPath());
				String entityPath = path + "/" + odataEntity;
				out.format(ENTITY, entityPath, e.getPath());
			});
			out.write(ENTITY_END);
		});
	}

	public static boolean isExposedEntity(CdsDefinition definition) {
		if(definition instanceof CdsEntity entity) {
			boolean autoexposed = CdsAnnotations.AUTOEXPOSED.isTrue(definition);
			boolean autoexpose = CdsAnnotations.AUTOEXPOSE.isTrue(definition);
			String name = definition.getName();
			return !(name.endsWith("_drafts") ||
					name.endsWith("_texts") || name.endsWith(".texts") ||
					name.endsWith("DraftAdministrativeData") ||
					(!DraftUtils.isDraftEnabled(entity) && autoexposed && !autoexpose));
		}
		return false;
	}

}
