/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.metadata;

import org.apache.olingo.commons.api.format.ContentType;
import org.apache.olingo.commons.api.http.HttpHeader;
import org.apache.olingo.commons.api.http.HttpMethod;
import org.apache.olingo.commons.api.http.HttpStatusCode;
import org.apache.olingo.server.api.OData;
import org.apache.olingo.server.api.ODataApplicationException;
import org.apache.olingo.server.api.ODataLibraryException;
import org.apache.olingo.server.api.ODataRequest;
import org.apache.olingo.server.api.ODataResponse;
import org.apache.olingo.server.api.ServiceMetadata;
import org.apache.olingo.server.api.etag.ETagHelper;
import org.apache.olingo.server.api.etag.ServiceMetadataETagSupport;
import org.apache.olingo.server.api.processor.MetadataProcessor;
import org.apache.olingo.server.api.serializer.ODataSerializer;
import org.apache.olingo.server.api.uri.UriInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.adapter.edmx.EdmxProvider;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

public class CustomMetadataProcessor implements MetadataProcessor {

	final static Logger logger = LoggerFactory.getLogger(CustomMetadataProcessor.class);

	private final EdmxProvider edmxProvider;
	private final String serviceName;
	private OData odata = null;
	private ServiceMetadata serviceMetadata = null;

	public CustomMetadataProcessor(EdmxProvider edmxProvider, String serviceName) {
		this.edmxProvider = edmxProvider;
		this.serviceName = serviceName;
	}

	@Override
	public void init(OData odata, ServiceMetadata serviceMetadata) {
		this.odata = odata;
		this.serviceMetadata = serviceMetadata;
	}

	@Override
	public void readMetadata(ODataRequest request, ODataResponse response, UriInfo uriInfo, ContentType responseFormat) throws ODataApplicationException, ODataLibraryException {
		boolean isNotModified = false;
		ServiceMetadataETagSupport eTagSupport = serviceMetadata.getServiceMetadataETagSupport();
		if (eTagSupport != null && eTagSupport.getMetadataETag() != null) {
			// Set application etag at response
			response.setHeader(HttpHeader.ETAG, eTagSupport.getMetadataETag());
			// Check if metadata document has been modified
			ETagHelper eTagHelper = odata.createETagHelper();
			isNotModified = eTagHelper.checkReadPreconditions(eTagSupport.getMetadataETag(), request.getHeaders(HttpHeader.IF_MATCH), request.getHeaders(HttpHeader.IF_NONE_MATCH));
			// allow caching, but enforce revalidation using etags
			response.setHeader(HttpHeader.CACHE_CONTROL, "max-age=0");
		}

		// Send the correct response
		if (isNotModified) {
		  	response.setStatusCode(HttpStatusCode.NOT_MODIFIED.getStatusCode());
		} else {
			// HTTP HEAD requires no payload but a 200 OK response
			if (HttpMethod.HEAD == request.getMethod()) {
				response.setStatusCode(HttpStatusCode.OK.getStatusCode());
			} else {
				if (responseFormat.isCompatible(ContentType.APPLICATION_XML)) {
					response.setContent(edmxProvider.getEdmx(serviceName));
				} else {
					ODataSerializer serializer = odata.createSerializer(responseFormat);
					response.setContent(serializer.metadataDocument(serviceMetadataForJson()).getContent());
				}
				response.setStatusCode(HttpStatusCode.OK.getStatusCode());
				response.setHeader(HttpHeader.CONTENT_TYPE, responseFormat.toContentTypeString());
			}
		}
	}

	private ServiceMetadata serviceMetadataForJson() {
		try {
			ODataEdmProvider edmProvider = new ODataEdmProvider(edmxProvider.getEdmx(serviceName));
			return odata.createServiceMetadata(edmProvider, edmProvider.getAllEDMXReference(), serviceMetadata.getServiceMetadataETagSupport());
		} catch (Exception e) {
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_METADATA, serviceName, e);
		}
	}

}
