/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.metadata.cds;

import static com.sap.cds.adapter.odata.v4.metadata.cds.CdsServiceEdmUtils.CDS_MAP_TYPE;

import java.util.List;

import org.apache.olingo.commons.api.edm.FullQualifiedName;
import org.apache.olingo.commons.api.edm.provider.CsdlAction;
import org.apache.olingo.commons.api.edm.provider.CsdlActionImport;
import org.apache.olingo.commons.api.edm.provider.CsdlAliasInfo;
import org.apache.olingo.commons.api.edm.provider.CsdlAnnotations;
import org.apache.olingo.commons.api.edm.provider.CsdlComplexType;
import org.apache.olingo.commons.api.edm.provider.CsdlEdmProvider;
import org.apache.olingo.commons.api.edm.provider.CsdlEntityContainer;
import org.apache.olingo.commons.api.edm.provider.CsdlEntityContainerInfo;
import org.apache.olingo.commons.api.edm.provider.CsdlEntitySet;
import org.apache.olingo.commons.api.edm.provider.CsdlEntityType;
import org.apache.olingo.commons.api.edm.provider.CsdlEnumType;
import org.apache.olingo.commons.api.edm.provider.CsdlFunction;
import org.apache.olingo.commons.api.edm.provider.CsdlFunctionImport;
import org.apache.olingo.commons.api.edm.provider.CsdlSchema;
import org.apache.olingo.commons.api.edm.provider.CsdlSingleton;
import org.apache.olingo.commons.api.edm.provider.CsdlTerm;
import org.apache.olingo.commons.api.edm.provider.CsdlTypeDefinition;
import org.apache.olingo.commons.api.ex.ODataException;

import com.sap.cds.adapter.odata.v4.utils.mapper.EdmxFlavourMapper;
import com.sap.cds.adapter.odata.v4.utils.mapper.EdmxFlavourMapper.EdmxFlavour;
import com.sap.cds.reflect.CdsService;

public class CdsServiceEdmProvider implements CsdlEdmProvider {

	private final CsdlEntityContainerInfo entityContainerInfo;
	private final CsdlEntityContainer entityContainer;
	private final CsdlSchema schema;

	public CdsServiceEdmProvider(CdsService service, EdmxFlavour flavour) {
		this.entityContainerInfo = new CsdlEntityContainerInfo()
			.setContainerName(new FullQualifiedName(service.getQualifiedName(), CdsServiceEdmEntityContainer.ENTITY_CONTAINER_NAME));

		EdmxFlavourMapper flavourMapper = EdmxFlavourMapper.create(flavour, false);
		this.entityContainer = new CdsServiceEdmEntityContainer(service, flavour, flavourMapper);
		this.schema = new CdsServiceEdmSchema(service, entityContainer, flavour, flavourMapper);
	}

	@Override
	public CsdlEntityContainerInfo getEntityContainerInfo(FullQualifiedName entityContainerName) throws ODataException {
		return entityContainerInfo;
	}

	@Override
	public List<CsdlSchema> getSchemas() throws ODataException {
		return List.of(schema);
	}

	@Override
	public CsdlEntityContainer getEntityContainer() throws ODataException {
		return entityContainer;
	}

	@Override
	public CsdlEntitySet getEntitySet(FullQualifiedName entityContainer, String entitySetName) throws ODataException {
		if (!entityContainer.equals(entityContainerInfo.getContainerName())) {
			return null;
		}
		return getEntityContainer().getEntitySet(entitySetName);
	}

	@Override
	public CsdlSingleton getSingleton(FullQualifiedName entityContainer, String singletonName) throws ODataException {
		if (!entityContainer.equals(entityContainerInfo.getContainerName())) {
			return null;
		}
		return getEntityContainer().getSingleton(singletonName);
	}

	@Override
	public CsdlActionImport getActionImport(FullQualifiedName entityContainer, String actionImportName) throws ODataException {
		if (!entityContainer.equals(entityContainerInfo.getContainerName())) {
			return null;
		}
		return getEntityContainer().getActionImport(actionImportName);
	}

	@Override
	public CsdlFunctionImport getFunctionImport(FullQualifiedName entityContainer, String functionImportName) throws ODataException {
		if (!entityContainer.equals(entityContainerInfo.getContainerName())) {
			return null;
		}
		return getEntityContainer().getFunctionImport(functionImportName);
	}

	@Override
	public CsdlEntityType getEntityType(FullQualifiedName entityTypeName) throws ODataException {
		if (!entityTypeName.getNamespace().equals(entityContainerInfo.getContainerName().getNamespace())) {
			return null;
		}
		return schema.getEntityType(entityTypeName.getName());
	}

	@Override
	public CsdlComplexType getComplexType(FullQualifiedName complexTypeName) throws ODataException {
		if (!(complexTypeName.getNamespace().equals(entityContainerInfo.getContainerName().getNamespace())
				|| CDS_MAP_TYPE.equals(complexTypeName.getName()))) {
			return null;
		}
		return schema.getComplexType(complexTypeName.getName());
	}

	@Override
	public List<CsdlAction> getActions(FullQualifiedName actionName) throws ODataException {
		if (!actionName.getNamespace().equals(entityContainerInfo.getContainerName().getNamespace())) {
			return null;
		}
		return schema.getActions(actionName.getName());
	}

	@Override
	public List<CsdlFunction> getFunctions(FullQualifiedName functionName) throws ODataException {
		if (!functionName.getNamespace().equals(entityContainerInfo.getContainerName().getNamespace())) {
			return null;
		}
		return schema.getFunctions(functionName.getName());
	}

	@Override
	public CsdlTypeDefinition getTypeDefinition(FullQualifiedName typeDefinitionName) throws ODataException {
		// no type definitions in CDS based EDMX
		return null;
	}

	@Override
	public CsdlEnumType getEnumType(FullQualifiedName enumTypeName) throws ODataException {
		// no enum types in CDS based EDMX
		return null;
	}

	@Override
	public CsdlTerm getTerm(FullQualifiedName termName) throws ODataException {
		// no terms in CDS based EDMX
		return null;
	}

	@Override
	public List<CsdlAliasInfo> getAliasInfos() throws ODataException {
		// no aliases in CDS based EDMX
		return null;
	}

	@Override
	public CsdlAnnotations getAnnotationsGroup(FullQualifiedName targetName, String qualifier) throws ODataException {
		// no annotations in CDS based EDMX
		return null;
	}

}
