/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.metadata.extension;

import java.util.ArrayList;
import java.util.List;

import org.apache.olingo.commons.api.edm.provider.CsdlComplexType;
import org.apache.olingo.commons.api.edm.provider.CsdlFunction;
import org.apache.olingo.commons.api.edm.provider.CsdlParameter;
import org.apache.olingo.commons.api.edm.provider.CsdlProperty;
import org.apache.olingo.commons.api.edm.provider.CsdlReturnType;
import org.apache.olingo.commons.api.edm.provider.CsdlSchema;

public class CustomVocabularyCatalog {

	private static final String TOP_LEVELS_EXPAND_TYPE = "Hierarchy.TopLevelsExpandType";
	public static final String COM_SAP_VOCABULARY_HIERARCHY = "com.sap.vocabularies.Hierarchy.v1";
	public static final String TOP_LEVELS = "TopLevels";

	/*
	 * Returns the first `n` levels of a hierarchical collection in preorder
	 * with individual nodes expanded or collapsed.
	 *
	 * This function can be used as a transformation whose input set has a recursive
	 * hierarchy defined by an
	 * [`Aggregation.RecursiveHierarchy`](https://oasis-tcs.github.io/odata-
	 * vocabularies/vocabularies/Org.OData.Aggregation.V1.html#RecursiveHierarchy)
	 * annotation on the entity type of the `HierarchyNodes`.
	 * (Its binding parameter is the unlimited hierarchy as defined
	 * [here](#RecursiveHierarchyType),
	 * its output is the limited hierarchy.) The output initially contains the nodes
	 * with less than n ancestors in the hierarchical collection given in the
	 * binding parameter.
	 * Then individual nodes are expanded, shown or collapsed in the output, which
	 * extends or reduces the limited hierarchy.
	 * Finally the output is sorted in preorder as with the `traverse`
	 * transformation with the hierarchy-specific definition of start nodes.
	 */
	private static CsdlFunction topLevels() {
		// <Function Name="TopLevels" IsBound="true" EntitySetPath="InputSet">
		CsdlFunction f = new CsdlFunction()
				.setName(TOP_LEVELS).setBound(true).setEntitySetPath("InputSet");

		List<CsdlParameter> parameters = new ArrayList<>();
		// <Parameter Name="InputSet" Type="Collection(Edm.EntityType)"
		// Nullable="false"/>
		parameters.add(new CsdlParameter()
				.setName("InputSet")
				.setType("Edm.EntityType")
				.setCollection(true)
				.setNullable(false));

		// <Parameter Name="HierarchyNodes" Type="Collection(Edm.EntityType)"
		// Nullable="false" />
		parameters.add(new CsdlParameter()
				.setName("HierarchyNodes")
				.setType("Edm.EntityType")
				.setCollection(true)
				.setNullable(false));

		// <Parameter Name="HierarchyQualifier" Type="Aggregation.HierarchyQualifier"
		// Nullable="false" />
		parameters.add(new CsdlParameter()
				.setName("HierarchyQualifier")
				.setType("Aggregation.HierarchyQualifier")
				.setNullable(false));

		// <Parameter Name="NodeProperty" Type="Edm.String" Nullable="false" />
		parameters.add(new CsdlParameter()
				.setName("NodeProperty")
				.setType("Edm.String")
				.setNullable(false));

		// <Parameter Name="Levels" Type="Edm.Int64" Nullable="true" />
		parameters.add(new CsdlParameter()
				.setName("Levels")
				.setType("Edm.Int64")
				.setNullable(true));

		parameters.add(new CsdlParameter()
				.setName("ExpandLevels")
				.setType(TOP_LEVELS_EXPAND_TYPE)
				.setNullable(true)
				.setCollection(true));

		f.setParameters(parameters);

		// <ReturnType Type="Collection(Edm.EntityType)" />
		f.setReturnType(new CsdlReturnType()
				.setType("Edm.EntityType")
				.setCollection(true));

		return f;
	}

	private static CsdlComplexType topLevelsExpandType() {
		CsdlComplexType t = new CsdlComplexType().setName(TOP_LEVELS_EXPAND_TYPE);

		CsdlProperty nodeId = new CsdlProperty().setName("NodeID").setType("Edm.String").setNullable(false);
		CsdlProperty levels = new CsdlProperty().setName("Levels").setType("Edm.Int64").setNullable(true);

		t.setProperties(List.of(nodeId, levels));

		return t;
	}

	public static CsdlSchema getHierarchySchema() {
		return new CsdlSchema()
				.setNamespace(COM_SAP_VOCABULARY_HIERARCHY)
				.setComplexTypes(List.of(topLevelsExpandType()))
				.setFunctions(List.of(topLevels()));
	}
}
