/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.metadata.provider;

import java.io.InputStream;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.olingo.commons.api.edm.provider.CsdlEdmProvider;

import com.sap.cds.adapter.odata.v4.metadata.MetadataInfo;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.LocaleUtils;

/**
 * Default implementation of the EDMX parsing. It looks for the EDMX in the applications resource folder.
 */
public abstract class AbstractDefaultEdmxProvider implements OlingoAwareEdmxProvider {

	private final Map<String, MetadataInfo> metadataInfoMap = new ConcurrentHashMap<>();
	private final CdsRuntime runtime;
	private final String basePath;

	public AbstractDefaultEdmxProvider(CdsRuntime runtime, String basePath) {
		this.runtime = runtime;
		this.basePath = basePath;
	}

	@Override
	public String getETag(String serviceName) {
		return getMetadataInfo(serviceName).getETag();
	}

	@Override
	public InputStream getEdmx(String serviceName) {
		return getMetadataInfo(serviceName).getEdmx();
	}

	@Override
	public CsdlEdmProvider getEdmProvider(String serviceName) {
		return getMetadataInfo(serviceName).getEdmProvider();
	}

	MetadataInfo getMetadataInfo(String serviceName) {
		Locale locale = RequestContext.getCurrent(runtime).getParameterInfo().getLocale();
		MetadataInfo info = getMetadataInfo(serviceName, locale);
		if(locale != null && info.isEmpty()) {
			// retry without locale
			info = getMetadataInfo(serviceName, null);
		}
		if (info.isEmpty()) {
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_METADATA, serviceName);
		}
		return info;
	}

	private MetadataInfo getMetadataInfo(String unlocalizedServiceName, Locale locale) {
		String serviceName;
		if(locale != null) {
			String language = LocaleUtils.getLocaleForBundle(locale).toString();
			serviceName = unlocalizedServiceName + "_" + language;
		} else {
			serviceName = unlocalizedServiceName;
		}

		return metadataInfoMap.computeIfAbsent(serviceName, (key) -> {
			try {
				String resourcePathToEdmx = basePath + "/" + serviceName + ".xml"; // getResourceAsStream accepts unnormalized paths containing e.g. "//"
				try(InputStream resource = getClass().getClassLoader().getResourceAsStream(resourcePathToEdmx)) {
					if (resource != null) {
						return MetadataInfo.create(serviceName, resource.readAllBytes(), runtime);
					}
				}
				return MetadataInfo.create(serviceName, null, runtime);
			} catch (Exception e) {
				throw new ErrorStatusException(CdsErrorStatuses.INVALID_METADATA, serviceName, e);
			}
		});
	}

}
