/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.metadata.provider;

import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Locale;

import org.apache.olingo.commons.api.edm.provider.CsdlEdmProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.adapter.odata.v4.metadata.MetadataInfo;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.LocaleUtils;
import com.sap.cds.services.utils.lib.mtx.MetaDataAccessor;
import com.sap.cds.services.utils.lib.mtx.ModelId;
import com.sap.cds.services.utils.lib.mtx.impl.MetaDataAccessorImpl.EdmxModelCreator;
import com.sap.cds.services.utils.model.DynamicModelUtils;

/**
 * An implementation of the EDMX parsing that uses the MTX Sidecar or the Model Provider to retrieve the EDMX files.
 */
public abstract class AbstractDynamicEdmxProvider implements OlingoAwareEdmxProvider {

	private static Logger log = LoggerFactory.getLogger(AbstractDynamicEdmxProvider.class);

	private final String odataVersion;
	private final CdsRuntime runtime;
	private final DynamicModelUtils utils;
	private final AbstractDefaultEdmxProvider previous;
	private final MetaDataAccessor<MetadataInfo> accessor;

	public AbstractDynamicEdmxProvider(String odataVersion, CdsRuntime runtime, DynamicModelUtils utils, AbstractDefaultEdmxProvider previous) {
		this.odataVersion = odataVersion;
		this.runtime = runtime;
		this.utils = utils;
		this.previous = previous;
		this.accessor = utils.createMetadataAccessor(new EdmxModelCreator<MetadataInfo>() {

			@Override
			public MetadataInfo parse(String edmx, String serviceName) {
				try {
					return MetadataInfo.create(serviceName, edmx.getBytes(StandardCharsets.UTF_8), runtime);
				} catch (Exception e) {
					throw new ErrorStatusException(CdsErrorStatuses.INVALID_METADATA, serviceName, e);
				}
			}

			@Override
			public MetadataInfo getBaseModel(String serviceName) {
				// TODO this is not the base model as interpreted by MTXS, as it contains all feature toggles
				return previous.getMetadataInfo(serviceName);
			}

		}, null, null);
	}

	@Override
	public String getETag(String serviceName) {
		return getMetadataInfo(serviceName).getETag();
	}

	@Override
	public InputStream getEdmx(String serviceName) {
		return getMetadataInfo(serviceName).getEdmx();
	}

	@Override
	public CsdlEdmProvider getEdmProvider(String serviceName) {
		return getMetadataInfo(serviceName).getEdmProvider();
	}

	private MetadataInfo getMetadataInfo(String serviceName) {
		if (utils.useStaticModel()) {
			return previous.getMetadataInfo(serviceName);
		}

		RequestContext requestContext = RequestContext.getCurrent(runtime);
		String localeString = null;
		if (!runtime.getEnvironment().getCdsProperties().getOdataV4().getLazyI18n().isEnabled()) {
			Locale locale = requestContext.getParameterInfo().getLocale();
			localeString = locale != null ? LocaleUtils.getLocaleForBundle(locale).toString() : Locale.ENGLISH.getLanguage();
		}

		ModelId modelId = utils.prepareModelId(requestContext.getUserInfo(), requestContext.getFeatureTogglesInfo())
				.odata(serviceName, localeString, odataVersion).build();
		log.debug("Retrieving model for service name '{}', locale '{}' and features {}", serviceName, localeString, modelId.getFeatures());
		try {
			return accessor.getEdmx(modelId, Integer.MAX_VALUE);
		} catch (ServiceException e) {
			throw e;
		} catch (Throwable t) {
			throw new ErrorStatusException(CdsErrorStatuses.EDMX_READ_FAILED, serviceName, t);
		}
	}

}
