/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.metadata.provider;

import java.io.StringReader;
import java.util.Locale;
import java.util.Map;

import com.google.common.hash.Hashing;
import com.sap.cds.adapter.edmx.EdmxI18nProvider;
import com.sap.cds.adapter.odata.v4.metadata.MetadataInfo;
import com.sap.cds.impl.localized.LocaleUtils;
import com.sap.cds.impl.parser.JsonParser;
import com.sap.cds.services.utils.lib.mtx.MetaDataAccessor;
import com.sap.cds.services.utils.lib.mtx.MetaDataAccessor.I18n;
import com.sap.cds.services.utils.lib.mtx.ModelId;
import com.sap.cds.services.utils.lib.mtx.impl.MetaDataAccessorImpl.I18nResourceCreator;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.model.DynamicModelUtils;

class DynamicEdmxI18nProvider implements EdmxI18nProvider {

	private final CdsRuntime runtime;
	private final DynamicModelUtils utils;
	private final EdmxI18nProvider previous;
	private final MetaDataAccessor<MetadataInfo> accessor;

	@SuppressWarnings("unchecked")
	public DynamicEdmxI18nProvider(CdsRuntime runtime, DynamicModelUtils utils, DefaultEdmxI18nProvider previous) {
		this.runtime = runtime;
		this.utils = utils;
		this.previous = previous;
		this.accessor = utils.createMetadataAccessor(null, null, new I18nResourceCreator() {

			@Override
			public I18n parse(String json, String language) {
				// TODO deduplicate texts with base model texts and create an overlay map
				String eTag = Hashing.sha256().hashUnencodedChars(json).toString();
				Map<String, String> texts = (Map<String, String>) JsonParser.map(new StringReader(json));
				return new I18n(eTag, texts);
			}

			@Override
			public I18n getBaseModel(String language) {
				return new I18n(previous.getETag(language), previous.getTexts(language));
			}

		});
	}

	@Override
	public String getETag(Locale locale) {
		if (utils.useStaticModel()) {
			return previous.getETag(locale);
		}
		return getI18n(locale).eTag();
	}

	@Override
	public Map<String, String> getTexts(Locale locale) {
		if (utils.useStaticModel()) {
			return previous.getTexts(locale);
		}
		return getI18n(locale).texts();
	}

	private I18n getI18n(Locale locale) {
		String tenant = RequestContext.getCurrent(runtime).getUserInfo().getTenant();
		// empty string represents the default language
		String localeString = locale == null ? "" : LocaleUtils.getLocaleString(locale);
		return accessor.getI18n(ModelId.create(tenant).language(localeString).allFeatures().build(), Integer.MAX_VALUE);
	}
}
