/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.metadata.provider;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.adapter.edmx.EdmxV4Provider;
import com.sap.cds.adapter.edmx.EdmxX4Provider;
import com.sap.cds.services.environment.CdsProperties;
import com.sap.cds.services.environment.CdsProperties.ODataV4;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.CdsRuntimeConfiguration;
import com.sap.cds.services.runtime.CdsRuntimeConfigurer;
import com.sap.cds.services.utils.model.DynamicModelUtils;

public class EdmxProviderConfiguration implements CdsRuntimeConfiguration {

	private static Logger logger = LoggerFactory.getLogger(EdmxProviderConfiguration.class);

	@Override
	public void providers(CdsRuntimeConfigurer configurer) {
		DynamicModelUtils utils = new DynamicModelUtils(configurer.getCdsRuntime());
		CdsProperties properties = configurer.getCdsRuntime().getEnvironment().getCdsProperties();

		// V4 metadata
		if (properties.getOdataV4().getEndpoint().isEnabled()) {
			DefaultEdmxV4Provider defaultV4Provider = new DefaultEdmxV4Provider(configurer.getCdsRuntime());
			configurer.provider(defaultV4Provider);
			if (utils.isModelProviderEnabled()) {
				configurer.provider(new DynamicEdmxV4Provider(configurer.getCdsRuntime(), utils, defaultV4Provider));
				logger.info("Initialized Dynamic EDMX V4 Provider");
			} else {
				logger.info("Initialized Default EDMX V4 Provider");
			}
		}

		// X4 metadata
		if (properties.getOdataX4().getEndpoint().isEnabled()) {
			DefaultEdmxX4Provider defaultX4Provider = new DefaultEdmxX4Provider(configurer.getCdsRuntime());
			configurer.provider(defaultX4Provider);
			if (utils.isModelProviderEnabled()) {
				configurer.provider(new DynamicEdmxX4Provider(configurer.getCdsRuntime(), utils, defaultX4Provider));
				logger.info("Initialized Dynamic EDMX X4 Provider");
			} else {
				logger.info("Initialized Default EDMX X4 Provider");
			}
		}

		// i18n bundles
		if (properties.getOdataV4().getLazyI18n().isEnabled()) {
			DefaultEdmxI18nProvider defaultI18nProvider = DefaultEdmxI18nProvider.create();
			configurer.provider(defaultI18nProvider);
			if (utils.isModelProviderEnabled()) {
				configurer.provider(new DynamicEdmxI18nProvider(configurer.getCdsRuntime(), utils, defaultI18nProvider));
				logger.info("Initialized Dynamic EDMX I18n Provider");
			} else {
				logger.info("Initialized Default EDMX I18n Provider");
			}
		}
	}

	@Override
	public int order() {
		return Integer.MIN_VALUE;
	}

	private static class DefaultEdmxV4Provider extends AbstractDefaultEdmxProvider implements EdmxV4Provider {

		// base path for unlocalized EDMX files
		private final static String UNLOCALIZED_BASE_PATH = "edmx/odata/v4";

		public DefaultEdmxV4Provider(CdsRuntime runtime) {
			super(runtime, determineBasePath(runtime));
		}

		private static String determineBasePath(CdsRuntime runtime) {
			ODataV4 config = runtime.getEnvironment().getCdsProperties().getOdataV4();
			if (EdmxProviderConfiguration.class.getClassLoader().getResource(config.getEdmxPath()) != null) {
				return config.getEdmxPath();
			} else if (config.getLazyI18n().isEnabled() && EdmxProviderConfiguration.class.getClassLoader().getResource(UNLOCALIZED_BASE_PATH) != null) {
				return UNLOCALIZED_BASE_PATH;
			}
			return "edmx"; // legacy default
		}

	}

	private static class DynamicEdmxV4Provider extends AbstractDynamicEdmxProvider implements EdmxV4Provider {

		public DynamicEdmxV4Provider(CdsRuntime runtime, DynamicModelUtils utils, DefaultEdmxV4Provider previous) {
			super("v4", runtime, utils, previous);
		}

	}

	private static class DefaultEdmxX4Provider extends AbstractDefaultEdmxProvider implements EdmxX4Provider {

		public DefaultEdmxX4Provider(CdsRuntime runtime) {
			// simpler basePath logic, as we assume unlocalized edmx resources from the beginning
			super(runtime, runtime.getEnvironment().getCdsProperties().getOdataX4().getEdmxPath());
		}

	}

	private static class DynamicEdmxX4Provider extends AbstractDynamicEdmxProvider implements EdmxX4Provider {

		public DynamicEdmxX4Provider(CdsRuntime runtime, DynamicModelUtils utils, DefaultEdmxX4Provider previous) {
			super("x4", runtime, utils, previous);
		}

	}

}
