/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.processors;

import static com.sap.cds.adapter.odata.v4.processors.request.CdsODataRequest.DELTA_DELETE;
import static com.sap.cds.adapter.odata.v4.processors.request.CdsODataRequest.DELTA_UPSERT;
import static com.sap.cds.adapter.odata.v4.utils.ElementUtils.aliasedGet;
import static com.sap.cds.services.utils.model.CqnUtils.addWhere;
import static jakarta.servlet.http.HttpServletResponse.SC_CREATED;
import static jakarta.servlet.http.HttpServletResponse.SC_NOT_MODIFIED;
import static jakarta.servlet.http.HttpServletResponse.SC_NO_CONTENT;
import static jakarta.servlet.http.HttpServletResponse.SC_OK;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.http.HttpHeaders;
import org.apache.olingo.commons.api.edm.EdmEntityType;
import org.apache.olingo.commons.api.edm.EdmKeyPropertyRef;
import org.apache.olingo.commons.api.edm.EdmOperation;
import org.apache.olingo.commons.api.edm.EdmProperty;
import org.apache.olingo.commons.api.edm.EdmType;
import org.apache.olingo.commons.api.http.HttpMethod;
import org.apache.olingo.server.api.prefer.Preferences.Return;
import org.apache.olingo.server.api.uri.UriInfo;
import org.apache.olingo.server.api.uri.UriParameter;
import org.apache.olingo.server.api.uri.UriResource;
import org.apache.olingo.server.api.uri.UriResourceComplexProperty;
import org.apache.olingo.server.api.uri.UriResourceEntitySet;
import org.apache.olingo.server.api.uri.UriResourceKind;
import org.apache.olingo.server.api.uri.UriResourceNavigation;
import org.apache.olingo.server.api.uri.UriResourcePartTyped;
import org.apache.olingo.server.api.uri.UriResourcePrimitiveProperty;
import org.apache.olingo.server.api.uri.UriResourceProperty;
import org.apache.olingo.server.api.uri.UriResourceSingleton;
import org.apache.olingo.server.api.uri.UriResourceValue;
import org.apache.olingo.server.api.uri.queryoption.ExpandOption;
import org.apache.olingo.server.api.uri.queryoption.SelectOption;
import org.apache.olingo.server.api.uri.queryoption.expression.Literal;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.CdsDataProcessor;
import com.sap.cds.Result;
import com.sap.cds.ResultBuilder;
import com.sap.cds.Row;
import com.sap.cds.SessionContext;
import com.sap.cds.adapter.odata.v4.CdsRequestGlobals;
import com.sap.cds.adapter.odata.v4.processors.request.CdsODataRequest;
import com.sap.cds.adapter.odata.v4.processors.response.CdsODataResponse;
import com.sap.cds.adapter.odata.v4.query.NextLinkInfo;
import com.sap.cds.adapter.odata.v4.query.SystemQueryLoader;
import com.sap.cds.adapter.odata.v4.utils.ETagHelper;
import com.sap.cds.adapter.odata.v4.utils.EdmUtils;
import com.sap.cds.adapter.odata.v4.utils.ElementUtils;
import com.sap.cds.adapter.odata.v4.utils.ODataUtils;
import com.sap.cds.adapter.odata.v4.utils.QueryLimitUtils;
import com.sap.cds.adapter.odata.v4.utils.StreamUtils;
import com.sap.cds.adapter.odata.v4.utils.TypeConverterUtils;
import com.sap.cds.adapter.odata.v4.utils.mapper.EdmxFlavourMapper;
import com.sap.cds.impl.DataProcessor;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Delete;
import com.sap.cds.ql.Expand;
import com.sap.cds.ql.Insert;
import com.sap.cds.ql.Predicate;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.Update;
import com.sap.cds.ql.Upsert;
import com.sap.cds.ql.cqn.AnalysisResult;
import com.sap.cds.ql.cqn.CqnAnalyzer;
import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnExpand;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.ql.cqn.CqnUpsert;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsService;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.services.ErrorStatuses;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CdsReadEventContext;
import com.sap.cds.services.changeset.ChangeSetContext;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.environment.CdsProperties;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.ListUtils;
import com.sap.cds.services.utils.ResultUtils;
import com.sap.cds.services.utils.SessionContextUtils;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.services.utils.model.CdsModelUtils;
import com.sap.cds.util.DataUtils;

public class CdsProcessor {

	private static final Logger logger = LoggerFactory.getLogger(CdsProcessor.class);
	private static final Result NOT_MODIFIED = ResultBuilder.selectedRows(Collections.emptyList()).result();
	private final CdsRequestGlobals globals;
	private final EdmxFlavourMapper requestMapper;
	private final EdmxFlavourMapper responseMapper;
	private final EdmUtils edmUtils;
	private CdsProperties cdsProperties;

	public CdsProcessor(CdsRequestGlobals globals) {
		this.globals = globals;
		this.cdsProperties = globals.getRuntime().getEnvironment().getCdsProperties();
		this.requestMapper = EdmxFlavourMapper.create(globals.getEdmxFlavour(), true);
		this.responseMapper = EdmxFlavourMapper.create(globals.getEdmxFlavour(), false);
		this.edmUtils = new EdmUtils(globals);
	}

	public void processRequest(CdsODataRequest request, Consumer<CdsODataResponse> responseProcessor) {
		ChangeSetContext changeSetContext = ChangeSetContext.getCurrent();
		if(changeSetContext == null) {
			globals.getRuntime().changeSetContext().run((context) -> {
				processRequest(request, responseProcessor, context);
				return null;
			});
		} else {
			processRequest(request, responseProcessor, changeSetContext);
		}
	}

	private void processRequest(CdsODataRequest cdsRequest, Consumer<CdsODataResponse> responseProcessor, ChangeSetContext changeSetContext) {
		globals.getRuntime().requestContext().clearMessages().parameters(cdsRequest).run(requestContext -> {
			CdsODataResponse response;
			try {
				response = delegateRequest(cdsRequest);
			} catch (ServiceException e) {
				markForCancel(changeSetContext, e);

				// only treat unexpected exceptions as errors
				if (e.getErrorStatus().getHttpStatus() >= 500 && e.getErrorStatus().getHttpStatus() < 600) {
					logger.error(e.getMessage(), e);
				} else {
					logger.debug(e.getMessage(), e);
				}
				response = ODataUtils.toErrorResponse(e, globals);
			} catch (Throwable e) {
				markForCancel(changeSetContext, e);

				// assume internal server error
				logger.error(e.getMessage(), e);
				response = ODataUtils.toErrorResponse(new ErrorStatusException(ErrorStatuses.SERVER_ERROR, e), globals);
			}

			try {
				responseProcessor.accept(response);
			} catch (Throwable e) {
				markForCancel(changeSetContext, e);
				throw e;
			}
		});
	}

	private void markForCancel(ChangeSetContext changeSet, Throwable e) {
		changeSet.markForCancel();
		logger.info("Exception marked the ChangeSet {} as cancelled: {}", changeSet.getId(), e.getMessage());
	}

	private CdsODataResponse delegateRequest(CdsODataRequest request) {
		UriResource lastResource = request.getLastResource();
		HttpMethod method = request.getODataRequest().getMethod();

		UriResourceKind kind = lastResource.getKind();
		switch (kind) {
		case action:
		case function:
			return operation(request);
		case primitiveProperty:
		case complexProperty:
		case value:
			switch (method) {
			case GET:
				return get(request, true);
			case POST:
				// A successful POST request to the edit URL of a collection property adds an item to the collection.
				// The body of the request MUST be a single item to be added to the collection.
				boolean isCollection = false;
				if (kind.equals(UriResourceKind.complexProperty)) {
					isCollection = ((UriResourceComplexProperty) lastResource).getProperty().isCollection();
				} else if (kind.equals(UriResourceKind.primitiveProperty)) {
					isCollection = ((UriResourcePrimitiveProperty) lastResource).getProperty().isCollection();
				}
				if (isCollection) {
					return patch(request);
				}
				throw new ErrorStatusException(ErrorStatuses.NOT_IMPLEMENTED);
				// DELETE requests on primitive properties are handled as PUT requests
				// with the respective property set to null, see:
				// https://docs.oasis-open.org/odata/odata/v4.01/odata-v4.01-part1-protocol.html#sec_SetaValuetoNull
			case PUT:
			case DELETE:
			case PATCH:
				return patch(request);
			default:
				throw new ErrorStatusException(ErrorStatuses.NOT_IMPLEMENTED);
			}
		default:
			switch (method) {
			case GET:
				return get(request, true);
			case POST:
				return post(request);
			case PUT:
				return put(request);
			case DELETE:
				return delete(request);
			case PATCH:
				if (request.isDeltaCollection()) {
					return deltaCollection(request);
				}
				return patch(request);
			default:
				throw new ErrorStatusException(ErrorStatuses.NOT_IMPLEMENTED);
			}
		}
	}

	private CdsODataResponse get(CdsODataRequest request, boolean remap) {
		ApplicationService applicationService = globals.getApplicationService();
		CdsEntity entity = request.getLastEntity();
		CdsService cdsServiceModel = applicationService.getDefinition();

		// contains content type of a media stream if such a property is requested
		String contentType = null;
		String contentDispoFilename = null;
		String contentDispoType = null;

		UriResource lastResource = request.getLastResource();
		UriInfo uriInfo = request.getUriInfo();

		Map<String, Object> parameters = new HashMap<>();
		Select<?> select = Select.from(toPathExpression(uriInfo.getUriResourceParts(), parameters));
		boolean transformInCqn = false;
		transformInCqn |= cdsProperties.getOdataV4().getApply().getTransformations().isEnabled();
		transformInCqn |= CdsAnnotations.ODATA_APPLY_TRANSFORMATIONS.isTrue(cdsServiceModel);
		SystemQueryLoader queryLoader = SystemQueryLoader.create(requestMapper, globals.getEdmxFlavour(), cdsProperties, transformInCqn);
		List<Select<?>> selects = queryLoader.updateSelectQuery(select, uriInfo, entity);

		String mediaTypeElementName = null;
		String contentDispoElementName = null;

		switch (lastResource.getKind()) {
		case primitiveProperty:
		case complexProperty:
			EdmProperty property = ((UriResourceProperty) lastResource).getProperty();
			String elementName = requestMapper.remap(property.getName(), entity);

			// if primitive property is a stream also try to get content type
			if (property.getType().getFullQualifiedName().getFullQualifiedNameAsString().equals("Edm.Stream")) {

				List<CqnSelectListItem> columns = new ArrayList<>();
				columns.add(aliasedGet(elementName));

				// try to find field containing the content type
				mediaTypeElementName = StreamUtils.getCoreMediaTypeElement(entity, elementName);
				if (!StringUtils.isEmpty(mediaTypeElementName)) {
					// also read media type from entity to use it as Content-Type header
					columns.add(getSelectListItem(mediaTypeElementName));
				} else {
					// use media type from annotation as content type
					contentType = StreamUtils.getCoreMediaTypeValue(entity, elementName);
				}

				// try to find field containing content disposition filename
				contentDispoElementName = StreamUtils.getContentDispositionElement(entity, elementName);
				if (!StringUtils.isEmpty((contentDispoElementName))) {
					columns.add(getSelectListItem(contentDispoElementName));
				} else {
					contentDispoFilename = StreamUtils.getContentDispositionValue(entity, elementName);
				}

				// try to get value value of annotation containing content disposition type
				contentDispoType = StreamUtils.getContentDispositionTypeValue(entity, elementName);

				select.columns(columns);
			} else {
				select.columns(aliasedGet(elementName));
			}
			break;
		case count:
			select.columns(CQL.count().as("count"));
			break;
		case value:
			UriResourcePartTyped lastTypedResource = request.getLastTypedResource();
			if(lastTypedResource.getKind() == UriResourceKind.primitiveProperty) {
				EdmProperty valueProperty = ((UriResourcePrimitiveProperty) lastTypedResource).getProperty();
				String valueElementName = requestMapper.remap(valueProperty.getName(), entity);
				select.columns(aliasedGet(valueElementName));
			} else {
				throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_URI_RESOURCE, lastTypedResource.getKind());
			}
			break;
		default:
			selects.forEach(s -> queryLoader.updateSelectColumns(s, uriInfo, entity)); // TODO entity -> rowType
		}

		boolean hasETag = ETagHelper.hasETag(entity);
		boolean keysSpecified = false;
		if (select.from().isRef()) {
			AnalysisResult analysis = CqnAnalyzer.create(globals.getModel()).analyze(select.ref());
			long keyCount = analysis.targetEntity().keyElements().count();
			keysSpecified = keyCount > 0 && analysis.targetKeyValues().size() == keyCount;
		}

		boolean readById = keysSpecified || !request.getLastEntityResource(true).isCollection();
		Result result;
		if (readById) {
			Select<?> readByIdSelect = selects.get(0);
			result = readById(request, applicationService, entity, parameters, readByIdSelect, hasETag);
			if (result == NOT_MODIFIED) {
				return new CdsODataResponse(SC_NOT_MODIFIED, result);
			}

			// use media type from corresponding field as content type, if available
			if (!StringUtils.isEmpty(mediaTypeElementName)) {
				contentType = (String) DataUtils.getOrDefault(result.single(), mediaTypeElementName, null);
			}
			if (!StringUtils.isEmpty(contentDispoElementName)) {
				contentDispoFilename = (String) DataUtils.getOrDefault(result.single(), contentDispoElementName, null);
			}
			if (remap) {
				result = remapResult(result, entity, readByIdSelect);
			}
			return new CdsODataResponse(SC_OK, result, null, contentType, contentDispoFilename, contentDispoType);
		} else {
			QueryLimitUtils queryLimitHandler = new QueryLimitUtils(cdsServiceModel, entity, uriInfo,
					cdsProperties.getQuery().getLimit());
			queryLimitHandler.handlePagination(selects);

			boolean analyticsQuery = uriInfo.getApplyOption() != null;
			Pair<Result, CqnSelect> resultPair = query(applicationService, entity, parameters, selects, analyticsQuery, remap);

			NextLinkInfo nextLinkInfo = queryLimitHandler.generateNextLink(resultPair.getLeft(), resultPair.getRight());
			return new CdsODataResponse(SC_OK, resultPair.getLeft(), nextLinkInfo, contentType, contentDispoFilename, contentDispoType);
		}
	}

	private Result readById(CdsODataRequest request, ApplicationService applicationService,
			CdsEntity entity, Map<String, Object> parameters, Select<?> select, boolean hasETag) {
		if (hasETag && ETagHelper.isETagHeaderInRequest(request)) {
			if(ETagHelper.hasStar(request, HttpHeaders.IF_NONE_MATCH)) {
				throw new ErrorStatusException(CdsErrorStatuses.ETAG_FAILED);
			}
			select = addWhere(select, ETagHelper.getETagPredicate(request, entity));
		}

		Result result = applicationService.run(select, parameters);
		long rowCount = result.rowCount();

		if (hasETag && ETagHelper.isETagHeaderInRequest(request) && rowCount == 0) {
			if (request.getHeader(HttpHeaders.IF_NONE_MATCH) != null) {
				return NOT_MODIFIED;
			}
			throw new ErrorStatusException(CdsErrorStatuses.ETAG_FAILED);
		}

		if (rowCount == 0) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_INSTANCE_NOT_FOUND, entity.getQualifiedName(),
					CdsModelUtils.getTargetKeysAsString(this.globals.getModel(), select));
		}
		return result;
	}

	private Pair<Result, CqnSelect> query(ApplicationService applicationService, CdsEntity entity, Map<String, Object> parameters,
					   List<Select<?>> selects, boolean analyticsQuery, boolean remap) {
		List<Row> allRows = new ArrayList<>();
		long[] inlineCount = new long[selects.size()];
		CqnSelect statement = null;

		for (int i = 0; i < selects.size(); i++) {
			Select<?> select = selects.get(i);
			CdsReadEventContext readEvent = CdsReadEventContext.create(entity.getQualifiedName());
			readEvent.setCqn(select);
			readEvent.setCqnNamedValues(parameters);
			applicationService.emit(readEvent);

			Result result = readEvent.getResult();
			inlineCount[i] = result.inlineCount();
			if (analyticsQuery) {
				Set<String> keys = select.items().stream().filter(sli -> sli.isValue())
					.map(sli -> sli.asValue().displayName()).collect(toSet());
				result.forEach(r -> addNullsForOmittedValues(r, keys));
			}
			if (remap) {
				result = remapResult(result, entity, select);
			}
			result.forEach(allRows::add);
			// statement is extracted only in case of single statement
			statement = i == 0 ? readEvent.getCqn() : null;
		}

		return Pair.of(ResultBuilder.selectedRows(allRows).inlineCount(inlineCount[0]).result(), statement);
	}

	private static void addNullsForOmittedValues(Map<String, Object> r, Set<String> keys) {
		for (String key : keys) {
			addNullForOmittedValue(r, key);
		}
	}

	@SuppressWarnings("unchecked")
	private static void addNullForOmittedValue(Map<String, Object> r, String key) {
		int i = key.indexOf('.');
		if (i == -1) {
			if (!r.containsKey(key)) {
				r.put(key, null);
			}
		} else {
			String seg = key.substring(0, i);
			Map<String, Object> inner = (Map<String, Object>) r.computeIfAbsent(seg, s -> new HashMap<String, Object>());
			addNullForOmittedValue(inner, key.substring(i + 1));
		}
	}

	private CdsODataResponse post(CdsODataRequest request) {
		ApplicationService applicationService = globals.getApplicationService();
		CdsEntity entity = request.getLastEntity();
		Map<String, Object> entityData = request.getBodyMap();

		StructuredType<?> ref = toPathExpression(request.getUriInfo().getUriResourceParts(), new HashMap<>());
		// remove the final filter, as CDS4J does not support it on CqnInsert statements
		// a filter might occur, if the post was triggered as an upsert operation during patch
		ref.filter((CqnPredicate) null);

		Result result;
		Insert insert = Insert.into(ref).entry(entityData);
		Object isActiveEntity = entityData.get(Drafts.IS_ACTIVE_ENTITY);
		if (DraftUtils.isDraftEnabled(entity) && (isActiveEntity == null || !(Boolean) isActiveEntity)) {
			result = ((DraftService) applicationService).newDraft(insert);
		} else {
			result = applicationService.run(insert);
		}
		return new CdsODataResponse(SC_CREATED, remapResult(result, entity, null));
	}

	private CdsODataResponse put(CdsODataRequest request) {
		CdsEntity entity = request.getLastEntity();
		Map<String, Object> entityData = request.getBodyMap();

		// fill up unspecified values
		SessionContext sessionContext = SessionContextUtils.toSessionContext(RequestContext.getCurrent(globals.getRuntime()));
		DataUtils dataUtils = DataUtils.create(() -> sessionContext, 7);
		CdsDataProcessor processor = DataProcessor.create() //
				.addGenerator((p, e, t) -> DataUtils.hasDefaultValue(e, t),
						(p, e, isNull) -> isNull ? null : dataUtils.defaultValue(e))
				.action(CdsProcessor::defaultToNull);
		processor.process(entityData, entity);

		// run a default patch
		return patch(request);
	}

	private static void defaultToNull(CdsStructuredType type, Map<String, Object> data) {
		List<String> potentialFkElements = new ArrayList<>();
		Map<String, CdsElement> associations = ElementUtils.recursiveElements(type, e -> e.getType().isAssociation());
		associations.forEach((path, assoc) -> {
			int firstDot = path.indexOf(".");
			assoc.getType().as(CdsAssociationType.class).onCondition().ifPresent(p -> p.accept(new CqnVisitor() {
				@Override
				public void visit(CqnElementRef elementRef) {
					Stream<String> ids = elementRef.stream().map(Segment::id);
					if (firstDot >= 0) {
						String parentPath = path.substring(firstDot + 1);
						ids = Stream.concat(Stream.of(parentPath), ids);
					}
					String fk = ids.collect(joining("."));
					potentialFkElements.add(fk);
				};
			}));
		});

		// do not set default null values for keys, foreign keys and associations
		ElementUtils.recursiveElements(type, e -> !e.isKey()
				&& !e.getType().isAssociation()
				&& CdsAnnotations.ODATA_FOREIGN_KEY_FOR.getOrDefault(e) == null)
				.keySet().stream()
				.filter(e -> !potentialFkElements.contains(e))
				.filter(e -> !DataUtils.containsKey(data, e, true))
				.forEach(e -> DataUtils.putPath(data, e, null));
	}

	@SuppressWarnings("unchecked")
	private CdsODataResponse deltaCollection(CdsODataRequest request) {
		ApplicationService applicationService = globals.getApplicationService();

		StructuredType<?> ref = toPathExpression(request.getUriInfo().getUriResourceParts(), new HashMap<>());
		Map<String, Object> bodyMap = request.getBodyMap();
		List<Map<String, Object>> deleteData = (List<Map<String, Object>>) bodyMap.get(DELTA_DELETE);
		if (!deleteData.isEmpty()) {
			CdsEntity entity = request.getLastEntity();
			CqnDelete delete = Delete.from(ref).byParams(com.sap.cds.util.CdsModelUtils.keyNames(entity));
			applicationService.run(delete, deleteData);
		}
		List<Map<String, Object>> upsertData = (List<Map<String, Object>>) bodyMap.get(DELTA_UPSERT);
		if (!upsertData.isEmpty()) {
			CqnUpsert upsert = Upsert.into(ref).entries(upsertData);
			applicationService.run(upsert);
		}
		// TODO Support Delta Response: The response, if requested, is a delta payload, in the same structure
		// and order as the request payload, representing the applied changes.
		return new CdsODataResponse(SC_NO_CONTENT, ResultBuilder.insertedRows(Collections.emptyList()).result());
	}

	@SuppressWarnings("unchecked")
	private CdsODataResponse patch(CdsODataRequest request) {
		ApplicationService applicationService = globals.getApplicationService();
		CdsEntity entity = request.getLastEntity();
		Map<String, Object> entityData = request.getBodyMap();

		// add target keys from the URL to the payload
		// this also ensures that target key values are not updatable
		StructuredType<?> ref = toPathExpression(request.getUriInfo().getUriResourceParts(), new HashMap<>());
		AnalysisResult analysis = CqnAnalyzer.create(globals.getModel()).analyze(ref.asRef());
		entityData.putAll(analysis.targetKeyValues());

		boolean isStreamProperty = false;
		boolean isCollectionProperty = false;
		UriResource lastResource = request.getLastResource();

		String elementName = null;
		if (lastResource instanceof UriResourceProperty property) {
			EdmProperty edmProperty = property.getProperty();
			isCollectionProperty = edmProperty.isCollection();
			elementName = requestMapper.remap(edmProperty.getName(), entity);

			// if requested property is type Edm.Stream, look also for an element
			if (edmProperty.getType().getFullQualifiedName().getFullQualifiedNameAsString().equals("Edm.Stream")) {
				isStreamProperty = true;
				String contentType = request.getHeader(HttpHeaders.CONTENT_TYPE);
				String mediaTypeElement = StreamUtils.getCoreMediaTypeElement(entity, elementName);
				if (mediaTypeElement != null) {
					DataUtils.putPath(entityData, mediaTypeElement, contentType);
				}
			}
		}

		// POST operation on collection property
		if (elementName != null && isCollectionProperty && request.getODataRequest().getMethod().equals(HttpMethod.POST)) {
			CdsODataResponse cdsODataResponse = get(request, false);
			Row row = cdsODataResponse.getResult().single();
			List<Object> existingValues = (List<Object>) row.get(elementName);
			if(existingValues == null) {
				existingValues = new ArrayList<>();
			}
			List<Object> addedValues = DataUtils.getOrDefault(entityData, elementName, new ArrayList<>());
			existingValues.addAll(addedValues);
			DataUtils.putPath(entityData, elementName, existingValues);
		}

		long rowCount = 0;
		Result updateResult = null;
		// with If-None-Match: *, the PATCH/PUT request must not perform an Update
		// but only have Insert behaviour.
		// this check is also active, even if the entity is not ETag-enabled
		boolean ifNoneMatchStar = ETagHelper.hasStar(request, HttpHeaders.IF_NONE_MATCH);
		if (!ifNoneMatchStar) {
			CqnUpdate update = Update.entity(ref).data(entityData);

			boolean hasETag = ETagHelper.hasETag(entity);
			if (hasETag) {
				if (ETagHelper.isETagHeaderInRequest(request)) {
					update = addWhere(update, ETagHelper.getETagPredicate(request, entity));
				} else {
					throw new ErrorStatusException(CdsErrorStatuses.ETAG_REQUIRED);
				}
			}

			if (DraftUtils.isDraftTarget(update.ref(), entity, globals.getModel())) {
				updateResult = ((DraftService) applicationService).patchDraft(update);
			} else {
				updateResult = applicationService.run(update);
			}
			rowCount = updateResult.rowCount();

			if(hasETag && rowCount == 0) {
				throw new ErrorStatusException(CdsErrorStatuses.ETAG_FAILED);
			}
		}

		if(rowCount == 0) {
			// an insert of a streamed property fails if the entity doesn't exist yet,
			// because the stream is already consumed for the update statement
			if (lastResource instanceof UriResourcePrimitiveProperty
					|| lastResource instanceof UriResourceValue) {
				throw new ErrorStatusException(CdsErrorStatuses.ENTITY_INSTANCE_NOT_FOUND, entity.getQualifiedName(),
						CdsModelUtils.getTargetKeysAsString(this.globals.getModel(), ref.asRef()));
			}
			// with If-Match: *, the PATCH/PUT request must not have upsert behaviour
			// this check is also active, even if the entity is not ETag-enabled
			if (ETagHelper.hasStar(request, HttpHeaders.IF_MATCH)) {
				throw new ErrorStatusException(CdsErrorStatuses.ETAG_FAILED);
			}
			// trigger an insert
			try {
				return post(request);
			} catch (ServiceException e) {
				// with If-None-Match: *, 409 Conflict errors are interpreted as 412
				// Precondition Failed, as an entry unexpectedly existed already
				if (ifNoneMatchStar && e.getErrorStatus() == CdsErrorStatuses.UNIQUE_CONSTRAINT_VIOLATED) {
					throw new ErrorStatusException(CdsErrorStatuses.ETAG_FAILED, e);
				}
				throw e;
			}
		} else {
			if (request.getReturnPreference() != Return.MINIMAL && !isStreamProperty) {
				Map<String, Object> parameters = new HashMap<>();
				Select<?> select = Select.from(toPathExpression(request.getUriInfo().getUriResourceParts(), parameters));
				// no need for server-driven paging, as we are processing a read by key
				select = SystemQueryLoader.create(requestMapper, globals.getEdmxFlavour(), cdsProperties).updateSelect(select, request.getUriInfo(), entity);
				Result readResult = applicationService.run(select, parameters);
				return new CdsODataResponse(SC_OK, remapResult(readResult, entity, select));
			} else {
				Map<String, Object> targetKeys = CqnAnalyzer.create(globals.getModel()).analyze(ref.asRef()).targetKeyValues();
				updateResult.single().putAll(targetKeys);
				return new CdsODataResponse(SC_NO_CONTENT, remapResult(updateResult, entity, null));
			}
		}
	}

	private CdsODataResponse delete(CdsODataRequest request) {
		ApplicationService applicationService = globals.getApplicationService();
		CqnDelete delete = Delete.from(toPathExpression(request.getUriInfo().getUriResourceParts(), new HashMap<>()));

		CdsEntity entity = request.getLastEntity();
		boolean hasETag = ETagHelper.hasETag(entity);
		if (hasETag) {
			if (ETagHelper.isETagHeaderInRequest(request)) {
				if(ETagHelper.hasStar(request, HttpHeaders.IF_NONE_MATCH)) {
					throw new ErrorStatusException(CdsErrorStatuses.ETAG_FAILED);
				}
				delete = addWhere(delete, ETagHelper.getETagPredicate(request, entity));
			} else {
				throw new ErrorStatusException(CdsErrorStatuses.ETAG_REQUIRED);
			}
		}

		Result result;
		if (DraftUtils.isDraftTarget(delete.ref(), entity, globals.getModel())) {
			result = ((DraftService) applicationService).cancelDraft(delete);
		} else {
			result = applicationService.run(delete);
		}

		if (result.rowCount() == 0) {
			if (hasETag) {
				throw new ErrorStatusException(CdsErrorStatuses.ETAG_FAILED);
			}
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_INSTANCE_NOT_FOUND, entity.getQualifiedName(),
					CdsModelUtils.getTargetKeysAsString(this.globals.getModel(), delete));
		}
		return new CdsODataResponse(SC_NO_CONTENT);
	}

	@SuppressWarnings("unchecked")
	private CdsODataResponse operation(CdsODataRequest request) {
		EventContext context;
		EdmOperation edmOperation = edmUtils.getEdmOperation(request.getLastTypedResource());
		if(edmOperation.isBound()) {
			CdsEntity entity = request.getLastEntity();
			validateETagForWriteWithSelect(request, entity);

			context = EventContext.create(edmOperation.getName(), entity.getQualifiedName());
			context.put("cqn", Select.from(toPathExpression(request.getUriInfo().getUriResourceParts(), new HashMap<>())));
		} else {
			context = EventContext.create(edmOperation.getName(), null);
		}

		request.getBodyMap().forEach(context::put);
		globals.getApplicationService().emit(context);

		Result result = null;
		Object returnValue = context.get("result");
		CdsType returnType = edmUtils.getCdsOperationReturnType(edmUtils.getCdsOperation(request));
		if(returnValue != null) {
			try {
				if (returnType != null && returnType.isSimple()) {
					Map<String, Object> data = new HashMap<>();
					data.put(edmOperation.getName(), returnValue);
					result = ResultUtils.convert(ListUtils.getList(data));
				} else if(returnValue instanceof Iterable) {
					result = ResultUtils.convert((Iterable<Map<String, Object>>) returnValue);
				} else if (returnValue instanceof Map) {
					result = ResultUtils.convert(ListUtils.getList((Map<String, Object>) returnValue));
				} else {
					throw new ErrorStatusException(CdsErrorStatuses.UNSUPPORTED_ACTION_FUNCTION_RETURN_TYPE, returnValue.getClass().getName());
				}
			} catch (ClassCastException e) {
				throw new ErrorStatusException(CdsErrorStatuses.UNSUPPORTED_ACTION_FUNCTION_RETURN_TYPE, returnValue.getClass().getName(), e);
			}
		}

		if (returnType instanceof CdsStructuredType && result != null) {
			// trigger additional read request, if expand and select options ask for more than the action returned
			Select<?> select = null;
			if (returnType instanceof CdsEntity) {
				CdsEntity returnTypeEntity = returnType.as(CdsEntity.class);
				// not all requested expand and select properties contained in result
				if (!matchesSelectExpand(result.list(), returnTypeEntity, request.getUriInfo().getSelectOption(), request.getUriInfo().getExpandOption())) {
					ApplicationService applicationService = globals.getApplicationService();
					select = Select.from(returnTypeEntity);
					// not clear what limits applied to a modifying action request mean
					select = SystemQueryLoader.create(requestMapper, globals.getEdmxFlavour(), cdsProperties).updateSelect(select, request.getUriInfo(), returnTypeEntity);

					CqnSelect additionalSelect = addWhere(select, fetchKeyValuesFromResult(result, returnTypeEntity));
					result = applicationService.run(additionalSelect); // NOSONAR
				}
			}
			result = remapResult(result, returnType.as(CdsStructuredType.class), select);
		}

		return new CdsODataResponse(SC_OK, result);
	}

	@SuppressWarnings("unchecked")
	private boolean matchesSelectExpand(List<? extends Map<String, Object>> result, CdsEntity entity, SelectOption selectOption, ExpandOption expandOption) {
		boolean matches = true;

		if (selectOption != null) {
			matches = selectOption.getSelectItems().stream().allMatch((selectItem) -> {
				if (selectItem.isStar()) {
					Set<String> nonAssociationElementNames = ElementUtils.recursiveElements(entity, e -> !e.getType().isAssociation()).keySet();
					return result.stream().allMatch(map -> containsAll(map, nonAssociationElementNames));
				} else if (selectItem.getResourcePath() != null) {
					List<UriResource> uriResources = selectItem.getResourcePath().getUriResourceParts();
					if (uriResources.size() > 1) {
						throw new ErrorStatusException(CdsErrorStatuses.SELECT_PARSING_FAILED);
					}
					String selectElement = requestMapper.remap(uriResources.get(0).getSegmentValue(), entity);
					return result.stream().allMatch(map -> DataUtils.containsKey(map, selectElement));
				}
				return true;
			});
		}

		if (matches && expandOption != null) {
			matches = expandOption.getExpandItems().stream().allMatch((expandItem) -> {
				if (expandItem.isStar()) {
					Set<String> associationElementNames = ElementUtils.recursiveElements(entity, e -> e.getType().isAssociation()).keySet();
					return result.stream().allMatch(map -> containsAll(map, associationElementNames));
				} else if (expandItem.getResourcePath() != null) {
					List<UriResource> uriResources = expandItem.getResourcePath().getUriResourceParts();
					String expandElement = requestMapper.remap(uriResources.get(uriResources.size() - 1).getSegmentValue(), entity);
					boolean expandItemMatches = result.stream().allMatch(map -> DataUtils.containsKey(map, expandElement));
					if(expandItemMatches) {
						CdsEntity expandEntity = entity.getTargetOf(expandElement);
						List<Map<String, Object>> expandResult = result.stream().map(map -> DataUtils.getOrDefault(map, expandElement, null)).flatMap(v -> {
							if(v instanceof Map) {
								return Stream.of((Map<String, Object>) v);
							} else if (v instanceof Iterable) {
								return StreamSupport.stream(((Iterable<Map<String, Object>>) v).spliterator(), false);
							}
							return Stream.empty();
						}).collect(toList());
						// recursively check all expanded items
						return matchesSelectExpand(expandResult, expandEntity, expandItem.getSelectOption(), expandItem.getExpandOption());
					}
					return false;
				}
				return true;
			});
		}

		return matches;
	}

	private static boolean containsAll(Map<String, Object> data, Set<String> paths) {
		return data.keySet().containsAll(paths) || paths.stream().allMatch(p -> DataUtils.containsKey(data, p));
	}

	private Predicate fetchKeyValuesFromResult(Result result, CdsEntity entity) {
		List<String> keyElements = entity.keyElements().map(k -> k.getName()).collect(toList());
		Predicate predicate = CQL.FALSE;
		// TODO use <key refs> in list<key values> once supported in CDS4j
		for (Row row : result.list()) {
			Predicate keys = CQL.TRUE;
			for (String key : keyElements) {
				keys = keys.and(CQL.get(key).eq(row.get(key)));
			}
			predicate = predicate.or(keys);
		}
		return predicate;
	}

	private StructuredType<?> toPathExpression(List<UriResource> uriResources, Map<String, Object> parameters) {
		if(uriResources.isEmpty()) {
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_URI_RESOURCE);
		}

		StructuredType<?> parent = null;
		String currentEntityName = null;
		UriResource rootResource = uriResources.get(0);
		if(rootResource instanceof UriResourceEntitySet rootEntitySetResource) {
			Map<String, Object> keys = getFilterKeys(rootEntitySetResource.getKeyPredicates(), rootEntitySetResource.getEntityType().getKeyPropertyRefs());
			if(edmUtils.isParametersEntityType(rootEntitySetResource.getEntityType())) {
				parameters.putAll(keys); // parameters of parameterized views are always primitive elements, never structured
			} else {
				currentEntityName = edmUtils.getCdsEntityName(rootEntitySetResource.getEntityType());
				Map<String, Object> mappedKeys = requestMapper.remap(keys, globals.getModel().getEntity(currentEntityName));
				parent = CQL.entity(currentEntityName).matching(mappedKeys);
			}
		} else if (rootResource instanceof UriResourceSingleton rootSingletonResource) {
			currentEntityName = edmUtils.getCdsEntityName(rootSingletonResource.getEntityType());
			parent = CQL.entity(currentEntityName);
		} else {
			throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_URI_RESOURCE, rootResource.getKind());
		}

		for (int i=1; i<uriResources.size(); i++) {
			UriResource resource = uriResources.get(i);
			if (resource instanceof UriResourceNavigation navigationResource) {
				EdmEntityType navigationType = navigationResource.getProperty().getType();
				Map<String, Object> keys = getFilterKeys(navigationResource.getKeyPredicates(), navigationType.getKeyPropertyRefs());
				if(edmUtils.isParametersEntityType(navigationType)) {
					parameters.putAll(keys); // parameters of parameterized views are always primitive elements, never structured
				} else {
					CdsEntity parentEntity = currentEntityName != null ? globals.getModel().getEntity(currentEntityName) : null;
					currentEntityName = edmUtils.getCdsEntityName(navigationType);
					if(parent == null) {
						parent = CQL.entity(currentEntityName).matching(keys);
					} else {
						String associationName = requestMapper.remap(navigationResource.getSegmentValue(), parentEntity);
						Map<String, Object> mappedKeys = requestMapper.remap(keys, globals.getModel().getEntity(currentEntityName));
						parent = parent.to(associationName).matching(mappedKeys);
					}
				}
			} else {
				break;
			}
		}

		if(parent == null) {
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_PARAMETERIZED_VIEW);
		}

		return parent;
	}

	private Map<String, Object> getFilterKeys(List<UriParameter> keyPreds, List<EdmKeyPropertyRef> keyRefs) {
		Map<String, EdmType> elementTypes = keyRefs.stream().collect(toMap(k -> k.getName(), k -> k.getProperty().getType()));
		Map<String, Object> keys = new HashMap<>();
		for (UriParameter key : keyPreds) {
			String name = key.getName();
			String text = null;
			if (key.getAlias() != null) {
				// Key predicates support only literals
				if (key.getExpression() instanceof Literal literal) {
					text = literal.getText();
				}
			} else {
				text = key.getText();
			}
			keys.put(name, TypeConverterUtils.convertToType(elementTypes.get(name), text));
		}
		return keys;
	}

	private void validateETagForWriteWithSelect(CdsODataRequest request, CdsEntity cdsEntity) {
		boolean hasETag = ETagHelper.hasETag(cdsEntity);
		if (hasETag) {
			if (ETagHelper.isETagHeaderInRequest(request)) {
				Map<String, Object> parameters = new HashMap<>();
				// TODO enable .lock() once CDS4j supports it
				CqnSelect select = Select.from(toPathExpression(request.getUriInfo().getUriResourceParts(), parameters));
				CqnSelect etagSelect = addWhere(select, ETagHelper.getETagPredicate(request, cdsEntity));

				Result queryResult = globals.getRuntime().requestContext().clearMessages().run((requestContext) -> {
					return globals.getApplicationService().run(etagSelect, parameters);
				});

				long rowCount = queryResult.rowCount();
				boolean ifNoneMatchStar = ETagHelper.hasStar(request, HttpHeaders.IF_NONE_MATCH);
				if ((rowCount == 0 && !ifNoneMatchStar) || (rowCount > 0 && ifNoneMatchStar)) {
					throw new ErrorStatusException(CdsErrorStatuses.ETAG_FAILED);
				}
			} else {
				throw new ErrorStatusException(CdsErrorStatuses.ETAG_REQUIRED);
			}
		}
	}

	private Result remapResult(Result result, CdsStructuredType entryType, CqnSelect select) {
		return responseMapper.remap(result, entryType, (path) -> select != null ? isExpanded(path, select.items()) : false);
	}

	private boolean isExpanded(String path, List<CqnSelectListItem> items) {
		for (CqnSelectListItem item : items) {
			if (item.isExpand()) {
				CqnExpand expand = item.asExpand();
				String displayName = expand.displayName();
				if (path.equals(displayName)) {
					return true;
				} else if (path.startsWith(displayName + ".")) {
					String remainingPath = path.substring(displayName.length() + 1);
					return isExpanded(remainingPath, expand.items());
				}
			}
		}
		return false;
	}

	/*
	 * Returns a CqnSelectListItem for the given element path. Element path could
	 * consists of several path segments separated by a dot If path consists of one
	 * element returns an ElementRef, otherwise an Expand.
	 *
	 * examples: "filename", "details.filename", "details.metadata.filename"
	 */
	protected static CqnSelectListItem getSelectListItem(String elementPath) {
		if (StringUtils.isEmpty(elementPath)) {
			return null;
		}

		String[] segments = elementPath.split("\\.");
		int length = segments.length;

		if (length == 1) {
			return CQL.get(segments[0]);
		} else if (length == 2) {
			return CQL.to(segments[0]).expand(segments[1]);
		} else {
			Expand<?> expand = CQL.to(segments[length - 2]).expand(segments[length - 1]);
			for (int i = length - 3; i >= 0; i--) {
				expand = CQL.to(segments[i]).expand(expand);
			}
			return expand;
		}
	}

}
