/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.processors;

import static com.sap.cds.adapter.odata.v4.utils.ODataUtils.preferences;
import static org.apache.olingo.commons.api.http.HttpHeader.PREFER;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.net.URI;
import java.nio.channels.Channels;
import java.nio.channels.WritableByteChannel;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.apache.commons.io.input.ReaderInputStream;
import org.apache.http.HttpStatus;
import org.apache.olingo.commons.api.data.ContextURL;
import org.apache.olingo.commons.api.edm.EdmEntityType;
import org.apache.olingo.commons.api.edm.EdmStructuredType;
import org.apache.olingo.commons.api.format.ContentType;
import org.apache.olingo.commons.api.http.HttpHeader;
import org.apache.olingo.commons.api.http.HttpMethod;
import org.apache.olingo.server.api.ODataContent;
import org.apache.olingo.server.api.ODataRequest;
import org.apache.olingo.server.api.ODataResponse;
import org.apache.olingo.server.api.prefer.Preferences.Return;
import org.apache.olingo.server.api.uri.UriInfo;
import org.apache.olingo.server.api.uri.UriResource;
import org.apache.olingo.server.api.uri.UriResourceKind;
import org.apache.olingo.server.api.uri.UriResourceNavigation;
import org.apache.olingo.server.api.uri.UriResourcePartTyped;
import org.apache.olingo.server.api.uri.queryoption.ExpandOption;
import org.apache.olingo.server.core.deserializer.helper.ExpandTreeBuilderImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.Base64Variants;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.sap.cds.Result;
import com.sap.cds.Row;
import com.sap.cds.Struct;
import com.sap.cds.adapter.odata.v4.CdsRequestGlobals;
import com.sap.cds.adapter.odata.v4.processors.response.ResultSetProcessor;
import com.sap.cds.adapter.odata.v4.query.NextLinkInfo;
import com.sap.cds.adapter.odata.v4.serializer.json.Apply2Json;
import com.sap.cds.adapter.odata.v4.serializer.json.Apply2JsonBuilder;
import com.sap.cds.adapter.odata.v4.serializer.json.Complex2Json;
import com.sap.cds.adapter.odata.v4.serializer.json.Complex2JsonBuilder;
import com.sap.cds.adapter.odata.v4.serializer.json.ComplexCollection2Json;
import com.sap.cds.adapter.odata.v4.serializer.json.ComplexCollection2JsonBuilder;
import com.sap.cds.adapter.odata.v4.serializer.json.Entity2Json;
import com.sap.cds.adapter.odata.v4.serializer.json.Entity2JsonBuilder;
import com.sap.cds.adapter.odata.v4.serializer.json.EntityCollection2Json;
import com.sap.cds.adapter.odata.v4.serializer.json.EntityCollection2JsonBuilder;
import com.sap.cds.adapter.odata.v4.serializer.json.Primitive2Json;
import com.sap.cds.adapter.odata.v4.serializer.json.Primitive2JsonBuilder;
import com.sap.cds.adapter.odata.v4.serializer.json.StructTypeHelper;
import com.sap.cds.adapter.odata.v4.serializer.json.api.Data2Json;
import com.sap.cds.adapter.odata.v4.serializer.json.api.PropertyInfo;
import com.sap.cds.adapter.odata.v4.serializer.json.options.Apply2JsonOptions;
import com.sap.cds.adapter.odata.v4.serializer.json.options.Primitive2JsonOptions;
import com.sap.cds.adapter.odata.v4.serializer.json.options.Struct2JsonOptions;
import com.sap.cds.adapter.odata.v4.serializer.json.options.StructCollection2JsonOptions;
import com.sap.cds.adapter.odata.v4.utils.ChangeSetContextAwareInputStream;
import com.sap.cds.adapter.odata.v4.utils.ETagHelper;
import com.sap.cds.adapter.odata.v4.utils.EdmUtils;
import com.sap.cds.adapter.odata.v4.utils.ODataUtils;
import com.sap.cds.impl.parser.JsonParser;
import com.sap.cds.services.changeset.ChangeSetContextSPI;
import com.sap.cds.services.pdf.PdfDocumentDescription;
import com.sap.cds.services.pdf.PdfService;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.StringUtils;

public class FastODataProcessor extends AbstractODataProcessor {

	private static final Logger logger = LoggerFactory.getLogger(FastODataProcessor.class);
	private static final String SAP_DOCUMENT_DESCRIPTION = "SAP-Document-Description";
	private static final JsonFactory jsonFactory = new JsonFactory()
			// prevent values to be written using scientific notation
			.configure(JsonGenerator.Feature.WRITE_BIGDECIMAL_AS_PLAIN, true);

	public FastODataProcessor(CdsRequestGlobals globals) {
		super(globals);
		if (isBuffered) {
			logger.debug("OData v4 JSON serializer is using buffered mode");
		}
	}

	@Override
	public void processEntity(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo,
			ContentType requestFormat, ContentType contentType) {
		processRequest(odataRequest, odataResponse, uriInfo, requestFormat, (cdsRequest, cdsResponse) -> {
			ODataContent odataContent = null;
			Row row = cdsResponse.getResult().first().orElse(null);
			if (row != null) {
				String locationPrefixTmp = null;
				boolean calculateKey = false;
				if (odataRequest.getMethod() != HttpMethod.GET
						&& cdsRequest.getLastResource() == cdsRequest.getLastEntityResource(true)) {
					// prepare location URL
					locationPrefixTmp = odataRequest.getRawBaseUri() + odataRequest.getRawODataPath();
					UriResource lastResource = cdsRequest.getLastResource();
					calculateKey = (!locationPrefixTmp.endsWith(")") &&
							(lastResource.getKind() == UriResourceKind.entitySet ||
									(lastResource.getKind() == UriResourceKind.navigationProperty
											&& ((UriResourceNavigation) lastResource).getProperty().isCollection())));
				}
				final String locationPrefix = locationPrefixTmp;

				EdmEntityType entityType = (EdmEntityType) cdsRequest.getResponseType();
				if (cdsResponse.getStatusCode() < HttpStatus.SC_NO_CONTENT
						&& cdsRequest.getReturnPreference() != Return.MINIMAL) {
					// TODO:
					// checkCountOptionsOnExpand(entity, uriInfo.getExpandOption());

					// action/function: in the original implementation the expand is calculated
					// based on data
					// See, ODataFunctionRequestITTest.testUnBoundFunctionsReturningEntities()
					// also DraftAdminData should be returned in draft scenario
					ExpandOption expand = uriInfo.getExpandOption() != null ? uriInfo.getExpandOption()
							: EdmUtils.createExpand(entityType, row);
					ContextURL contextUrl = getContextUrl(cdsRequest, false, uriInfo.getSelectOption(), expand);


					Struct2JsonOptions options = Struct2JsonOptions
							.with(contentType, ODataUtils.getODataVersion(odataRequest), globals,
									preferences(cdsRequest.getHeader(PREFER)))
							.contextURL(contextUrl)
							.select(uriInfo.getSelectOption())
							.expand(expand)
							.autoExpand(true)
							.build();

					Entity2Json entity2Json = Entity2JsonBuilder.createRoot(options, entityType, contentType);
					// Set location Header
					setLocationHeader(odataResponse, locationPrefix,
							calculateKey ? entity2Json.keyPredicate().apply(row) : null);
					odataContent = new JsonRowODataContent(entity2Json, row);
				} else {
					setLocationHeader(odataResponse, locationPrefix, ResultSetProcessor.getEntityId(entityType, row));
				}

				String eTag = ETagHelper.getEtagValue(globals, entityType, row);
				if (eTag != null) {
					odataResponse.setHeader(HttpHeader.ETAG, eTag);
				}
			}
			setODataResponse(cdsRequest, cdsResponse, odataResponse, odataContent, contentType);
		});
	}

	@Override
	public void processEntities(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo,
			ContentType requestFormat, ContentType contentType) {
		processRequest(odataRequest, odataResponse, uriInfo, requestFormat, (cdsRequest, cdsResponse) -> {
			ODataContent odataContent = null;
			Result result = cdsResponse.getResult();
			URI nextLink = null;
			NextLinkInfo nextLinkInfo = cdsResponse.getNextLinkInfo();
			if (nextLinkInfo != null) {
				nextLink = nextLinkInfo.getNextLink(odataRequest);
			}
			EdmEntityType entityType = (EdmEntityType) cdsRequest.getResponseType();
			// Calculate deep expand only for actions and functions
			ExpandOption expand = uriInfo.getExpandOption() != null ? uriInfo.getExpandOption()
					: isActionOrFunction(cdsRequest) ? EdmUtils.createExpand(entityType, result)
							: ExpandTreeBuilderImpl.create().build();
			ContextURL contextUrl = getContextUrl(cdsRequest, true, uriInfo.getSelectOption(), expand);

			if (uriInfo.getApplyOption() != null) {
				Apply2JsonOptions options = Apply2JsonOptions
						.with(contentType, ODataUtils.getODataVersion(odataRequest), globals)
						.contextURL(contextUrl)
						.count(uriInfo.getCountOption())
						.build();
				Apply2Json dynamic2Json = Apply2JsonBuilder.create(options, nextLink, result.inlineCount());
				odataContent = new JsonResultODataContent(dynamic2Json, result);
			} else if (contentType.isCompatible(ContentType.APPLICATION_PDF)) {
				PdfService service = globals.getRuntime().getServiceCatalog()
						.getService(PdfService.class, PdfService.DEFAULT_NAME);
				if (service != null) {
					Map<String, String> headers = globals.getRuntime().getProvidedParameterInfo().getHeaders();
					PdfDocumentDescription docDesc = parsePdfHeader(headers.get(SAP_DOCUMENT_DESCRIPTION));
					odataContent = new PdfContent(service, result, docDesc);
				} else {
					throw new ErrorStatusException(CdsErrorStatuses.PDF_SERVICE_NOT_AVAILABLE);
				}
			} else {
				StructCollection2JsonOptions options = StructCollection2JsonOptions
						.with(contentType, ODataUtils.getODataVersion(odataRequest), globals,
								preferences(cdsRequest.getHeader(PREFER)))
						.contextURL(contextUrl)
						.select(uriInfo.getSelectOption())
						.expand(expand)
						.count(uriInfo.getCountOption())
						.autoExpand(true)
						.build();
				EntityCollection2Json entities2Json = EntityCollection2JsonBuilder.createRoot(
						options, entityType, contentType, nextLink, result.inlineCount());
				odataContent = new JsonResultODataContent(entities2Json, result);
			}
			setODataResponse(cdsRequest, cdsResponse, odataResponse, odataContent, contentType);
		});
	}

	@Override
	public void processSingleComplex(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo,
			ContentType requestFormat, ContentType responseFormat) {
		processRequest(odataRequest, odataResponse, uriInfo, requestFormat, (cdsRequest, cdsResponse) -> {
			ODataContent odataContent = null;

			if (cdsRequest.getReturnPreference() != Return.MINIMAL) {
				Row row = cdsResponse.getResult().single();
				EdmStructuredType entityType = (EdmStructuredType) cdsRequest.getResponseType();
				ExpandOption expand = uriInfo.getExpandOption() != null ? uriInfo.getExpandOption()
						: EdmUtils.createExpand(entityType, row);
				ContextURL contextUrl = getContextUrl(cdsRequest, false, uriInfo.getSelectOption(), expand);

				Struct2JsonOptions options = Struct2JsonOptions
						.with(responseFormat, ODataUtils.getODataVersion(odataRequest), globals,
								preferences(cdsRequest.getHeader(PREFER)))
						.contextURL(contextUrl)
						.select(uriInfo.getSelectOption())
						.expand(expand)
						.build();

				Complex2Json entity2Json = Complex2JsonBuilder.createRoot(options, entityType, responseFormat);
				odataContent = new JsonRowODataContent(entity2Json, row);
			}
			setODataResponse(cdsRequest, cdsResponse, odataResponse, odataContent, responseFormat);
		});
	}

	@Override
	public void processCollectionComplex(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo,
			ContentType requestFormat, ContentType contentType) {
		processRequest(odataRequest, odataResponse, uriInfo, requestFormat, (cdsRequest, cdsResponse) -> {
			ODataContent odataContent = null;
			Result result = cdsResponse.getResult();
			EdmStructuredType entityType = (EdmStructuredType) cdsRequest.getResponseType();
			ExpandOption expand = uriInfo.getExpandOption() != null ? uriInfo.getExpandOption()
					: isActionOrFunction(cdsRequest) ? EdmUtils.createExpand(entityType, result)
							: ExpandTreeBuilderImpl.create().build();

			ContextURL contextURL = getContextUrl(cdsRequest, true, uriInfo.getSelectOption(), expand);
			if (cdsRequest.getReturnPreference() != Return.MINIMAL) {
				PropertyInfo complexPropertyInfo = StructTypeHelper.getPropertyInfo(cdsRequest);

				URI nextLink = null;
				NextLinkInfo nextLinkInfo = cdsResponse.getNextLinkInfo();
				if (nextLinkInfo != null) {
					nextLink = nextLinkInfo.getNextLink(odataRequest);
				}
				
				StructCollection2JsonOptions options = StructCollection2JsonOptions
						.with(contentType, ODataUtils.getODataVersion(odataRequest), globals,
								preferences(cdsRequest.getHeader(PREFER)))
						.contextURL(contextURL)
						.select(uriInfo.getSelectOption())
						.expand(expand)
						.count(uriInfo.getCountOption())
						.build();

				ComplexCollection2Json complex2Json = ComplexCollection2JsonBuilder.createRoot(
						options, entityType, contentType, nextLink, result.inlineCount());

				List<?> payload;
				UriResource resource = cdsRequest.getLastTypedResource();
				if (resource.getKind().equals(UriResourceKind.complexProperty)) {
					Row row = result.single();
					payload = (List<?>) row.get(complexPropertyInfo.getName());
				} else {
					payload = result.list();
				}

				odataContent = new JsonResultODataContent(complex2Json, payload);
			}

			setODataResponse(cdsRequest, cdsResponse, odataResponse, odataContent, contentType);
		});
	}

	@Override
	public void processSinglePrimitive(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo,
			ContentType requestFormat, ContentType responseFormat) {
		final ChangeSetContextSPI changeSetContext;
		boolean isStream = isGetStreamContext(odataRequest, uriInfo, requestFormat);
		if (isStream) {
			changeSetContext = ChangeSetContextSPI.open(); // NOSONAR
			globals.getUnclosedChangeSetTracker().set(true);
		} else {
			changeSetContext = null;
		}

		processRequest(odataRequest, odataResponse, uriInfo, requestFormat, (cdsRequest, cdsResponse) -> {
			ContentType contentType;
			Row row = cdsResponse.getResult().single();
			if (!StringUtils.isEmpty(cdsResponse.getContentType())) {
				contentType = ContentType.create(cdsResponse.getContentType());
			} else {
				contentType = responseFormat;
			}

			ODataContent odataContent = null;
			ContextURL contextURL = getContextUrl(cdsRequest, false, null, null);
			// Simple primitive property action or function
			PropertyInfo property = StructTypeHelper.getPropertyInfo(cdsRequest);

			if (isStream) {
				Data2Json<Map<String, Object>> primitive = StructTypeHelper.createPrimitive(property);
				Object value = primitive.getValue(row);
				if (value != null && cdsRequest.getReturnPreference() != Return.MINIMAL) {
					odataContent = new PrimitiveValueODataContent(value, contentType, changeSetContext);
					setCacheControlHeader(odataResponse, cdsRequest.getLastEntity(), property.getName());
				}
			} else {
				Primitive2JsonOptions options = Primitive2JsonOptions
						.with(contentType, ODataUtils.getODataVersion(odataRequest), globals)
						.contextURL(contextURL)
						.build();
				Primitive2Json primitive2json = Primitive2JsonBuilder.create(options, property, contentType);
				odataContent = new JsonRowODataContent(primitive2json, row);
			}
			setContentDispositionHeaderIfNotNull(odataResponse, cdsResponse.getContentDispositionFilename(), cdsResponse.getContentDispositionType());
			setODataResponse(cdsRequest, cdsResponse, odataResponse, odataContent, contentType);
		});
	}

	@Override
	public void processSinglePrimitiveValue(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo,
			ContentType requestFormat, ContentType responseFormat) {
		processRequest(odataRequest, odataResponse, uriInfo, requestFormat, (cdsRequest, cdsResponse) -> {
			if (isValueRequestOnStreamProperty(cdsRequest)) {
				UriResourcePartTyped lastTypedResource = cdsRequest.getLastTypedResource();
				throw new ErrorStatusException(CdsErrorStatuses.VALUE_ACCESS_NOT_ALLOWED,
						lastTypedResource.getSegmentValue());
			}

			ODataContent odataContent = null;
			Row row = cdsResponse.getResult().single();
			PropertyInfo propertyInfo = StructTypeHelper.getPropertyInfo(cdsRequest);
			Data2Json<Map<String, Object>> primitive = StructTypeHelper.createPrimitive(propertyInfo);
			Object value = primitive.getValue(row);
			if (value != null && cdsRequest.getReturnPreference() != Return.MINIMAL) {
				odataContent = new PrimitiveValueODataContent(value, responseFormat, null);
			}
			setODataResponse(cdsRequest, cdsResponse, odataResponse, odataContent, responseFormat);
		});
	}

	@Override
	public void processCollectionPrimitive(ODataRequest odataRequest, ODataResponse odataResponse, UriInfo uriInfo,
			ContentType requestFormat, ContentType responseFormat) {
		processRequest(odataRequest, odataResponse, uriInfo, requestFormat, (cdsRequest, cdsResponse) -> {
			ODataContent odataContent = null;
			if (cdsRequest.getReturnPreference() != Return.MINIMAL) {
				Row row = cdsResponse.getResult().single();
				ContextURL contextURL = getContextUrl(cdsRequest, true, null, null);
				PropertyInfo property = StructTypeHelper.getPropertyInfo(cdsRequest);

				Primitive2JsonOptions options = Primitive2JsonOptions
						.with(responseFormat, ODataUtils.getODataVersion(odataRequest), globals)
						.contextURL(contextURL)
						.build();
				Primitive2Json primitive2json = Primitive2JsonBuilder.create(options, property, responseFormat);
				odataContent = new JsonRowODataContent(primitive2json, row);
			}
			setContentDispositionHeaderIfNotNull(odataResponse, cdsResponse.getContentDispositionFilename(), cdsResponse.getContentDispositionType());
			setODataResponse(cdsRequest, cdsResponse, odataResponse, odataContent, responseFormat);
		});
	}

	/* HELPERS */
	private interface Executable {
		public abstract void execute() throws IOException;
	}

	private static void serialize(Executable exec, OutputStream out) {
		try {
			exec.execute();
		} catch (Exception t) {
			String message = "ERROR: Failed to serialize payload ";
			logger.error(message, t);
			try {
				out.write(message.getBytes(StandardCharsets.UTF_8));
			} catch (IOException e) {
				throw new ErrorStatusException(CdsErrorStatuses.SERIALIZER_FAILED);
			}
		}
	}

	static final class JsonResultODataContent implements ODataContent {
		final Data2Json<Iterable<Map<String, Object>>> serializer;
		final Iterable<?> result;

		JsonResultODataContent(Data2Json<Iterable<Map<String, Object>>> serializer, Iterable<?> result) {
			this.serializer = serializer;
			this.result = result;
		}

		@Override
		public void write(WritableByteChannel channel) {
			write(Channels.newOutputStream(channel)); // TODO
		}

		@Override
		@SuppressWarnings({ "unchecked", "rawtypes" })
		public void write(OutputStream outputStream) {
			try (JsonGenerator json = jsonFactory.createGenerator(outputStream)) {
				serialize(() -> serializer.toJson((Iterable) result, json), outputStream);
			} catch (IOException t) {
				throw new ErrorStatusException(CdsErrorStatuses.SERIALIZER_FAILED);
			}
		}
	}

	static final class JsonRowODataContent implements ODataContent {
		final Data2Json<Map<String, Object>> serializer;
		final Row row;

		JsonRowODataContent(Data2Json<Map<String, Object>> serializer, Row row) {
			this.serializer = serializer;
			this.row = row;
		}

		@Override
		public void write(WritableByteChannel channel) {
			write(Channels.newOutputStream(channel));
		}

		@Override
		public void write(OutputStream outputStream) {
			try (JsonGenerator json = jsonFactory.createGenerator(outputStream)) {
				serialize(() -> serializer.toJson(row, json), outputStream);
			} catch (IOException e) {
				throw new ErrorStatusException(CdsErrorStatuses.SERIALIZER_FAILED);
			}
		}
	}

	private static final class PdfContent implements ODataContent {
		final PdfService serializer;
		final Result r;
		final PdfDocumentDescription docDesc;

		PdfContent(PdfService serializer, Result r, PdfDocumentDescription docDesc) {
			this.serializer = serializer;
			this.r = r;
			this.docDesc = docDesc;
		}

		@Override
		public void write(WritableByteChannel channel) {
			write(Channels.newOutputStream(channel));
		}

		@Override
		public void write(OutputStream outputStream) {
			try {
				serialize(() -> serializer.export(docDesc, r, outputStream), outputStream);
			} catch (Exception e) {
				throw new ErrorStatusException(CdsErrorStatuses.SERIALIZER_FAILED);
			}
		}
	}

	private final class PrimitiveValueODataContent implements ODataContent {
		private final Object value;
		private final ContentType contentType;
		private final ChangeSetContextSPI changeSetContext;

		PrimitiveValueODataContent(Object value, ContentType contentType,
				ChangeSetContextSPI changeSetContext) {
			this.value = value;
			this.contentType = contentType;
			this.changeSetContext = changeSetContext;
		}

		@Override
		public void write(WritableByteChannel channel) {
			write(Channels.newOutputStream(channel));
		}

		@Override
		@SuppressWarnings("deprecation")
		public void write(OutputStream outputStream) {
			InputStream is = null;
			if (value instanceof Reader reader) {
				is = new ChangeSetContextAwareInputStream(
						new ReaderInputStream(reader, ODataUtils.getCharset(contentType)), changeSetContext,
						globals.getUnclosedChangeSetTracker());
			} else if (value instanceof InputStream stream) {
				is = new ChangeSetContextAwareInputStream(stream, changeSetContext,
						globals.getUnclosedChangeSetTracker());
			} else if( value instanceof byte[] bytes) {
				is = new ByteArrayInputStream(Base64Variants.getDefaultVariant().encode(bytes).getBytes(StandardCharsets.UTF_8));
			} else {
				is = new ByteArrayInputStream(String.valueOf(value).getBytes());
			}
			try (InputStream s = is) {
				serialize(() -> IOUtils.copy(s, outputStream), outputStream);
			} catch (IOException t) {
				throw new ErrorStatusException(CdsErrorStatuses.SERIALIZER_FAILED);
			}
		}
	}

	private static PdfDocumentDescription parsePdfHeader(String header) {
		try {
			if (!StringUtils.isEmpty(header)) {
				String metaJson = new String(Base64.getDecoder().decode(header), StandardCharsets.UTF_8);
				Map<String, Object> pdfMeta = JsonParser.map(JsonParser.parseJson(metaJson));
				return Struct.access(pdfMeta).as(PdfDocumentDescription.class);
			}
		} catch (Exception e) {
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_PDF_DESC_HEADER);
		}
		throw new ErrorStatusException(CdsErrorStatuses.NO_PDF_DESC_HEADER);
	}

}
