/**************************************************************************
 * (C) 2019-2025 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.processors.request;

import static com.sap.cds.adapter.odata.v4.processors.request.CdsODataRequest.DELTA;
import static com.sap.cds.adapter.odata.v4.processors.request.CdsODataRequest.DELTA_DELETE;
import static com.sap.cds.adapter.odata.v4.processors.request.CdsODataRequest.DELTA_UPSERT;
import static com.sap.cds.adapter.odata.v4.utils.TypeConverterUtils.convertToType;
import static java.util.stream.Collectors.toList;

import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import org.apache.olingo.commons.api.data.Delta;
import org.apache.olingo.commons.api.data.Property;
import org.apache.olingo.commons.api.edm.EdmEntityType;
import org.apache.olingo.commons.api.edm.EdmFunction;
import org.apache.olingo.commons.api.edm.EdmParameter;
import org.apache.olingo.commons.api.edm.EdmProperty;
import org.apache.olingo.commons.api.edm.EdmType;
import org.apache.olingo.commons.api.format.ContentType;
import org.apache.olingo.server.api.ODataRequest;
import org.apache.olingo.server.api.deserializer.DeserializerException;
import org.apache.olingo.server.api.deserializer.DeserializerResult;
import org.apache.olingo.server.api.deserializer.FixedFormatDeserializer;
import org.apache.olingo.server.api.deserializer.ODataDeserializer;
import org.apache.olingo.server.api.uri.UriInfo;
import org.apache.olingo.server.api.uri.UriParameter;
import org.apache.olingo.server.api.uri.UriResourceAction;
import org.apache.olingo.server.api.uri.UriResourceFunction;
import org.apache.olingo.server.api.uri.UriResourcePartTyped;
import org.apache.olingo.server.api.uri.queryoption.AliasQueryOption;
import org.apache.olingo.server.api.uri.queryoption.QueryOption;

import com.sap.cds.adapter.odata.v4.CdsRequestGlobals;
import com.sap.cds.adapter.odata.v4.utils.ODataUtils;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

/**
 * Extracts data from request bodies with different content types and returns the data as a Map.
 * It returns the structure based on the EDMX definition.
 * However, types are already mapped to types expected by CDS.
 */
public class RequestBodyExtractor {

	private final CdsRequestGlobals globals;
	private final ODataRequest odataRequest;
	private final ContentType requestFormat;
	private final PayloadProcessor payloadProcessor;

	public RequestBodyExtractor(CdsRequestGlobals globals, ODataRequest odataRequest, ContentType requestFormat) {
		this.globals = globals;
		this.odataRequest = odataRequest;
		this.requestFormat = requestFormat;
		this.payloadProcessor = new PayloadProcessor(globals);
	}


	public Map<String, Object> extractBodyFromProperty(Optional<EdmProperty> edmProperty,
			UriResourcePartTyped resource) {
		if (edmProperty.isPresent()) {
			Map<String, Object> bodyMap = new HashMap<>();
			String name = edmProperty.get().getName();

			if (resource.isCollection()) {
				bodyMap.put(name, new ArrayList<>());
			} else {
				bodyMap.put(name, null);
			}
			return bodyMap;
		} else {
			throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_URI_RESOURCE, resource.getKind());
		}
	}

	public Map<String, Object> extractBodyFromFunctionParameters(UriResourcePartTyped resource, UriInfo uriInfo) {
		Map<String, Object> bodyMap = new HashMap<>();

		EdmFunction function = ((UriResourceFunction) resource).getFunction();
		for (UriParameter parameter : ((UriResourceFunction) resource).getParameters()) {
			EdmParameter edmParameter = function.getParameter(parameter.getName());
			if (parameter.getAlias() != null) {
				// Explicit alias can contain expression, but this is not supported by functions.
				Optional<AliasQueryOption> alias = uriInfo.getAliases().stream()
					.filter(a -> parameter.getAlias().equals(a.getName()))
					.findFirst();
				bodyMap.put(parameter.getName(), parseParameterValue(alias.map(QueryOption::getText).orElse(null), edmParameter));
			} else {
				// Custom query option can contain value that TypeConverterUtils.convertToType(...) cannot handle.
				uriInfo.getCustomQueryOptions().stream()
						.filter(o -> parameter.getName().equals(o.getName())).findFirst().ifPresentOrElse(
								qOpt -> bodyMap.put(qOpt.getName(), parseParameterValue(qOpt.getText(), edmParameter)),
								() -> bodyMap.put(parameter.getName(),
										convertToType(edmParameter.getType(), parameter.getText())));
			}
		}
		return bodyMap;
	}

	public Map<String, Object> extractBodyFromPrimitiveValue(Optional<EdmProperty> edmProperty,
			UriResourcePartTyped resource) {
		if (edmProperty.isPresent()) {
			try {
				FixedFormatDeserializer deserializer = globals.getOData().createFixedFormatDeserializer();
				Object value = deserializer.primitiveValue(odataRequest.getBody(), edmProperty.get());

				Map<String, Object> bodyMap = new HashMap<>();
				bodyMap.put(edmProperty.get().getName(), value);
				return bodyMap;
			} catch (DeserializerException e) {
				throw new ErrorStatusException(CdsErrorStatuses.INVALID_PAYLOAD, e.getLocalizedMessage(), e);
			}
		} else {
			throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_URI_RESOURCE, resource.getKind());
		}
	}

	public Map<String, Object> extractDeltaCollectionFromValue(UriResourcePartTyped resource, EdmType responseType) {
		ODataDeserializer deserializer;
		try {
			deserializer = globals.getOData().createDeserializer(requestFormat, globals.getServiceMetadata(),
					Arrays.asList(ODataUtils.getODataVersion(odataRequest)));
		} catch (DeserializerException e) {
			throw new ErrorStatusException(CdsErrorStatuses.DESERIALIZER_FAILED, e);
		}
		switch (resource.getKind()) {
		case entitySet:
			try {
				Map<String, Object> bodyMap = new HashMap<>();
				bodyMap.put(DELTA, true);
				Delta delta = deserializer.delta(odataRequest.getBody(), (EdmEntityType) responseType).getDelta();
				bodyMap.put(DELTA_UPSERT, delta.getEntities().stream()
						.map(payloadProcessor::getEntity).filter(this::assertActive).collect(toList()));
				bodyMap.put(DELTA_DELETE, delta.getDeletedEntities().stream()
						.map(payloadProcessor::getEntity).filter(this::assertActive).collect(toList()));
				return bodyMap;
			} catch (DeserializerException e) {
				throw new ErrorStatusException(CdsErrorStatuses.INVALID_PAYLOAD, e.getLocalizedMessage(), e);
			}
		default:
			throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_URI_RESOURCE, resource.getKind());
		}
	}

	public boolean assertActive(Map<String, Object> entityData) {
		if (Boolean.FALSE.equals(entityData.get(Drafts.IS_ACTIVE_ENTITY))) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_ACTIVE);
		}
		return true;
	}

	public Map<String, Object> extractBodyFromJson(Optional<EdmProperty> edmProperty, UriResourcePartTyped resource,
			EdmType responseType) {

		ODataDeserializer deserializer;
		try {
			deserializer = globals.getOData().createDeserializer(requestFormat, globals.getServiceMetadata(),
					Arrays.asList(ODataUtils.getODataVersion(odataRequest)));
		} catch (DeserializerException e) {
			throw new ErrorStatusException(CdsErrorStatuses.DESERIALIZER_FAILED, e);
		}

		Map<String, Object> bodyMap = new HashMap<>();
		switch (resource.getKind()) {
		case singleton:
		case entitySet:
		case navigationProperty:
			try {
				DeserializerResult result = deserializer.entity(odataRequest.getBody(), (EdmEntityType) responseType);
				bodyMap.putAll(payloadProcessor.getEntity(result.getEntity()));
			} catch (DeserializerException e) {
				throw new ErrorStatusException(CdsErrorStatuses.INVALID_PAYLOAD, e.getLocalizedMessage(), e);
			}
			break;
		case action:
			try {
				DeserializerResult result = deserializer.actionParameters(odataRequest.getBody(),
						((UriResourceAction) resource).getAction());
				result.getActionParameters()
						.forEach((name, value) -> bodyMap.put(name, payloadProcessor.getValue(value)));
			} catch (DeserializerException e) {
				throw new ErrorStatusException(CdsErrorStatuses.INVALID_PAYLOAD, e.getLocalizedMessage(), e);
			}
			break;
		case primitiveProperty:
		case complexProperty:
			try {
				if (edmProperty.isPresent()) {
					EdmProperty edmProp = edmProperty.get();
					Property property = deserializer.property(odataRequest.getBody(), edmProp).getProperty();
					bodyMap.put(property.getName(), payloadProcessor.getValue(property));
				} else {
					throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_URI_RESOURCE, resource.getKind());
				}
			} catch (DeserializerException e) {
				throw new ErrorStatusException(CdsErrorStatuses.INVALID_PAYLOAD, e.getLocalizedMessage(), e);
			}
			break;
		default:
			throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_URI_RESOURCE, resource.getKind());
		}
		return bodyMap;
	}

	public Map<String, Object> extractBodyFromBinaryValue(Optional<EdmProperty> edmProperty,
			UriResourcePartTyped resource, Charset charset) {
		if (edmProperty.isPresent()) {
			Map<String, Object> bodyMap = new HashMap<>();
			Object value = charset != null ? new InputStreamReader(odataRequest.getBody(), charset) : odataRequest.getBody();
			bodyMap.put(edmProperty.get().getName(), value);
			return bodyMap;
		} else {
			throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_URI_RESOURCE, resource.getKind());
		}
	}

	private Object parseParameterValue(String parameterValue, EdmParameter edmParameter) {
		if (parameterValue != null) {
			FixedFormatDeserializer deserializer = globals.getOData().createFixedFormatDeserializer();
			try {
				return payloadProcessor.getValue(deserializer.parameter(parameterValue, edmParameter));
			} catch (DeserializerException e) {
				throw new ErrorStatusException(CdsErrorStatuses.INVALID_PAYLOAD, e.getLocalizedMessage(), e);
			}
		}
		return null;
	}

}
