/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.query.apply;

import static com.sap.cds.ql.cqn.transformation.CqnTransformation.IDENTITY;

import java.util.ArrayList;
import java.util.List;

import org.apache.olingo.server.api.uri.queryoption.ApplyItem;
import org.apache.olingo.server.api.uri.queryoption.ApplyItem.Kind;
import org.apache.olingo.server.api.uri.queryoption.ApplyOption;
import org.apache.olingo.server.api.uri.queryoption.apply.Aggregate;
import org.apache.olingo.server.api.uri.queryoption.apply.Ancestors;
import org.apache.olingo.server.api.uri.queryoption.apply.Compute;
import org.apache.olingo.server.api.uri.queryoption.apply.Concat;
import org.apache.olingo.server.api.uri.queryoption.apply.CustomFunction;
import org.apache.olingo.server.api.uri.queryoption.apply.Descendants;
import org.apache.olingo.server.api.uri.queryoption.apply.Filter;
import org.apache.olingo.server.api.uri.queryoption.apply.GroupBy;
import org.apache.olingo.server.api.uri.queryoption.apply.OrderBy;
import org.apache.olingo.server.api.uri.queryoption.apply.Search;
import org.apache.olingo.server.api.uri.queryoption.apply.Skip;
import org.apache.olingo.server.api.uri.queryoption.apply.Top;

import com.sap.cds.adapter.odata.v4.query.ExpressionParser;
import com.sap.cds.ql.cqn.transformation.CqnTransformation;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

public class ApplyConverter {

	private ExpressionParser expressionParser;

	public ApplyConverter(ExpressionParser expressionParser) {
		this.expressionParser = expressionParser;
	}

	public List<List<CqnTransformation>> unfoldAndConvert(List<ApplyItem> items) {
		int n = items.size();
		ApplyItem last;
		if (n > 0 && (last = items.get(n - 1)).getKind() == Kind.CONCAT) {
			List<ApplyItem> stem = items.subList(0, n - 1);
			return unfoldConcats(stem, (Concat) last);
		} else {
			return List.of(items.stream().map(this::transformation).toList());
		}
	}

	private List<List<CqnTransformation>> unfoldConcats(List<ApplyItem> stem, Concat concat) {
		List<ApplyOption> concats = concat.getApplyOptions();
		int m = concats.size();
		List<List<CqnTransformation>> all = new ArrayList<>(m);
		for (int i = 0; i < m; i++) {
			List<ApplyItem> unfolded = new ArrayList<>(stem);
			List<ApplyItem> concatItems = concats.get(i).getApplyItems();
			unfolded.addAll(concatItems);

			all.addAll(unfoldAndConvert(unfolded));
		}
		return all;
	}

	private CqnTransformation transformation(ApplyItem applyItem) {
		Kind kind = applyItem.getKind();
		switch (kind) {
			case IDENTITY:
				return IDENTITY;
			case AGGREGATE:
				return AggregateConverter.of((Aggregate) applyItem, expressionParser);
			case GROUP_BY:
				return GroupByConverter.of((GroupBy) applyItem, expressionParser);
			case ORDERBY:
				return OrderByConverter.of((OrderBy) applyItem, expressionParser);
			case TOP:
				return LimitConverter.top((Top) applyItem);
			case SKIP:
				return LimitConverter.skip((Skip) applyItem);
			case SEARCH:
				return SearchConverter.of((Search) applyItem);
			case FILTER:
				return FilterConverter.of((Filter) applyItem, expressionParser);
			case COMPUTE:
				return ComputeConverter.of((Compute) applyItem, expressionParser);
			case CUSTOM_FUNCTION:
				CustomFunction function = (CustomFunction) applyItem;
				if (TopLevelsConverter.handles(function)) {
					return TopLevelsConverter.of(function, expressionParser);
				}
				return new CustomConverter((CustomFunction) applyItem, expressionParser);
			case ANCESTORS:
				return HierarchySubsetConverter.ancestors((Ancestors) applyItem, expressionParser);
			case DESCENDANTS:
				return HierarchySubsetConverter.descendants((Descendants) applyItem, expressionParser);
			case CONCAT:
				throw new ErrorStatusException(CdsErrorStatuses.UNSUPPORTED_CONCAT_LAST);
			case EXPAND:
			default:
				throw new ErrorStatusException(CdsErrorStatuses.UNSUPPORTED_APPLY_OPTION, kind.name());
		}
	}

}
