/**************************************************************************
 * (C) 2019-2025 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.query.apply;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.olingo.server.api.uri.UriParameter;
import org.apache.olingo.server.api.uri.queryoption.apply.CustomFunction;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cds.adapter.odata.v4.metadata.extension.CustomVocabularyCatalog;
import com.sap.cds.adapter.odata.v4.query.ExpressionParser;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.transformation.CqnTopLevelsTransformation;
import com.sap.cds.ql.impl.transformations.TopLevelsTrafo;

public class TopLevelsConverter {
	private static final Logger logger = LoggerFactory.getLogger(TopLevelsConverter.class);
	private static final String NAME = CustomVocabularyCatalog.COM_SAP_VOCABULARY_HIERARCHY + "."
			+ CustomVocabularyCatalog.TOP_LEVELS;

	private TopLevelsConverter() {
		// empty
	}

	public static CqnTopLevelsTransformation of(CustomFunction custom, ExpressionParser expressionParser) {
		ParamConverter converter = new ParamConverter(custom, expressionParser);
		return TopLevelsTrafo.topLevels(converter.hierarchyQualifier(), converter.levels(), converter.expandLevels())
				.hierarchy(converter.hierarchyReference(), converter.nodeProperty());
	}

	private static class ParamConverter {
		private static final long DEFAULT_ALL = -1L;
		private static final String EXPAND_LEVELS = "ExpandLevels";
		private static final String NODE_ID = "NodeID";
		private static final String LEVELS = "Levels";

		private final ExpressionParser parser;
		private final Map<String, UriParameter> params;

		private static final ObjectMapper mapper = new ObjectMapper();

		public ParamConverter(CustomFunction transformation, ExpressionParser expressionParser) {
			params = transformation.getParameters().stream().collect(Collectors.toMap(UriParameter::getName, p -> p));
			this.parser = expressionParser;
		}

		CqnStructuredTypeRef hierarchyReference() {
			UriParameter ref = params.get("HierarchyNodes");
			return parser.getTargetTypeRef(ref.getExpression());
		}

		String hierarchyQualifier() {
			return unquote(params.get("HierarchyQualifier").getText());
		}

		CqnElementRef nodeProperty() {
			UriParameter node = params.get("NodeProperty");
			return CQL.get(node.getText());
		}

		Map<Object, Long> expandLevels() {
			if (!params.containsKey(EXPAND_LEVELS)) {
				return Map.of();
			}
			UriParameter node = params.get(EXPAND_LEVELS);
			String jsonArray = node.getText();
			List<Map<String, Object>> deserialized;
			try {
				deserialized = mapper.readValue(jsonArray, new TypeReference<List<Map<String, Object>>>() {
				});
			} catch (JsonProcessingException e) {
				logger.error("Unable to deserialize values of " + EXPAND_LEVELS, e);
				return Map.of();
			}
			return deserialized.stream()
					.collect(Collectors.toMap(
							m -> m.get(NODE_ID),
							m -> {
								Object lev = m.get(LEVELS);
								return lev != null ? ((Number) lev).longValue() : DEFAULT_ALL;
							}));
		}

		long levels() {
			return params.containsKey(LEVELS) ? Long.parseLong(params.get(LEVELS).getText()) : DEFAULT_ALL;
		}
	}

	private static String unquote(String text) {
		if (text.startsWith("'")) {
			text = text.substring(1, text.length() - 1);
		}
		return text;
	}

	public static boolean handles(CustomFunction function) {
		return NAME.equals(function.getFunction().getFullQualifiedName().getFullQualifiedNameAsString());
	}

}
