/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.serializer.json;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import org.apache.olingo.commons.api.edm.EdmAction;
import org.apache.olingo.commons.api.edm.EdmEntityType;
import org.apache.olingo.commons.api.edm.EdmFunction;
import org.apache.olingo.commons.api.edm.EdmStructuredType;
import org.apache.olingo.commons.api.edm.EdmType;
import org.apache.olingo.commons.api.edm.FullQualifiedName;
import org.apache.olingo.commons.api.edm.constants.EdmTypeKind;
import org.apache.olingo.commons.core.edm.primitivetype.EdmBinary;
import org.apache.olingo.commons.core.edm.primitivetype.EdmBoolean;
import org.apache.olingo.commons.core.edm.primitivetype.EdmByte;
import org.apache.olingo.commons.core.edm.primitivetype.EdmDecimal;
import org.apache.olingo.commons.core.edm.primitivetype.EdmDouble;
import org.apache.olingo.commons.core.edm.primitivetype.EdmInt16;
import org.apache.olingo.commons.core.edm.primitivetype.EdmInt32;
import org.apache.olingo.commons.core.edm.primitivetype.EdmInt64;
import org.apache.olingo.commons.core.edm.primitivetype.EdmSByte;
import org.apache.olingo.commons.core.edm.primitivetype.EdmSingle;
import org.apache.olingo.commons.core.edm.primitivetype.EdmStream;
import org.apache.olingo.commons.core.edm.primitivetype.EdmString;
import org.apache.olingo.server.api.ServiceMetadata;
import org.apache.olingo.server.api.uri.UriResource;
import org.apache.olingo.server.api.uri.UriResourceAction;
import org.apache.olingo.server.api.uri.UriResourceComplexProperty;
import org.apache.olingo.server.api.uri.UriResourceFunction;
import org.apache.olingo.server.api.uri.UriResourcePrimitiveProperty;
import org.apache.olingo.server.api.uri.queryoption.SelectOption;
import org.apache.olingo.server.core.serializer.utils.ExpandSelectHelper;

import com.sap.cds.adapter.odata.v4.processors.request.CdsODataRequest;
import com.sap.cds.adapter.odata.v4.serializer.json.api.Data2Json;
import com.sap.cds.adapter.odata.v4.serializer.json.api.PropertyInfo;
import com.sap.cds.adapter.odata.v4.serializer.json.options.CdsODataOptions;
import com.sap.cds.adapter.odata.v4.serializer.json.primitive.Boolean2Json;
import com.sap.cds.adapter.odata.v4.serializer.json.primitive.BooleanArray2Json;
import com.sap.cds.adapter.odata.v4.serializer.json.primitive.Decimal2Json;
import com.sap.cds.adapter.odata.v4.serializer.json.primitive.DecimalArray2Json;
import com.sap.cds.adapter.odata.v4.serializer.json.primitive.Number2Json;
import com.sap.cds.adapter.odata.v4.serializer.json.primitive.NumberArray2Json;
import com.sap.cds.adapter.odata.v4.serializer.json.primitive.Object2Json;
import com.sap.cds.adapter.odata.v4.serializer.json.primitive.ObjectArray2Json;
import com.sap.cds.adapter.odata.v4.serializer.json.primitive.Stream2Json;
import com.sap.cds.adapter.odata.v4.serializer.json.primitive.String2Json;
import com.sap.cds.adapter.odata.v4.serializer.json.primitive.StringArray2Json;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

public final class StructTypeHelper {

	public static Object NOOP = new Object();

	/** SELECT */
	public static Set<List<String>> getSelectedPaths(SelectOption selectOption) {
		if (null == selectOption) {
			return null;
		}
		return selectOption.getSelectItems() == null ? null
				: ExpandSelectHelper.getSelectedPaths(selectOption.getSelectItems());
	}

	public static boolean isPropertySelected(Set<List<String>> selectedPaths, String propertyName) {
		boolean isSelected = true;
		if (null != selectedPaths) {
			isSelected = ExpandSelectHelper.isSelected(selectedPaths, propertyName);
		}
		return isSelected;
	}

	public static Set<List<String>> getReducedSelectedPaths(Set<List<String>> selectedPaths, String propertyName) {
		if (null == selectedPaths) {
			return null;
		}
		return ExpandSelectHelper.getReducedSelectedPaths(selectedPaths, propertyName);
	}

	public static Set<List<String>> getSelectedPaths(SelectOption selectOption, String propertyName) {
		if (null == selectOption) {
			return null;
		}
		return selectOption.getSelectItems() == null ? null
				: ExpandSelectHelper.getSelectedPaths(selectOption.getSelectItems(), propertyName);
	}

	/** HELPER */
	public static PropertyInfo getPropertyInfo(CdsODataRequest cdsRequest) {
		EdmType responseType = cdsRequest.getResponseType();
		if (responseType.getKind() != EdmTypeKind.PRIMITIVE && 
			responseType.getKind() != EdmTypeKind.COMPLEX) {
			throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_EDM_TYPE, responseType.getKind());
		}
		UriResource resource = cdsRequest.getLastTypedResource();
		String name;
		boolean isCollection;
		switch (resource.getKind()) {
			case primitiveProperty:
				UriResourcePrimitiveProperty p = (UriResourcePrimitiveProperty) resource;
				name = p.getProperty().getName();
				isCollection = p.isCollection();
				break;
			case complexProperty:
				UriResourceComplexProperty cp = (UriResourceComplexProperty) resource;
				name = cp.getProperty().getName();
				isCollection = cp.isCollection();
				break;
			case action:
				EdmAction action = ((UriResourceAction) resource).getAction();
				name = action.getName();
				isCollection = action.getReturnType().isCollection();
				break;
			case function:
				EdmFunction function = ((UriResourceFunction) resource).getFunction();
				name = function.getName();
				isCollection = function.getReturnType().isCollection();
				break;
			default:
				throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_URI_RESOURCE, resource.getKind());
		}
		return PropertyInfo.create(name, responseType, isCollection);
	}

	public static String2Json<Map<String, Object>> createPropertyMetaTypeSerializer(PropertyInfo propertyInfo,
			CdsODataOptions options) {
		String typeName = propertyInfo.getName() + options.getConstants().getType();
		EdmType type = propertyInfo.getType();
		if (type.getKind() == EdmTypeKind.ENUM || type.getKind() == EdmTypeKind.DEFINITION) {
			if (propertyInfo.isCollection()) {
				return String2Json.constant(typeName,
						"#Collection(" + type.getFullQualifiedName().getFullQualifiedNameAsString() + ")");
			}
			return String2Json.constant(typeName, "#" + type.getFullQualifiedName().getFullQualifiedNameAsString());
		} else if (type.getKind() == EdmTypeKind.PRIMITIVE) {
			if (propertyInfo.isCollection()) {
				return String2Json.constant(typeName, "#Collection(" + type.getFullQualifiedName().getName() + ")");
			}
			// exclude the properties that can be heuristically determined
			if (type != EdmBoolean.getInstance() &&
					type != EdmDouble.getInstance() &&
					type != EdmString.getInstance()) {
				return String2Json.constant(typeName, "#" + type.getFullQualifiedName().getName());
			}
			return null;
		} else if (type.getKind() == EdmTypeKind.COMPLEX) {
			// non-collection case written in writeComplex method directly.
			if (propertyInfo.isCollection()) {
				return String2Json.constant(typeName,
						"#Collection(" + type.getFullQualifiedName().getFullQualifiedNameAsString() + ")");
			}
			return null;
		}
		throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_PROPERTY_TYPE, type.getKind(),
				propertyInfo.getName());
	}

	/**
	 *  Resolve the given derivedTypeName to the corresponding EdmStructuredType from the model, ensuring baseType is its valid base type.
	 */
	public static EdmStructuredType resolveEntityType(ServiceMetadata metadata, 
			EdmStructuredType baseType, String derivedTypeName) {
		if (derivedTypeName == null ||
				baseType.getFullQualifiedName().getFullQualifiedNameAsString().equals(derivedTypeName)) {
			return baseType;
		}
		EdmEntityType derivedType = metadata.getEdm().getEntityType(new FullQualifiedName(derivedTypeName));
		if (derivedType == null) {
			throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_EDM_TYPE, derivedTypeName);
		}
		EdmEntityType type = derivedType.getBaseType();
		while (type != null) {
			if (type.getFullQualifiedName().equals(baseType.getFullQualifiedName())) {
				return derivedType;
			}
			type = type.getBaseType();
		}
		// base type does not match derivedType
		throw new ErrorStatusException(CdsErrorStatuses.WRONG_BASE_EDM_TYPE, 
				baseType.getFullQualifiedName().getFullQualifiedNameAsString());
	}

	public static Data2Json<Map<String, Object>> createPropertySerializer(PropertyInfo propertyInfo,
			CdsODataOptions options) {
		EdmType edmType = propertyInfo.getType();
		switch (edmType.getKind()) {
			case PRIMITIVE:
			case DEFINITION:
				return createPrimitive(propertyInfo, options);
			default:
				throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_EDM_TYPE, edmType.getKind());
		}
	}

	public static Data2Json<Map<String, Object>> createPrimitive(PropertyInfo propertyInfo, CdsODataOptions options) {
		return createPrimitive(propertyInfo, propertyInfo.getName(), options);
	}

	public static Data2Json<Map<String, Object>> createPrimitive(PropertyInfo propertyInfo) {
		return createPrimitive(propertyInfo, propertyInfo.getName(), null);
	}

	public static Data2Json<Map<String, Object>> createPrimitive(PropertyInfo propertyInfo, String label,
			CdsODataOptions options) {
		boolean isIEEE754Compatible = options != null && options.isIEEE754Compatible();
		EdmType edmType = propertyInfo.getType();

		if (edmType == EdmBoolean.getInstance()) {
			return boolSerializer(label, propertyInfo);
		} else if (edmType == EdmByte.getInstance()
				|| edmType == EdmDouble.getInstance()
				|| edmType == EdmInt16.getInstance()
				|| edmType == EdmInt32.getInstance()
				|| edmType == EdmSByte.getInstance()
				|| edmType == EdmSingle.getInstance()
				|| edmType == EdmInt64.getInstance() && !isIEEE754Compatible) {
			return numberSerializer(label, propertyInfo);
		} else if ((edmType == EdmDecimal.getInstance() && !isIEEE754Compatible)) {
			return decimalSerializer(label, propertyInfo);
		} else if (edmType == EdmBinary.getInstance()) {
			return objectSerializer(label, propertyInfo);
		} else if (edmType == EdmStream.getInstance()) {
			return streamSerializer(label, propertyInfo);
		} else {
			return stringSerializer(label, propertyInfo);
		}
	}

	private static Data2Json<Map<String, Object>> stringSerializer(String label, PropertyInfo propertyType) {
		if (propertyType.isCollection()) {
			return StringArray2Json.val(label, fromProperty(propertyType));
		}
		return String2Json.val(label, fromProperty(propertyType));
	}

	private static Data2Json<Map<String, Object>> boolSerializer(String label, PropertyInfo propertyType) {
		if (propertyType.isCollection()) {
			return BooleanArray2Json.val(label, fromProperty(propertyType));
		}
		return Boolean2Json.val(label, fromProperty(propertyType));
	}

	private static Data2Json<Map<String, Object>> numberSerializer(String label, PropertyInfo propertyType) {
		if (propertyType.isCollection()) {
			return NumberArray2Json.val(label, fromProperty(propertyType));
		}
		return Number2Json.val(label, fromProperty(propertyType));
	}

	private static Data2Json<Map<String, Object>> decimalSerializer(String label, PropertyInfo propertyType) {
		if (propertyType.isCollection()) {
			return DecimalArray2Json.val(label, fromProperty(propertyType));
		}
		return Decimal2Json.val(label, fromProperty(propertyType));
	}

	private static Data2Json<Map<String, Object>> objectSerializer(String label, PropertyInfo propertyType) {
		if (propertyType.isCollection()) {
			return ObjectArray2Json.val(label, fromProperty(propertyType));
		}
		return Object2Json.val(label, fromProperty(propertyType));
	}

	private static Data2Json<Map<String, Object>> streamSerializer(String label, PropertyInfo propertyType) {
		return Stream2Json.val(label, fromProperty(propertyType));
	}

	private static <T> Function<Map<String, Object>, T> fromProperty(PropertyInfo propertyInfo) {
		String name = propertyInfo.getName();
		return fromProperty(name, propertyInfo.isOmitNulls());
	}

	@SuppressWarnings("unchecked")
	public static <T> Function<Map<String, Object>, T> fromProperty(String key, boolean omitNulls) {
		return map -> {
			Object val = map.get(key);
			if (omitNulls && null == val) {
				return (T) NOOP;
			}
			return (T) val;
		};
	}
}
