/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.utils;

import java.util.List;
import java.util.Map.Entry;
import java.util.Optional;

import org.apache.olingo.commons.api.edm.EdmEntityType;
import org.apache.olingo.commons.api.edm.EdmStructuredType;

import com.sap.cds.Row;
import com.sap.cds.adapter.odata.v4.CdsRequestGlobals;
import com.sap.cds.adapter.odata.v4.processors.request.CdsODataRequest;
import com.sap.cds.adapter.odata.v4.utils.mapper.EdmxFlavourMapper;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.CdsDataException;
import com.sap.cds.ql.Predicate;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ETagUtils;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.HttpHeaders;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.util.CdsTypeUtils;

public class ETagHelper {

	public static boolean isETagHeaderInRequest(CdsODataRequest odataRequest) {
		return odataRequest.getHeader(HttpHeaders.IF_MATCH) != null
				|| odataRequest.getHeader(HttpHeaders.IF_NONE_MATCH) != null;
	}

	public static boolean hasStar(CdsODataRequest request, String header) {
		List<String> headerValueList = ETagUtils.parseHeader(request.getHeader(header));
		if (headerValueList != null && headerValueList.contains("*")) {
			return true;
		}
		return false;
	}

	public static boolean hasETag(CdsEntity entity) {
		return getETagElement(entity).map(e -> e.getKey()).isPresent();
	}

	public static String getETagElementName(CdsStructuredType entity) {
		return getETagElement(entity).map(e -> e.getKey()).orElse(null);
	}

	public static CqnPredicate getETagPredicate(CdsODataRequest request, CdsEntity entity) {
		Entry<String, CdsElement> etagElement = getETagElement(entity)
				.orElseThrow(() -> new IllegalStateException("Entity must have an ETag"));

		CqnPredicate predicate = null;
		List<String> ifMatchValues = headerValues(request, HttpHeaders.IF_MATCH);
		if (ifMatchValues != null && !ifMatchValues.contains("*")) {
			predicate = etagPredicate(etagElement, ifMatchValues);
		}

		List<String> ifNoneMatchValues = headerValues(request, HttpHeaders.IF_NONE_MATCH);
		if (ifNoneMatchValues != null) {
			ifNoneMatchValues.remove("*"); // remove * from IF-None-Match header. It's handled separately
			if (!ifNoneMatchValues.isEmpty()) {
				CqnPredicate predicateIfNone = CQL.not(etagPredicate(etagElement, ifNoneMatchValues));
				predicate = CQL.and(predicate, predicateIfNone);
			}
		}

		return predicate;
	}

	private static Predicate etagPredicate(Entry<String, CdsElement> etagElement, List<String> values) {
		return CQL.eTag(CQL.get(etagElement.getKey()),
				CQL.list(values.stream().map(v -> convertETag(v, etagElement.getValue())).toList()));
	}

	private static List<String> headerValues(CdsODataRequest r, String h) {
		return ETagUtils.parseHeader(r.getHeader(h));
	}

	private static Optional<Entry<String, CdsElement>> getETagElement(CdsStructuredType entity) {
		return ElementUtils.recursiveElements(entity, e -> CdsAnnotations.ETAG.isTrue(e)).entrySet().stream()
				.findFirst();
	}

	private static CqnValue convertETag(String value, CdsElement element) {
		CdsBaseType type = element.getType().as(CdsSimpleType.class).getType();
		try {
			return CQL.val(CdsTypeUtils.parse(type, value));
		} catch (CdsDataException e) {
			throw new ErrorStatusException(CdsErrorStatuses.ETAG_VALUE_INVALID, type.cdsName(), e);
		}
	}

	public static String getETagProperty(CdsRequestGlobals globals, EdmStructuredType entityType) {
		String edmEntityName = entityType.getFullQualifiedName().getFullQualifiedNameAsString();
		String cdsEntityName = globals.getCdsEntityNames().getOrDefault(edmEntityName, edmEntityName);
		Optional<CdsEntity> cdsEntity = globals.getModel().findEntity(cdsEntityName);
		if (cdsEntity.isPresent()) {
			String eTagElement = ETagHelper.getETagElementName(cdsEntity.get());
			if (eTagElement != null) {
				return EdmxFlavourMapper.create(globals.getEdmxFlavour(), false).remap(eTagElement, cdsEntity.get());
			}
		}
		return null;
	}

	public static String getEtagValue(CdsRequestGlobals globals, EdmEntityType entityType, Row row) {
		String eTagProperty = getETagProperty(globals, entityType);
		if (null != eTagProperty) {
			Object value = row.get(eTagProperty);
			if (null != value) {
				return ETagUtils.createETagHeaderValue(value.toString());
			}
		}
		return null;
	}
}
